# 機能設計書 3-商品一括取得

## 概要

本ドキュメントは、eShopアプリケーションにおける商品一括取得機能の設計仕様を定義する。複数の商品IDを指定して、該当する商品データを一括で取得するREST APIエンドポイントを提供する。

### 本機能の処理概要

**業務上の目的・背景**：買い物かごに複数商品が入っている場合や、関連商品を一度に表示する場合など、複数の商品情報を効率的に取得する必要がある。個別のAPI呼び出しを繰り返すと通信オーバーヘッドが大きくなるため、一括取得機能を提供することでパフォーマンスを向上させる。

**機能の利用シーン**：買い物かご画面での複数商品表示、チェックアウト時の商品確認、関連商品表示、AIチャットボットでの複数商品推薦など、複数商品を同時に取得する必要がある場面で利用される。

**主要な処理内容**：
1. クエリパラメータから商品ID配列を取得
2. IN句を使用してデータベースから該当商品を一括取得
3. 取得した商品リストをレスポンスとして返却

**関連システム・外部連携**：PostgreSQLデータベースからEntity Framework Coreを通じてデータを取得する。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | カート画面 | 参照画面 | 買い物かご内の複数商品情報を一括取得 |
| 4 | チェックアウト画面 | 参照画面 | 注文確定時の商品確認 |

## 機能種別

CRUD操作（Read）/ バッチ取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ids | int[] | Yes | 取得する商品IDの配列 | クエリパラメータで複数指定可能 |

### 入力データソース

- HTTPクエリパラメータ: `/api/catalog/items/by?ids=1&ids=2&ids=3`

## 出力仕様

### 出力データ

**正常時（200 OK）: List&lt;CatalogItem&gt;**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| (配列) | List&lt;CatalogItem&gt; | 商品データの配列 |

**CatalogItem構造**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | 商品ID |
| Name | string | 商品名 |
| Description | string? | 商品説明 |
| Price | decimal | 価格 |
| PictureFileName | string? | 商品画像ファイル名 |
| CatalogTypeId | int | 商品タイプID |
| CatalogBrandId | int | ブランドID |
| AvailableStock | int | 在庫数 |
| RestockThreshold | int | 再発注閾値 |
| MaxStockThreshold | int | 最大在庫閾値 |
| OnReorder | bool | 再発注フラグ |

### 出力先

- HTTP Response Body（JSON配列形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ クエリパラメータからids配列を取得

2. データベースクエリ実行
   ├─ CatalogItemsテーブルに対してIN句でフィルタ
   └─ ids.Contains(item.Id)で該当商品を抽出

3. レスポンス生成
   └─ List<CatalogItem>をJSONで返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[ids配列取得]
    B --> C[DB検索実行]
    C --> D[WHERE Id IN ids]
    D --> E[ToListAsync]
    E --> F[List返却]
    F --> G[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 存在チェックなし | 存在しないIDが含まれていてもエラーにならず、存在する商品のみ返却 | 常時 |
| BR-002 | 順序不定 | 返却される商品の順序はids配列の順序と一致するとは限らない | 常時 |
| BR-003 | 重複除去なし | 同一IDが複数指定されても重複チェックは行わない | 常時 |

### 計算ロジック

本機能に計算ロジックはない。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品一括取得 | Catalog | SELECT | IN句で指定IDの商品を取得 |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE Id IN ({ids}) | Embeddingは除外（JsonIgnore） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | パラメータ形式が不正 | ProblemDetails形式でエラー詳細を返却 |
| 500 | InternalServerError | データベース接続エラー等 | サーバーエラーとして処理 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。

## パフォーマンス要件

- IN句による一括検索のため、個別APIを複数回呼び出すより効率的
- 大量のIDを指定する場合はIN句のパラメータ数制限に注意

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- SQLインジェクション対策: Entity Framework CoreのLINQクエリによりパラメータ化

## 備考

- 空の配列が指定された場合は空のリストを返却
- CatalogBrand、CatalogTypeはIncludeされていないため、レスポンスには含まれない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 商品エンティティの構造 |

**読解のコツ**: この機能ではCatalogBrandやCatalogTypeのIncludeがないため、ナビゲーションプロパティはnullになる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemsByIdsメソッドの実装 |

**主要処理フロー**:
1. **31-35行目**: `/items/by` エンドポイント定義
2. **162-168行目**: `GetItemsByIds` メソッド実装
3. **164行目**: Description属性で"List of ids for catalog items to return"を定義
4. **166行目**: Contains句でIN条件を構築
5. **167行目**: TypedResults.Okでリスト返却

#### Step 3: データアクセス層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | DbSet定義 |

**主要処理フロー**:
- **14行目**: CatalogItemsへのアクセス

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items/by?ids=1&ids=2)
    │
    └─ CatalogApi.GetItemsByIds()
           │
           └─ CatalogServices.Context (CatalogContext)
                  └─ DbSet<CatalogItem>.CatalogItems
                         ├─ Where(item => ids.Contains(item.Id))
                         └─ ToListAsync()
                                │
                                └─ List<CatalogItem> → TypedResults.Ok
```

### データフロー図

```
[入力]                    [処理]                        [出力]

HTTPリクエスト            CatalogApi.GetItemsByIds()   HTTPレスポンス
  │                            │                          │
  └─ ids[] ──────────────────▶│                          │
     [1, 2, 3]                 │                          │
                               │   PostgreSQL             │
                               │   WHERE Id IN (1,2,3)    │
                               │      │                   │
                               │      ▼                   │
                               └─▶ List<CatalogItem> ────▶ JSON配列
                                    ├─ CatalogItem[0]
                                    ├─ CatalogItem[1]
                                    └─ CatalogItem[2]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | ソース | サービスコンテナ |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | DbContext |
