# 機能設計書 30-猶予期間確認処理

## 概要

本ドキュメントは、eShopシステムにおける猶予期間確認処理機能の設計を記述するものである。OrderProcessorバックグラウンドサービスにおいて、猶予期間を超過した注文を定期的に検出し、在庫確認プロセスを開始するための統合イベントを発行する機能仕様を定義する。

### 本機能の処理概要

猶予期間確認処理機能は、GracePeriodManagerServiceとして実装されたバックグラウンドサービスであり、一定間隔でデータベースを検索し、注文日時から猶予期間が経過した「Submitted」ステータスの注文を検出する。検出された各注文に対してGracePeriodConfirmedIntegrationEventを発行し、Ordering.APIでの在庫確認処理を開始するトリガーとなる。

**業務上の目的・背景**：ECサイトでは、顧客が注文確定直後にキャンセルする可能性がある。即座に在庫確認・決済処理を開始すると、キャンセルされた注文に対して不要なリソースを消費することになる。猶予期間（Grace Period）を設けることで、顧客のキャンセル機会を保証しつつ、システムリソースを効率的に活用する。猶予期間が経過した注文のみを次のステップに進めることで、実際に処理が必要な注文を適切に管理する。

**機能の利用シーン**：
- システム起動時に自動的にバックグラウンドサービスが開始される
- 設定された間隔（CheckUpdateTime）で定期的に実行される
- 注文作成後、猶予期間が経過した際にイベントを発行

**主要な処理内容**：
1. BackgroundServiceとして定期的に実行
2. PostgreSQLデータベースに直接クエリを発行
3. 猶予期間を経過したSubmitted注文を検出
4. 各注文に対してGracePeriodConfirmedIntegrationEventを発行

**関連システム・外部連携**：
- PostgreSQLデータベースに直接接続（Npgsql）
- イベントバスを通じてGracePeriodConfirmedIntegrationEventを発行
- Ordering.APIが発行されたイベントを受信して処理

**権限による制御**：
- システム内部処理のため、外部からの直接呼び出しは不可
- バックグラウンドサービスとして自動実行

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | なし | - | バックグラウンドサービスのため画面なし |

## 機能種別

バックグラウンド処理 / 定期実行処理 / イベント駆動

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| GracePeriodTime | int | Yes | 猶予期間（分）| 設定ファイルから取得 |
| CheckUpdateTime | int | Yes | 確認間隔（秒）| 設定ファイルから取得 |

### 入力データソース

- BackgroundTaskOptions設定（環境変数/設定ファイル）
- PostgreSQLデータベース（ordering.ordersテーブル）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 検出注文数 | int | 猶予期間を経過した注文の数 |
| イベント | GracePeriodConfirmedIntegrationEvent | 各注文に対して発行されるイベント |

### 出力先

- イベントバス（RabbitMQ）
- ログ出力

## 処理フロー

### 処理シーケンス

```
1. サービス起動
   └─ BackgroundServiceとしてExecuteAsync()が開始される

2. 定期実行ループ
   └─ CheckUpdateTimeで指定された間隔で繰り返し

3. CheckConfirmedGracePeriodOrders()の実行
   └─ ログ出力（確認開始）

4. GetConfirmedGracePeriodOrders()の実行
   └─ NpgsqlDataSourceからコネクション取得
   └─ SQLクエリを実行
      └─ 条件: CURRENT_TIMESTAMP - "OrderDate" >= GracePeriodTime
      └─ 条件: "OrderStatus" = 'Submitted'
   └─ 該当する注文IDリストを取得

5. 各注文に対してイベント発行
   └─ GracePeriodConfirmedIntegrationEventを生成
   └─ ログ出力（イベント発行の記録）
   └─ IEventBus.PublishAsync()でイベントを発行

6. 次の実行まで待機
   └─ Task.Delay(CheckUpdateTime)
```

### フローチャート

```mermaid
flowchart TD
    A[サービス起動] --> B[ExecuteAsync開始]
    B --> C{キャンセル要求?}
    C -->|Yes| D[サービス終了]
    C -->|No| E[CheckConfirmedGracePeriodOrders]
    E --> F[GetConfirmedGracePeriodOrders]
    F --> G[SQLクエリ実行]
    G --> H{注文あり?}
    H -->|No| I[待機]
    H -->|Yes| J[各注文を処理]
    J --> K[GracePeriodConfirmedIntegrationEvent生成]
    K --> L[EventBus.PublishAsync]
    L --> M{全注文処理完了?}
    M -->|No| J
    M -->|Yes| I
    I --> N[Task.Delay CheckUpdateTime]
    N --> C
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-30-01 | 猶予期間判定 | 注文日時から設定された猶予期間が経過していること | SQLクエリ条件 |
| BR-30-02 | ステータス条件 | Submittedステータスの注文のみ対象 | SQLクエリ条件 |
| BR-30-03 | 定期実行 | CheckUpdateTimeで指定された間隔で繰り返し実行 | 常に適用 |
| BR-30-04 | エラー継続 | データベースエラー時も空リストを返却してサービス継続 | 例外発生時 |

### 計算ロジック

**猶予期間経過判定（SQLクエリ）**:
```sql
SELECT "Id"
FROM ordering.orders
WHERE CURRENT_TIMESTAMP - "OrderDate" >= @GracePeriodTime
  AND "OrderStatus" = 'Submitted'
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 猶予期間経過注文検索 | ordering.orders | SELECT | 条件に合致する注文IDを取得 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE CURRENT_TIMESTAMP - "OrderDate" >= @GracePeriodTime AND "OrderStatus" = 'Submitted' | 猶予期間経過注文のIDのみ取得 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | NpgsqlException | データベース接続エラー | ログ出力、空リスト返却、サービス継続 |

### リトライ仕様

- データベースエラー時は次の実行サイクルで再試行
- イベント発行はイベントバスのリトライポリシーに従う

## トランザクション仕様

- 読み取り専用操作のためトランザクション管理は不要
- 各イベント発行は独立して行われる

## パフォーマンス要件

- 実行間隔: CheckUpdateTimeで設定（秒単位）
- データベースクエリ: インデックスを活用した効率的な検索
- メモリ使用: 検出された注文IDのリストのみ保持

## セキュリティ考慮事項

- データベース接続はNpgsqlDataSourceで管理
- 内部サービスのため、外部からの直接アクセスは不可
- 注文データの読み取りのみ（書き込みなし）

## 備考

- ASP.NET Core BackgroundServiceとして実装
- Npgsqlを使用した直接データベースアクセス（Entity Frameworkを使用しない）
- 設定変更はサービス再起動で反映（IOptionsは起動時に読み込み）
- PostgreSQL固有のSQL構文を使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、設定クラスと統合イベントの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BackgroundTaskOptions.cs | `src/OrderProcessor/BackgroundTaskOptions.cs` | 設定クラスの構造（GracePeriodTime, CheckUpdateTime） |
| 1-2 | GracePeriodConfirmedIntegrationEvent.cs | `src/OrderProcessor/Events/GracePeriodConfirmedIntegrationEvent.cs` | 出力イベントの構造（OrderIdのみ） |

**読解のコツ**: GracePeriodTimeは分単位、CheckUpdateTimeは秒単位で設定される。SQLクエリではTimeSpanに変換して使用する。

#### Step 2: バックグラウンドサービスを理解する

定期実行の仕組みを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GracePeriodManagerService.cs | `src/OrderProcessor/Services/GracePeriodManagerService.cs` | メインのサービスクラス |

**主要処理フロー**:
1. **8-12行目**: コンストラクタ（依存性注入）
2. **14行目**: BackgroundTaskOptionsをフィールドに保持
3. **16-41行目**: ExecuteAsync()メソッド（BackgroundService継承）
4. **18行目**: CheckUpdateTimeからTimeSpanを作成
5. **26-36行目**: whileループ（キャンセル要求まで継続）
6. **33行目**: CheckConfirmedGracePeriodOrders()を呼び出し
7. **35行目**: Task.Delayで待機

#### Step 3: 猶予期間確認処理を理解する

猶予期間経過注文の検出とイベント発行を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GracePeriodManagerService.cs | `src/OrderProcessor/Services/GracePeriodManagerService.cs` | CheckConfirmedGracePeriodOrders()メソッド |

**主要処理フロー**:
- **44-61行目**: CheckConfirmedGracePeriodOrders()メソッド
- **51行目**: GetConfirmedGracePeriodOrders()を呼び出し
- **53-59行目**: 各注文IDに対してイベント発行ループ
- **55行目**: GracePeriodConfirmedIntegrationEventを生成
- **59行目**: EventBus.PublishAsync()でイベント発行

#### Step 4: データベースクエリを理解する

SQLクエリの詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | GracePeriodManagerService.cs | `src/OrderProcessor/Services/GracePeriodManagerService.cs` | GetConfirmedGracePeriodOrders()メソッド |

**主要処理フロー**:
- **63-93行目**: GetConfirmedGracePeriodOrders()メソッド
- **67行目**: NpgsqlDataSourceからコネクション取得
- **68行目**: コマンドオブジェクト作成
- **69-73行目**: SQLクエリ定義
- **74行目**: パラメータバインド（GracePeriodTimeをTimeSpanに変換）
- **78-84行目**: クエリ実行と結果読み取り
- **87-89行目**: 例外処理（ログ出力してサービス継続）

### プログラム呼び出し階層図

```
GracePeriodManagerService (BackgroundService)
    │
    └─ ExecuteAsync() [サービス開始時に呼び出し]
           │
           └─ while (!stoppingToken.IsCancellationRequested)
                  │
                  ├─ CheckConfirmedGracePeriodOrders()
                  │      │
                  │      ├─ GetConfirmedGracePeriodOrders()
                  │      │      │
                  │      │      ├─ NpgsqlDataSource.CreateConnection()
                  │      │      │
                  │      │      ├─ SQL: SELECT "Id" FROM ordering.orders
                  │      │      │       WHERE CURRENT_TIMESTAMP - "OrderDate" >= @GracePeriodTime
                  │      │      │         AND "OrderStatus" = 'Submitted'
                  │      │      │
                  │      │      └─ List<int> orderIds
                  │      │
                  │      └─ foreach (orderId in orderIds)
                  │             │
                  │             ├─ GracePeriodConfirmedIntegrationEvent 生成
                  │             │
                  │             ├─ Logger.LogInformation()
                  │             │
                  │             └─ IEventBus.PublishAsync()
                  │
                  └─ Task.Delay(CheckUpdateTime)
```

### データフロー図

```
[設定]                              [処理]                                [出力]

BackgroundTaskOptions
    │
    ├─ GracePeriodTime (分)
    │
    └─ CheckUpdateTime (秒)
                │
                ▼
        GracePeriodManagerService
                │
                ├─ ExecuteAsync() [定期実行ループ]
                │
                └─ CheckConfirmedGracePeriodOrders()
                        │
                        ▼
                GetConfirmedGracePeriodOrders()
                        │
                        ▼
                ┌───────────────────────────┐
                │     PostgreSQL Database   │
                │                           │
                │  ordering.orders          │
                │    - Id                   │
                │    - OrderDate            │
                │    - OrderStatus          │
                │                           │
                │  WHERE:                   │
                │    NOW() - OrderDate      │
                │      >= GracePeriodTime   │
                │    AND Status = Submitted │
                └───────────────────────────┘
                        │
                        ▼
                List<int> orderIds
                        │
                        ▼
                foreach orderId:
                        │
                        ├─ GracePeriodConfirmedIntegrationEvent
                        │
                        └─▶ EventBus (RabbitMQ)
                               │
                               ▼
                          Ordering.API
                               │
                               └─ SetAwaitingValidationOrderStatusCommand
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GracePeriodManagerService.cs | `src/OrderProcessor/Services/GracePeriodManagerService.cs` | ソース | メインサービスクラス |
| BackgroundTaskOptions.cs | `src/OrderProcessor/BackgroundTaskOptions.cs` | ソース | 設定クラス |
| GracePeriodConfirmedIntegrationEvent.cs | `src/OrderProcessor/Events/GracePeriodConfirmedIntegrationEvent.cs` | ソース | 出力統合イベント |
| Program.cs | `src/OrderProcessor/Program.cs` | ソース | サービスエントリーポイント |
| Extensions.cs | `src/OrderProcessor/Extensions/Extensions.cs` | ソース | 拡張メソッド |
