# 機能設計書 31-ユーザーログイン

## 概要

本ドキュメントは、eShopアプリケーションにおけるユーザーログイン機能の設計仕様を定義するものである。本機能は、Identity.APIサービスが提供する認証基盤の中核機能であり、メールアドレスとパスワードによるユーザー認証を実現する。

### 本機能の処理概要

本機能は、ユーザーがeShopシステムにアクセスする際の認証処理を担当し、セキュアなログインフローを提供する。IdentityServer4をベースとした認証基盤を使用し、OAuth2.0/OpenID Connect標準に準拠した認証処理を実行する。

**業務上の目的・背景**：ECサイトにおいて、ユーザー認証は注文処理、個人情報保護、購入履歴管理などの基盤となる重要な機能である。不正アクセスを防止し、ユーザー固有のデータ（注文履歴、カート内容、配送先住所）へのアクセス制御を実現するために、信頼性の高い認証機能が必要となる。本機能は、ASP.NET Core Identityを活用し、パスワードのハッシュ化、アカウントロックアウト機能などのセキュリティ機能を提供する。

**機能の利用シーン**：ユーザーがeShopウェブアプリケーションやモバイルアプリからログインする際に使用される。商品購入、注文履歴確認、カート操作など、認証が必要な機能へアクセスする前の認証ゲートウェイとして機能する。

**主要な処理内容**：
1. ログインフォーム画面の表示（GET /Account/Login）
2. ユーザー名（メールアドレス）とパスワードの受信
3. SignInManagerによるパスワード認証の実行
4. 認証成功時のユーザーセッション確立とリダイレクト
5. 認証失敗時のエラーハンドリングとロックアウト制御

**関連システム・外部連携**：IdentityServer4を介してWebApp、WebhookClient、HybridAppなどのクライアントアプリケーションと連携する。OAuth2.0認可フローにより、クライアントアプリケーションにアクセストークンを発行する。

**権限による制御**：本機能自体は匿名ユーザーがアクセス可能（AllowAnonymous属性）である。認証成功後、ユーザーには付与されたスコープに応じたアクセス権限が設定される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 6 | ログイン画面 | 主画面 | Authorize属性によりIdentity.APIへリダイレクト |
| 9 | ログイン画面 | 主画面 | メールアドレス・パスワードによるユーザー認証 |
| 12 | アクセス拒否画面 | 結果表示画面 | アクセス権限不足時のエラー表示 |
| 13 | ホーム画面 | 参照画面 | IdentityServer4管理リンクへのナビゲーション |
| 14 | 同意画面 | 参照画面 | OAuth同意画面でクライアントへの権限付与確認 |
| 17 | デバイスコード入力画面 | 参照画面 | デバイスフロー認証用ユーザーコード入力 |
| 18 | デバイス確認画面 | 参照画面 | デバイスフロー認証の権限確認 |
| 20 | エラー画面 | 結果表示画面 | 認証エラー時のエラーメッセージ・リクエストID表示 |
| 23 | ログイン画面 | 参照画面 | WebhookClientからのログインリダイレクト |

## 機能種別

CRUD操作（認証処理） / セキュリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Username | string | Yes | ユーザーのメールアドレス | Required属性による必須チェック |
| Password | string | Yes | ユーザーのパスワード | Required属性による必須チェック |
| RememberLogin | bool | No | ログイン状態の永続化フラグ | - |
| ReturnUrl | string | No | 認証成功後のリダイレクト先URL | IsLocalUrlによるセキュリティチェック |

### 入力データソース

- 画面入力: ログインフォームからのユーザー名・パスワード
- URLパラメータ: ReturnUrl（認可リクエスト時のコールバックURL）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 認証Cookie | Cookie | .AspNetCore.Cookies認証クッキー |
| リダイレクトレスポンス | HTTP 302 | 認証成功時のリダイレクト |
| LoginViewModel | ViewModel | 認証失敗時のフォーム再表示データ |
| UserLoginSuccessEvent | イベント | 認証成功時のログイベント |
| UserLoginFailureEvent | イベント | 認証失敗時のログイベント |

### 出力先

- 画面表示: ログインフォームまたはリダイレクト先画面
- 認証Cookie: クライアントブラウザ
- イベントログ: IdentityServer4イベントシステム

## 処理フロー

### 処理シーケンス

```
1. ログインページ表示（GET /Account/Login）
   └─ BuildLoginViewModelAsync()でビューモデルを構築
   └─ 外部プロバイダー設定の読み込み
   └─ クライアント設定に基づくローカルログイン有効化判定

2. ログインフォーム送信（POST /Account/Login）
   └─ ModelState.IsValidでバリデーション確認
   └─ キャンセルボタン処理の分岐

3. パスワード認証の実行
   └─ SignInManager.PasswordSignInAsync()呼び出し
   └─ lockoutOnFailure: true でアカウントロックアウト有効

4. 認証結果の処理
   └─ 成功時: UserLoginSuccessEventを発行
   └─ 成功時: ReturnUrlまたはホームへリダイレクト
   └─ 失敗時: UserLoginFailureEventを発行
   └─ 失敗時: エラーメッセージを表示してフォーム再表示
```

### フローチャート

```mermaid
flowchart TD
    A[ログイン画面表示リクエスト] --> B{外部ログインのみ?}
    B -->|Yes| C[外部プロバイダーへリダイレクト]
    B -->|No| D[ログインフォーム表示]
    D --> E[ユーザー名/パスワード入力]
    E --> F{キャンセル?}
    F -->|Yes| G{認可コンテキストあり?}
    G -->|Yes| H[アクセス拒否レスポンス]
    G -->|No| I[ホームへリダイレクト]
    F -->|No| J{バリデーション成功?}
    J -->|No| K[エラー表示してフォーム再表示]
    J -->|Yes| L[PasswordSignInAsync実行]
    L --> M{認証成功?}
    M -->|Yes| N[UserLoginSuccessEvent発行]
    N --> O{ReturnUrl有効?}
    O -->|Yes| P[ReturnUrlへリダイレクト]
    O -->|No| Q[ホームへリダイレクト]
    M -->|No| R[UserLoginFailureEvent発行]
    R --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-31-01 | アカウントロックアウト | 連続した認証失敗時にアカウントを一時的にロック | lockoutOnFailure: true設定時 |
| BR-31-02 | ローカルログイン有効化 | クライアント設定でローカルログインの許可/禁止を制御 | EnableLocalLogin設定 |
| BR-31-03 | ReturnURL検証 | セキュリティのため、ローカルURLのみ許可 | リダイレクト処理時 |
| BR-31-04 | Remember Login | ユーザー選択に基づき永続的認証クッキーを発行 | RememberLogin=true時 |
| BR-31-05 | 認可コンテキスト検証 | GetAuthorizationContextAsyncで認可リクエストの妥当性を確認 | 全ログインリクエスト |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ユーザー検索 | AspNetUsers | SELECT | メールアドレスでユーザーを検索 |
| パスワード検証 | AspNetUsers | SELECT | パスワードハッシュの照合 |
| ログイン失敗回数更新 | AspNetUsers | UPDATE | AccessFailedCount, LockoutEnd更新 |

### テーブル別操作詳細

#### AspNetUsers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | UserName, PasswordHash | WHERE NormalizedEmail = @email | ユーザー検索 |
| UPDATE | AccessFailedCount | +1 | 認証失敗時 |
| UPDATE | LockoutEnd | 現在時刻 + ロックアウト期間 | ロックアウト発動時 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 認証失敗 | パスワード不一致 | "Invalid username or password"メッセージ表示 |
| - | アカウントロックアウト | 連続認証失敗超過 | ロックアウトメッセージ表示 |
| - | 無効なReturnURL | 外部URLへのリダイレクト試行 | 例外をスロー |
| - | モデルバリデーションエラー | 必須項目未入力 | フォーム再表示 |

### リトライ仕様

- ユーザーによる再入力が可能
- アカウントロックアウト時は一定時間経過後に再試行可能

## トランザクション仕様

- ASP.NET Core Identityが内部でトランザクション管理を実行
- ログイン失敗カウント更新は自動コミット

## パフォーマンス要件

- ログイン処理: 2秒以内の応答
- パスワードハッシュ検証: ASP.NET Core Identity標準実装

## セキュリティ考慮事項

- ValidateAntiForgeryToken属性によるCSRF対策
- SecurityHeaders属性によるセキュリティヘッダー付与
- パスワードはPBKDF2でハッシュ化して保存（ASP.NET Core Identity標準）
- IdentityServer4によるOAuth2.0/OIDC標準準拠
- HTTPSによる通信暗号化（本番環境）

## 備考

- IdentityServer4のQuickstartテンプレートをベースに実装
- 外部認証プロバイダー（Google、Facebook等）への拡張が可能な設計

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、認証に使用されるデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | ユーザーエンティティの拡張フィールド（カード情報、住所など） |
| 1-2 | LoginInputModel.cs | `src/Identity.API/Quickstart/Account/LoginInputModel.cs` | ログインフォームからの入力データ構造 |
| 1-3 | LoginViewModel.cs | `src/Identity.API/Quickstart/Account/LoginViewModel.cs` | ログイン画面表示用ViewModel |

**読解のコツ**: ApplicationUserはASP.NET Core IdentityのIdentityUserを継承しており、基本的な認証フィールド（UserName, PasswordHash等）に加えて、eShop固有のフィールドが追加されている。

#### Step 2: エントリーポイントを理解する

処理の起点となるAccountControllerを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | ログイン処理の全体フロー |

**主要処理フロー**:
1. **39-54行目**: GET Login - ログインフォーム表示処理
2. **59-140行目**: POST Login - ログイン認証処理
3. **95行目**: PasswordSignInAsync呼び出し（認証の核心部分）
4. **96-128行目**: 認証成功時のリダイレクト処理分岐
5. **130-131行目**: 認証失敗時のイベント発行とエラーメッセージ設定

#### Step 3: サービス層を理解する

認証サービスの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | ログインサービスのインターフェース定義 |
| 3-2 | EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | Entity Frameworkベースのログインサービス実装 |

**主要処理フロー**:
- **14-17行目**: FindByUsername - メールアドレスでユーザーを検索
- **19-22行目**: ValidateCredentials - パスワード検証
- **24-27行目**: SignIn - 認証クッキーの発行

#### Step 4: データアクセス層を理解する

データベースコンテキストを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ApplicationDbContext.cs | `src/Identity.API/Data/ApplicationDbContext.cs` | IdentityDbContextの継承とカスタマイズ |

### プログラム呼び出し階層図

```
AccountController.Login (POST)
    │
    ├─ _interaction.GetAuthorizationContextAsync()
    │      └─ IdentityServer4認可コンテキスト取得
    │
    ├─ _signInManager.PasswordSignInAsync()
    │      └─ UserManager.CheckPasswordAsync()
    │             └─ PasswordHasher.VerifyHashedPassword()
    │
    ├─ _userManager.FindByNameAsync()
    │      └─ ApplicationDbContext.Users
    │
    └─ _events.RaiseAsync()
           └─ UserLoginSuccessEvent / UserLoginFailureEvent
```

### データフロー図

```
[入力]                    [処理]                         [出力]

LoginInputModel    ───▶  AccountController.Login   ───▶  HTTP 302 Redirect
 ├─ Username              │                              または
 ├─ Password              ├─ SignInManager               LoginViewModel
 ├─ RememberLogin         ├─ UserManager
 └─ ReturnUrl             └─ EventService         ───▶  認証Cookie
                                                        (.AspNetCore.Cookies)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | ソース | ログイン処理のメインコントローラー |
| EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | ソース | Entity Frameworkベースの認証サービス |
| ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | ソース | ログインサービスのインターフェース |
| ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | ソース | ユーザーエンティティモデル |
| LoginInputModel.cs | `src/Identity.API/Quickstart/Account/LoginInputModel.cs` | ソース | 入力モデル |
| LoginViewModel.cs | `src/Identity.API/Quickstart/Account/LoginViewModel.cs` | ソース | ビューモデル |
| AccountOptions.cs | `src/Identity.API/Quickstart/Account/AccountOptions.cs` | ソース | アカウント設定オプション |
| ApplicationDbContext.cs | `src/Identity.API/Data/ApplicationDbContext.cs` | ソース | データベースコンテキスト |
| Config.cs | `src/Identity.API/Configuration/Config.cs` | 設定 | IdentityServer4設定 |
| Login.cshtml | `src/Identity.API/Views/Account/Login.cshtml` | テンプレート | ログインフォームビュー |
