# 機能設計書 32-ユーザー検索

## 概要

本ドキュメントは、eShopアプリケーションにおけるユーザー検索機能の設計仕様を定義するものである。本機能は、Identity.APIサービスが提供する認証基盤の一部であり、メールアドレスをキーとしてユーザー情報を検索・取得する機能を提供する。

### 本機能の処理概要

本機能は、認証プロセスの前段階でユーザーの存在確認を行うための検索機能を提供する。ASP.NET Core IdentityのUserManagerを活用し、データベースからユーザー情報を効率的に取得する。

**業務上の目的・背景**：認証処理において、パスワード検証の前にユーザーの存在確認が必要である。また、ユーザー管理機能やパスワードリセットなどの機能でも、メールアドレスによるユーザー特定が必要となる。本機能は、これらの処理の基盤となるユーザー検索機能を提供し、認証システム全体の整合性を保つ役割を果たす。

**機能の利用シーン**：ログイン処理時のユーザー存在確認、パスワードリセット時のユーザー特定、管理者によるユーザー検索など、ユーザー情報が必要な様々な場面で使用される。主にEFLoginServiceから呼び出され、認証フローの一部として機能する。

**主要な処理内容**：
1. メールアドレス（ユーザー名）の受け取り
2. UserManagerを使用したデータベース検索
3. 正規化されたメールアドレスによる高速検索
4. ApplicationUserエンティティの返却または null 返却

**関連システム・外部連携**：ASP.NET Core IdentityのUserManagerと連携し、ApplicationDbContextを介してSQL Serverデータベースにアクセスする。EFLoginServiceから呼び出され、AccountControllerの認証フローの一部として機能する。

**権限による制御**：本機能は内部サービスとして使用され、直接的なエンドポイントは公開されていない。呼び出し元のコントローラーで認証・認可制御が行われる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | ログイン画面 | 参照画面 | ログイン時のユーザー存在確認 |

## 機能種別

CRUD操作（Read） / データ検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| user | string | Yes | 検索対象のメールアドレス（ユーザー名） | null/空文字の場合はnull返却 |

### 入力データソース

- 内部呼び出し: EFLoginServiceからのメソッド呼び出し
- 画面入力経由: ログインフォームから入力されたユーザー名

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ApplicationUser | ApplicationUser | 検索されたユーザー情報（存在する場合） |
| null | null | ユーザーが見つからない場合 |

### ApplicationUserの主要フィールド

| フィールド名 | 型 | 説明 |
|-------------|-----|------|
| Id | string | ユーザー一意識別子 |
| UserName | string | ユーザー名（メールアドレス） |
| Email | string | メールアドレス |
| PasswordHash | string | パスワードハッシュ |
| CardNumber | string | カード番号 |
| SecurityNumber | string | セキュリティコード |
| Expiration | string | カード有効期限 |
| CardHolderName | string | カード名義人 |
| CardType | int | カードタイプ |
| Street | string | 住所（通り） |
| City | string | 市区町村 |
| State | string | 都道府県/州 |
| Country | string | 国 |
| ZipCode | string | 郵便番号 |
| Name | string | 名前 |
| LastName | string | 姓 |

### 出力先

- 呼び出し元メソッド: EFLoginService.FindByUsername()の戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. メールアドレス（ユーザー名）の受け取り
   └─ FindByUsername(string user)の引数として受信

2. UserManagerによる検索実行
   └─ _userManager.FindByEmailAsync(user)を呼び出し
   └─ 正規化されたメールアドレスで検索
   └─ AspNetUsersテーブルからデータ取得

3. 検索結果の返却
   └─ ユーザーが存在する場合: ApplicationUserオブジェクトを返却
   └─ ユーザーが存在しない場合: nullを返却
```

### フローチャート

```mermaid
flowchart TD
    A[FindByUsername呼び出し] --> B[メールアドレス受け取り]
    B --> C[UserManager.FindByEmailAsync実行]
    C --> D{ユーザーが存在?}
    D -->|Yes| E[ApplicationUserを返却]
    D -->|No| F[nullを返却]
    E --> G[終了]
    F --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-32-01 | メールアドレスによる検索 | ユーザー名としてメールアドレスを使用して検索 | 全検索リクエスト |
| BR-32-02 | 正規化検索 | メールアドレスは正規化（大文字化）されて検索 | 全検索リクエスト |
| BR-32-03 | 存在しない場合の処理 | ユーザーが存在しない場合はnullを返却 | 検索結果なしの場合 |
| BR-32-04 | 単一結果保証 | 同一メールアドレスのユーザーは1件のみ | メールアドレスはユニーク制約 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ユーザー検索 | AspNetUsers | SELECT | 正規化メールアドレスでユーザーを検索 |

### テーブル別操作詳細

#### AspNetUsers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE NormalizedEmail = UPPER(@email) | UserManagerが正規化を実行 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 該当なし | ユーザーが見つからない | nullを返却（例外は発生しない） |
| - | データベース接続エラー | DBアクセス障害時 | 呼び出し元で例外処理 |

### リトライ仕様

- 本機能自体にはリトライ機能なし
- Entity Frameworkのリトライポリシーに従う

## トランザクション仕様

- 読み取り専用操作のため、トランザクション管理は不要
- 読み取り一貫性はEntity Frameworkの既定動作に依存

## パフォーマンス要件

- 検索処理: 100ms以内の応答
- NormalizedEmailにインデックスが存在するため高速検索が可能

## セキュリティ考慮事項

- ユーザー情報（パスワードハッシュを含む）を返却するため、内部サービスとしてのみ使用
- 直接的なAPIエンドポイントは公開されていない
- 呼び出し元での適切な認証・認可チェックが必要

## 備考

- ASP.NET Core IdentityのUserManager<T>を活用
- FindByEmailAsyncは内部でFindByNameAsyncと同様の動作（メールアドレス=ユーザー名として設定されている場合）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、検索対象となるユーザーエンティティの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | ユーザーエンティティの拡張フィールド定義 |

**読解のコツ**: ApplicationUserはIdentityUserを継承しており、基本フィールド（Id, UserName, Email, PasswordHash等）は継承元で定義されている。eShop固有のフィールド（カード情報、住所）が追加されている点に注目。

#### Step 2: インターフェースを理解する

サービスインターフェースで定義されたメソッドシグネチャを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | FindByUsernameメソッドの定義 |

**主要処理フロー**:
- **7行目**: `Task<T> FindByUsername(string user)` - ジェネリック型でユーザーエンティティを返却

#### Step 3: 実装を理解する

EFLoginServiceの具体的な実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | FindByUsernameの実装 |

**主要処理フロー**:
- **5行目**: UserManager<ApplicationUser>のインジェクション
- **14-17行目**: FindByUsernameの実装（FindByEmailAsyncを呼び出し）

#### Step 4: データアクセス層を理解する

UserManagerがアクセスするデータベースコンテキストを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ApplicationDbContext.cs | `src/Identity.API/Data/ApplicationDbContext.cs` | IdentityDbContextの継承と設定 |

### プログラム呼び出し階層図

```
EFLoginService.FindByUsername(string user)
    │
    └─ _userManager.FindByEmailAsync(user)
           │
           └─ UserManager<ApplicationUser>
                  │
                  └─ ApplicationDbContext
                         │
                         └─ AspNetUsers テーブル (SQL Server)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

string user        ───▶  EFLoginService              ───▶  ApplicationUser
(メールアドレス)         .FindByUsername()                  または null
                              │
                              ▼
                         UserManager
                         .FindByEmailAsync()
                              │
                              ▼
                         ApplicationDbContext
                              │
                              ▼
                         AspNetUsers
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | ソース | ユーザー検索機能の実装 |
| ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | ソース | サービスインターフェース定義 |
| ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | ソース | ユーザーエンティティモデル |
| ApplicationDbContext.cs | `src/Identity.API/Data/ApplicationDbContext.cs` | ソース | データベースコンテキスト |
| AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | ソース | 呼び出し元コントローラー |
