# 機能設計書 33-認証情報検証

## 概要

本ドキュメントは、eShopアプリケーションにおける認証情報検証機能の設計仕様を定義するものである。本機能は、Identity.APIサービスが提供する認証基盤の一部であり、ユーザーが入力したパスワードの正当性を検証する。

### 本機能の処理概要

本機能は、ログインプロセスにおいてユーザーのパスワードを検証し、認証の成否を判定する機能を提供する。ASP.NET Core IdentityのUserManagerを活用し、ハッシュ化されたパスワードとの照合を安全に実行する。

**業務上の目的・背景**：ECサイトにおいて、ユーザー認証はセキュリティの要である。パスワード検証は、不正アクセスを防止し、正規ユーザーのみがシステムにアクセスできることを保証するための重要な機能である。本機能は、パスワードのハッシュ照合という暗号学的に安全な検証を実行し、平文パスワードがデータベースに保存されないことを保証する。

**機能の利用シーン**：ユーザーがログインフォームでパスワードを入力した後、EFLoginServiceから呼び出されてパスワードの正当性を検証する。また、パスワード変更時の現行パスワード確認にも使用される可能性がある。

**主要な処理内容**：
1. ユーザーオブジェクトとパスワード文字列の受け取り
2. UserManager.CheckPasswordAsyncによるパスワードハッシュ照合
3. PBKDF2（Password-Based Key Derivation Function 2）によるセキュアな比較
4. 検証結果（true/false）の返却

**関連システム・外部連携**：ASP.NET Core IdentityのUserManagerと連携し、PasswordHasherを使用してセキュアなパスワード検証を実行する。EFLoginServiceから呼び出され、AccountControllerの認証フローの一部として機能する。

**権限による制御**：本機能は内部サービスとして使用され、直接的なエンドポイントは公開されていない。パスワード情報を扱うため、呼び出し元での適切なセキュリティ管理が必須である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | ログイン画面 | 参照画面 | EFLoginServiceでパスワード検証 |

## 機能種別

セキュリティ機能 / 認証処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| user | ApplicationUser | Yes | 検証対象のユーザーオブジェクト | null不可 |
| password | string | Yes | 検証するパスワード（平文） | null/空文字はfalse返却 |

### 入力データソース

- 内部呼び出し: EFLoginServiceからのメソッド呼び出し
- user: FindByUsernameで取得したユーザーオブジェクト
- password: ログインフォームから入力されたパスワード

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 検証結果 | bool | パスワードが正しい場合true、不正な場合false |

### 出力先

- 呼び出し元メソッド: EFLoginService.ValidateCredentials()の戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. ユーザーオブジェクトとパスワードの受け取り
   └─ ValidateCredentials(ApplicationUser user, string password)の引数として受信

2. UserManagerによるパスワード検証実行
   └─ _userManager.CheckPasswordAsync(user, password)を呼び出し
   └─ 内部でPasswordHasher.VerifyHashedPassword()が実行
   └─ PBKDF2アルゴリズムによるハッシュ比較

3. 検証結果の返却
   └─ パスワードが一致する場合: true
   └─ パスワードが不一致の場合: false
```

### フローチャート

```mermaid
flowchart TD
    A[ValidateCredentials呼び出し] --> B[ユーザーとパスワード受け取り]
    B --> C[UserManager.CheckPasswordAsync実行]
    C --> D[PasswordHasher.VerifyHashedPassword]
    D --> E{ハッシュ一致?}
    E -->|Yes| F[true を返却]
    E -->|No| G[false を返却]
    F --> H[終了]
    G --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-33-01 | ハッシュベース検証 | パスワードはハッシュ値との比較で検証 | 全検証リクエスト |
| BR-33-02 | タイミング攻撃対策 | 一定時間での比較を実行し、タイミング攻撃を防止 | 全検証リクエスト |
| BR-33-03 | ソルト使用 | パスワードハッシュにはユーザー固有のソルトを使用 | 全検証リクエスト |
| BR-33-04 | 反復回数確保 | PBKDF2で十分な反復回数を適用してブルートフォース攻撃を防止 | 全検証リクエスト |

### 計算ロジック

**パスワード検証アルゴリズム（PBKDF2）**:
1. 入力パスワードにストレージされたソルトを適用
2. 設定された反復回数でハッシュ関数を適用
3. 計算されたハッシュとストレージされたハッシュを比較
4. 一致すればtrue、不一致ならfalse

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| パスワードハッシュ取得 | AspNetUsers | SELECT | ユーザーのPasswordHashを参照（引数として渡される） |

### テーブル別操作詳細

#### AspNetUsers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| - | PasswordHash | 引数のuserオブジェクト経由で取得済み | 本機能ではDB直接アクセスなし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | パスワード不一致 | ハッシュが一致しない | falseを返却（例外は発生しない） |
| - | ユーザーnull | userパラメータがnull | 呼び出し元で事前チェックが必要 |
| - | パスワードnull/空 | passwordがnullまたは空文字 | falseを返却 |

### リトライ仕様

- 本機能自体にはリトライ機能なし
- アカウントロックアウトは呼び出し元（AccountController）で管理

## トランザクション仕様

- 読み取り専用のメモリ内操作のため、トランザクション管理は不要
- データベースアクセスなし（引数で渡されたオブジェクトを使用）

## パフォーマンス要件

- 検証処理: 100-500ms（PBKDF2の反復回数に依存）
- 意図的に計算コストを高く設定（ブルートフォース攻撃対策）

## セキュリティ考慮事項

- パスワードは平文でログ出力しない
- PBKDF2によるセキュアなハッシュ化
- タイミング攻撃を防止する定時間比較
- ユーザー固有のソルトによるレインボーテーブル攻撃対策
- 十分な反復回数によるブルートフォース攻撃対策

## 備考

- ASP.NET Core IdentityのPasswordHasherを使用
- デフォルトではPBKDF2-SHA256が使用される
- 反復回数はIdentityOptionsで設定可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、検証に使用されるユーザーエンティティの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | IdentityUserからPasswordHashフィールドを継承 |

**読解のコツ**: ApplicationUserはIdentityUserを継承しており、PasswordHashフィールドは継承元のIdentityUserで定義されている。このフィールドにPBKDF2でハッシュ化されたパスワードが格納される。

#### Step 2: インターフェースを理解する

サービスインターフェースで定義されたメソッドシグネチャを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | ValidateCredentialsメソッドの定義 |

**主要処理フロー**:
- **5行目**: `Task<bool> ValidateCredentials(T user, string password)` - 検証結果をboolで返却

#### Step 3: 実装を理解する

EFLoginServiceの具体的な実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | ValidateCredentialsの実装 |

**主要処理フロー**:
- **5行目**: UserManager<ApplicationUser>のインジェクション
- **19-22行目**: ValidateCredentialsの実装
- **21行目**: `_userManager.CheckPasswordAsync(user, password)` - 実際のパスワード検証

#### Step 4: 呼び出し元を理解する

AccountControllerでの使用パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | PasswordSignInAsyncの使用（内部でCheckPasswordAsyncを呼び出し） |

**主要処理フロー**:
- **95行目**: `_signInManager.PasswordSignInAsync()` - 内部でパスワード検証を実行

### プログラム呼び出し階層図

```
EFLoginService.ValidateCredentials(ApplicationUser user, string password)
    │
    └─ _userManager.CheckPasswordAsync(user, password)
           │
           └─ PasswordHasher<ApplicationUser>.VerifyHashedPassword()
                  │
                  ├─ PBKDF2ハッシュ計算
                  └─ ハッシュ値比較
```

### データフロー図

```
[入力]                        [処理]                         [出力]

ApplicationUser user    ───▶  EFLoginService              ───▶  bool
 └─ PasswordHash              .ValidateCredentials()            (true/false)
                                    │
string password         ───▶       ▼
(平文)                        UserManager
                              .CheckPasswordAsync()
                                    │
                                    ▼
                              PasswordHasher
                              .VerifyHashedPassword()
                                    │
                                    ▼
                              PBKDF2-SHA256比較
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | ソース | パスワード検証機能の実装 |
| ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | ソース | サービスインターフェース定義 |
| ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | ソース | ユーザーエンティティモデル（PasswordHash含む） |
| AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | ソース | 呼び出し元コントローラー |
