# 機能設計書 34-サインイン処理

## 概要

本ドキュメントは、eShopアプリケーションにおけるサインイン処理機能の設計仕様を定義するものである。本機能は、Identity.APIサービスが提供する認証基盤の一部であり、認証成功後のユーザーセッションを確立する機能を提供する。

### 本機能の処理概要

本機能は、パスワード検証成功後にユーザーの認証状態を確立し、認証クッキーを発行することで、後続のリクエストでユーザーを識別可能にする。ASP.NET Core IdentityのSignInManagerを活用し、安全なセッション管理を実現する。

**業務上の目的・背景**：認証処理においてパスワード検証が成功した後、ユーザーのセッションを確立し、認証状態を維持する必要がある。これにより、ユーザーは一度ログインすれば、セッションが有効な間は再認証なしでシステムを利用できる。本機能は、この認証状態の永続化を担当し、ユーザー体験の向上とセキュリティの両立を実現する。

**機能の利用シーン**：ログイン認証成功後の認証状態確立、OAuth同意画面での認証処理、デバイスフロー認証での認証状態確立など、様々な認証フローの最終段階で使用される。

**主要な処理内容**：
1. ユーザーオブジェクトと認証オプションの受け取り
2. SignInManagerによる認証クッキーの生成
3. ユーザープリンシパルの構築とクレームの設定
4. 認証クッキーの発行（HttpOnlyクッキー）
5. 永続ログイン設定の適用（Remember Me機能）

**関連システム・外部連携**：ASP.NET Core IdentityのSignInManagerと連携し、クッキー認証ミドルウェアを介して認証クッキーを発行する。IdentityServer4との連携により、OAuth/OIDCフローでのセッション管理も実現する。

**権限による制御**：本機能は内部サービスとして使用され、認証成功後にのみ呼び出される。サインイン処理自体には権限制御はないが、サインインされたユーザーには認証済みユーザーとしての権限が付与される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | ログイン画面 | 主画面 | 認証成功時のサインイン状態確立 |
| 14 | 同意画面 | 参照画面 | 同意後のサインイン処理 |
| 15 | 権限管理画面 | 参照画面 | 権限取り消し処理 |
| 18 | デバイス確認画面 | 参照画面 | デバイス認証同意後の処理 |

## 機能種別

セキュリティ機能 / セッション管理

## 入力仕様

### 入力パラメータ

#### SignIn メソッド
| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| user | ApplicationUser | Yes | サインイン対象のユーザー | null不可 |

#### SignInAsync メソッド
| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| user | ApplicationUser | Yes | サインイン対象のユーザー | null不可 |
| properties | AuthenticationProperties | Yes | 認証プロパティ（永続化設定等） | null不可 |
| authenticationMethod | string | No | 使用された認証方法の識別子 | - |

### 入力データソース

- 内部呼び出し: EFLoginServiceからのメソッド呼び出し
- user: FindByUsernameで取得し、ValidateCredentialsで検証されたユーザー
- properties: 永続ログイン設定やリダイレクトURL等を含む

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 認証クッキー | Cookie | .AspNetCore.Cookies認証クッキー |
| ClaimsPrincipal | ClaimsPrincipal | ユーザーの認証情報を含むプリンシパル |

### 出力先

- ブラウザクッキー: HTTP応答に認証クッキーを含める
- HttpContext.User: 認証済みユーザープリンシパルを設定

## 処理フロー

### 処理シーケンス

```
1. ユーザーオブジェクトの受け取り
   └─ SignIn(ApplicationUser user)またはSignInAsync()の引数として受信

2. ユーザープリンシパルの構築
   └─ SignInManager内部でClaimsPrincipalを生成
   └─ ユーザーID、名前、ロール等のクレームを設定

3. 認証クッキーの生成
   └─ クッキー認証ハンドラーがクッキーを生成
   └─ HttpOnlyフラグを設定（XSS対策）
   └─ Secureフラグを設定（HTTPS通信時のみ送信）

4. 永続化オプションの適用
   └─ IsPersistentがtrueの場合、永続クッキーを発行
   └─ ExpiresUtcで有効期限を設定

5. HttpContext.Userへの設定
   └─ 現在のリクエストコンテキストにユーザー情報を設定
```

### フローチャート

```mermaid
flowchart TD
    A[SignIn/SignInAsync呼び出し] --> B[ユーザーオブジェクト受け取り]
    B --> C[ClaimsPrincipal構築]
    C --> D{永続ログイン?}
    D -->|Yes| E[永続クッキー生成]
    D -->|No| F[セッションクッキー生成]
    E --> G[認証クッキーをレスポンスに追加]
    F --> G
    G --> H[HttpContext.Userを設定]
    H --> I[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-34-01 | HttpOnlyクッキー | 認証クッキーはHttpOnlyフラグを設定しJavaScriptからアクセス不可 | 全サインイン処理 |
| BR-34-02 | セキュアクッキー | HTTPS環境ではSecureフラグを設定 | HTTPS通信時 |
| BR-34-03 | 永続ログイン | Remember Me選択時は永続クッキーを発行 | IsPersistent=true時 |
| BR-34-04 | セッション有効期限 | デフォルトのセッション有効期限を適用 | 永続ログイン非選択時 |
| BR-34-05 | 認証方法記録 | 使用された認証方法（pwd、external等）をクレームに記録 | authenticationMethod指定時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能ではデータベースアクセスなし |

### テーブル別操作詳細

本機能はメモリ内でのセッション確立処理であり、データベースへの直接的なアクセスは行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | userパラメータがnull | 呼び出し元で事前チェックが必要 |
| - | クッキー生成エラー | 内部エラー | エラーログ出力、ログイン失敗として処理 |

### リトライ仕様

- 本機能自体にはリトライ機能なし
- エラー発生時は呼び出し元でリトライ判断

## トランザクション仕様

- メモリ内操作のため、トランザクション管理は不要

## パフォーマンス要件

- サインイン処理: 50ms以内の応答
- クッキー生成は軽量な処理

## セキュリティ考慮事項

- HttpOnlyフラグによるXSS攻撃対策
- Secureフラグによる通信経路保護
- SameSiteフラグによるCSRF対策
- 認証クッキーの暗号化（DataProtection API）
- セッションの有効期限設定

## 備考

- ASP.NET Core IdentityのSignInManager<T>を活用
- IdentityServer4との統合により、OAuth/OIDCセッション管理も実現
- 2種類のメソッド（SignIn、SignInAsync）を提供し、異なるユースケースに対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、サインイン処理で使用されるデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | サインイン対象のユーザーエンティティ |

**読解のコツ**: ApplicationUserはIdentityUserを継承しており、サインイン時にユーザー情報がClaimsPrincipalに変換される。Id、UserName、Emailなどがクレームとして設定される。

#### Step 2: インターフェースを理解する

サービスインターフェースで定義されたメソッドシグネチャを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | SignIn/SignInAsyncメソッドの定義 |

**主要処理フロー**:
- **9行目**: `Task SignIn(T user)` - シンプルなサインイン
- **11行目**: `Task SignInAsync(T user, AuthenticationProperties properties, string authenticationMethod = null)` - 詳細オプション付きサインイン

#### Step 3: 実装を理解する

EFLoginServiceの具体的な実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | SignIn/SignInAsyncの実装 |

**主要処理フロー**:
- **6行目**: SignInManager<ApplicationUser>のインジェクション
- **24-27行目**: SignInの実装（永続ログイン=true固定）
- **29-32行目**: SignInAsyncの実装（カスタムプロパティ対応）

#### Step 4: 呼び出し元を理解する

AccountControllerでの使用パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | SignInManagerの使用パターン |

**主要処理フロー**:
- **11行目**: SignInManager<ApplicationUser>のインジェクション
- **95行目**: PasswordSignInAsync - 内部でSignInAsyncを呼び出す
- **175行目**: SignOutAsync - ログアウト処理

### プログラム呼び出し階層図

```
EFLoginService.SignIn(ApplicationUser user)
    │
    └─ _signInManager.SignInAsync(user, true)
           │
           ├─ ClaimsFactory.CreateAsync() ─── ClaimsPrincipal構築
           │
           └─ HttpContext.SignInAsync()
                  │
                  └─ CookieAuthenticationHandler
                         │
                         ├─ クッキー生成
                         └─ DataProtection暗号化

EFLoginService.SignInAsync(ApplicationUser user, AuthenticationProperties properties, string authenticationMethod)
    │
    └─ _signInManager.SignInAsync(user, properties, authenticationMethod)
           │
           └─ (上記と同様)
```

### データフロー図

```
[入力]                        [処理]                         [出力]

ApplicationUser user    ───▶  EFLoginService              ───▶  認証クッキー
                              .SignIn()                        (.AspNetCore.Cookies)
                                    │
                                    ▼
                              SignInManager
                              .SignInAsync()
                                    │
                              ┌─────┴─────┐
                              ▼           ▼
                         ClaimsFactory   HttpContext
                         (プリンシパル    .SignInAsync()
                          構築)                │
                                               ▼
                                         CookieAuthHandler
                                         (クッキー発行)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| EFLoginService.cs | `src/Identity.API/Services/EFLoginService.cs` | ソース | サインイン処理の実装 |
| ILoginService.cs | `src/Identity.API/Services/ILoginService.cs` | ソース | サービスインターフェース定義 |
| ApplicationUser.cs | `src/Identity.API/Models/ApplicationUser.cs` | ソース | ユーザーエンティティモデル |
| AccountController.cs | `src/Identity.API/Quickstart/Account/AccountController.cs` | ソース | 呼び出し元コントローラー |
| ConsentController.cs | `src/Identity.API/Quickstart/Consent/ConsentController.cs` | ソース | OAuth同意後のサインイン処理 |
| DeviceController.cs | `src/Identity.API/Quickstart/Device/DeviceController.cs` | ソース | デバイスフロー認証でのサインイン |
