# 機能設計書 35-Webhook一覧取得

## 概要

本ドキュメントは、eShopアプリケーションにおけるWebhook一覧取得機能の設計仕様を定義するものである。本機能は、Webhooks.APIサービスが提供するWebhook管理機能の一部であり、ログインユーザーが登録したWebhookサブスクリプションの一覧を取得する。

### 本機能の処理概要

本機能は、認証済みユーザーが自身で登録したWebhookサブスクリプションの一覧を取得する機能を提供する。RESTful APIエンドポイント（GET /api/webhooks）として実装され、ユーザーごとのサブスクリプション管理を可能にする。

**業務上の目的・背景**：マイクロサービスアーキテクチャにおいて、外部システムとの連携にWebhookは重要な役割を果たす。eShopでは、注文の支払完了（OrderPaid）、注文発送（OrderShipped）、商品価格変更（CatalogItemPriceChange）などのイベントをWebhook経由で外部システムに通知できる。本機能は、ユーザーが登録したWebhookサブスクリプションを確認・管理するための基盤機能である。

**機能の利用シーン**：WebhookClientアプリケーションのホーム画面で登録済みWebhook一覧を表示する際に使用される。ユーザーは自身が登録したWebhookの状態を確認し、必要に応じて追加・削除の判断を行う。

**主要な処理内容**：
1. 認証済みユーザーのIDを取得
2. WebhooksContextからユーザーに紐づくサブスクリプションを取得
3. Webhookサブスクリプション一覧をJSON形式で返却

**関連システム・外部連携**：WebhookClientアプリケーションから呼び出される。Identity.APIから発行されたアクセストークンによる認証が必要。

**権限による制御**：認証済みユーザーのみがアクセス可能。各ユーザーは自身のサブスクリプションのみ取得可能（他ユーザーのサブスクリプションは取得不可）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 21 | ホーム画面 | 主画面 | WebhooksClient.LoadWebhooksで登録済みWebhook一覧を取得 |

## 機能種別

CRUD操作（Read） / データ検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | URLパラメータなし | - |

### 認証情報

| 項目 | 説明 |
|-----|------|
| Authorization | Bearer {access_token} |
| User ID | JWTトークンのsubクレームから取得 |

### 入力データソース

- HTTPリクエストヘッダー: Authorizationヘッダーからアクセストークン
- ClaimsPrincipal: トークンからユーザーID（sub）を抽出

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| WebhookSubscription[] | Array | Webhookサブスクリプション一覧 |

### WebhookSubscriptionの構造

| フィールド名 | 型 | 説明 |
|-------------|-----|------|
| Id | int | サブスクリプションID |
| Type | WebhookType | Webhookタイプ（OrderPaid, OrderShipped, CatalogItemPriceChange） |
| Date | DateTime | 登録日時 |
| DestUrl | string | Webhook送信先URL |
| Token | string | 認証用トークン |
| UserId | string | 所有者のユーザーID |

### 出力先

- HTTPレスポンス: JSON形式でWebhookサブスクリプション配列を返却
- HTTPステータス: 200 OK

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエストの受信
   └─ GET /api/webhooks

2. 認証・認可の確認
   └─ Authorizationヘッダーからトークンを検証
   └─ ClaimsPrincipalからユーザーIDを取得

3. データベースからサブスクリプション取得
   └─ WebhooksContext.Subscriptionsに対してクエリ
   └─ WHERE UserId = @userId でフィルタリング

4. レスポンスの返却
   └─ WebhookSubscription配列をJSONシリアライズ
   └─ 200 OKで返却
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/webhooks] --> B[認証トークン検証]
    B --> C{認証成功?}
    C -->|No| D[401 Unauthorized]
    C -->|Yes| E[ユーザーID取得]
    E --> F[DBからサブスクリプション取得]
    F --> G[JSON形式でレスポンス]
    G --> H[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-35-01 | ユーザースコープ | 各ユーザーは自身のサブスクリプションのみ取得可能 | 全リクエスト |
| BR-35-02 | 認証必須 | 有効なアクセストークンが必要 | 全リクエスト |
| BR-35-03 | 全件取得 | ページネーションなしで全サブスクリプションを取得 | 現在の実装 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| サブスクリプション取得 | Subscriptions | SELECT | ユーザーIDでフィルタリングして取得 |

### テーブル別操作詳細

#### Subscriptions

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Type, Date, DestUrl, Token, UserId | WHERE UserId = @userId | ユーザーのサブスクリプション全件 |

### インデックス

| テーブル | インデックス | カラム |
|---------|------------|-------|
| Subscriptions | IX_Subscriptions_UserId | UserId |
| Subscriptions | IX_Subscriptions_Type | Type |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証トークンが無効または欠落 | 再ログインを促す |
| 500 | Internal Server Error | データベースアクセスエラー | エラーログ出力、リトライ |

### リトライ仕様

- クライアント側でリトライを実装
- サーバー側ではEntity Frameworkのリトライポリシーに従う

## トランザクション仕様

- 読み取り専用操作のため、トランザクション管理は不要
- 読み取り一貫性はEntity Frameworkの既定動作に依存

## パフォーマンス要件

- 検索処理: 500ms以内の応答
- UserIdインデックスによる高速検索

## セキュリティ考慮事項

- Bearer Token認証による認可
- ユーザースコープによるデータ分離
- HTTPS通信による暗号化
- JWTトークンの検証（署名、有効期限）

## 備考

- ASP.NET Core Minimal APIsを使用した実装
- Entity Framework Coreによるデータアクセス
- WebhookClientアプリケーションとの連携を想定

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Webhookサブスクリプションのデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | サブスクリプションエンティティの定義 |
| 1-2 | WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | Webhookタイプの列挙型 |

**読解のコツ**: WebhookSubscriptionは、どのユーザー（UserId）が、どのイベント（Type）を、どこに（DestUrl）通知するかを定義している。

#### Step 2: エントリーポイントを理解する

API定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | Minimal APIs定義 |

**主要処理フロー**:
- **9-11行目**: ルートグループの定義（/api/webhooks）
- **13-18行目**: GET / - 一覧取得エンドポイント
- **15行目**: `user.GetUserId()` - ClaimsPrincipalからユーザーID取得
- **16行目**: `context.Subscriptions.Where(s => s.UserId == userId)` - フィルタリング

#### Step 3: データアクセス層を理解する

WebhooksContextを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | DbContext定義 |

**主要処理フロー**:
- **10行目**: `DbSet<WebhookSubscription> Subscriptions` - テーブルマッピング
- **14-18行目**: インデックス設定（UserId, Type）

### プログラム呼び出し階層図

```
GET /api/webhooks
    │
    ├─ ClaimsPrincipal.GetUserId()
    │      └─ JWTトークンからsubクレーム取得
    │
    └─ WebhooksContext.Subscriptions
           │
           └─ .Where(s => s.UserId == userId)
                  │
                  └─ .ToListAsync()
                         │
                         └─ SQL Server (Subscriptions テーブル)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Authorization Header ───▶ ClaimsPrincipal           ───▶ WebhookSubscription[]
(Bearer Token)            .GetUserId()                   (JSON)
                              │
                              ▼
                         WebhooksContext
                         .Subscriptions
                              │
                              ▼
                         WHERE UserId = @userId
                              │
                              ▼
                         TypedResults.Ok(data)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | ソース | APIエンドポイント定義 |
| WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | ソース | エンティティモデル |
| WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | ソース | Webhookタイプ列挙型 |
| WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | ソース | DbContext |
| Extensions.cs | `src/Webhooks.API/Extensions/Extensions.cs` | ソース | 拡張メソッド（GetUserId等） |
| Program.cs | `src/Webhooks.API/Program.cs` | ソース | アプリケーション構成 |
