# 機能設計書 36-Webhook詳細取得

## 概要

本ドキュメントは、eShopアプリケーションにおけるWebhook詳細取得機能の設計仕様を定義するものである。本機能は、Webhooks.APIサービスが提供するWebhook管理機能の一部であり、指定されたIDのWebhookサブスクリプション詳細情報を取得する。

### 本機能の処理概要

本機能は、認証済みユーザーが自身で登録した特定のWebhookサブスクリプションの詳細情報を取得する機能を提供する。RESTful APIエンドポイント（GET /api/webhooks/{id}）として実装され、サブスクリプションの詳細確認を可能にする。

**業務上の目的・背景**：Webhook管理において、個別のサブスクリプション詳細を確認する必要がある場合がある。例えば、特定のWebhookの設定内容（送信先URL、認証トークン、イベントタイプ）を確認したい場合や、トラブルシューティング時に設定を検証する場合に使用される。本機能は、このような詳細確認のニーズに応える。

**機能の利用シーン**：WebhookClientアプリケーションで特定のWebhook設定を詳細表示する際、またはAPI経由でプログラム的に設定を取得する際に使用される。

**主要な処理内容**：
1. 認証済みユーザーのIDとサブスクリプションIDを取得
2. WebhooksContextから指定IDのサブスクリプションを取得
3. ユーザーIDの一致を確認（所有者チェック）
4. サブスクリプション詳細をJSON形式で返却

**関連システム・外部連携**：WebhookClientアプリケーションから呼び出される。Identity.APIから発行されたアクセストークンによる認証が必要。

**権限による制御**：認証済みユーザーのみがアクセス可能。各ユーザーは自身のサブスクリプションのみ取得可能（他ユーザーのサブスクリプションは取得不可、存在しても404を返却）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 21 | ホーム画面 | 参照画面 | 登録済みWebhook詳細の確認 |

## 機能種別

CRUD操作（Read） / データ検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | サブスクリプションID | 正の整数 |

### 認証情報

| 項目 | 説明 |
|-----|------|
| Authorization | Bearer {access_token} |
| User ID | JWTトークンのsubクレームから取得 |

### 入力データソース

- URLパス: /api/webhooks/{id} からサブスクリプションID
- HTTPリクエストヘッダー: Authorizationヘッダーからアクセストークン
- ClaimsPrincipal: トークンからユーザーID（sub）を抽出

## 出力仕様

### 出力データ

#### 成功時（200 OK）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| WebhookSubscription | Object | Webhookサブスクリプション詳細 |

#### 失敗時（404 Not Found）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| message | string | "Subscriptions {id} not found" |

### WebhookSubscriptionの構造

| フィールド名 | 型 | 説明 |
|-------------|-----|------|
| Id | int | サブスクリプションID |
| Type | WebhookType | Webhookタイプ |
| Date | DateTime | 登録日時 |
| DestUrl | string | Webhook送信先URL |
| Token | string | 認証用トークン |
| UserId | string | 所有者のユーザーID |

### 出力先

- HTTPレスポンス: JSON形式でWebhookサブスクリプションを返却
- HTTPステータス: 200 OK または 404 Not Found

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエストの受信
   └─ GET /api/webhooks/{id}

2. 認証・認可の確認
   └─ Authorizationヘッダーからトークンを検証
   └─ ClaimsPrincipalからユーザーIDを取得

3. データベースからサブスクリプション取得
   └─ WebhooksContext.Subscriptionsに対してクエリ
   └─ WHERE Id = @id AND UserId = @userId でフィルタリング
   └─ SingleOrDefaultAsync()で単一結果取得

4. 結果判定と返却
   └─ サブスクリプションが存在する場合: 200 OK + JSON
   └─ サブスクリプションが存在しない場合: 404 Not Found
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/webhooks/id] --> B[認証トークン検証]
    B --> C{認証成功?}
    C -->|No| D[401 Unauthorized]
    C -->|Yes| E[ユーザーID取得]
    E --> F[DBからサブスクリプション検索]
    F --> G{サブスクリプション存在?<br>かつ所有者一致?}
    G -->|Yes| H[200 OK + JSON]
    G -->|No| I[404 Not Found]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-36-01 | ユーザースコープ | 各ユーザーは自身のサブスクリプションのみ取得可能 | 全リクエスト |
| BR-36-02 | 認証必須 | 有効なアクセストークンが必要 | 全リクエスト |
| BR-36-03 | 所有者チェック | IDが存在しても、所有者でない場合は404を返却 | セキュリティ対策 |
| BR-36-04 | 単一結果保証 | IDはユニークなため、結果は0件または1件 | データ整合性 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| サブスクリプション取得 | Subscriptions | SELECT | IDとユーザーIDで単一レコード取得 |

### テーブル別操作詳細

#### Subscriptions

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Type, Date, DestUrl, Token, UserId | WHERE Id = @id AND UserId = @userId | SingleOrDefaultAsync |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証トークンが無効または欠落 | 再ログインを促す |
| 404 | Not Found | サブスクリプションが存在しない、または他ユーザーの所有 | クライアントでエラー表示 |
| 500 | Internal Server Error | データベースアクセスエラー | エラーログ出力、リトライ |

### リトライ仕様

- クライアント側でリトライを実装
- サーバー側ではEntity Frameworkのリトライポリシーに従う

## トランザクション仕様

- 読み取り専用操作のため、トランザクション管理は不要

## パフォーマンス要件

- 検索処理: 200ms以内の応答
- 主キー検索による高速アクセス

## セキュリティ考慮事項

- Bearer Token認証による認可
- ユーザースコープによるデータ分離（他ユーザーのデータは404として扱う）
- HTTPS通信による暗号化
- ID推測攻撃対策（所有者チェックにより、IDを知っていても他ユーザーのデータはアクセス不可）

## 備考

- ASP.NET Core Minimal APIsを使用した実装
- TypedResultsを使用した型安全なレスポンス返却

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Webhookサブスクリプションのデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | サブスクリプションエンティティの定義 |
| 1-2 | WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | Webhookタイプの列挙型 |

**読解のコツ**: WebhookSubscriptionのIdフィールドが主キーとして使用され、SingleOrDefaultAsyncでの一意検索の基盤となる。

#### Step 2: エントリーポイントを理解する

API定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | Minimal APIs定義 |

**主要処理フロー**:
- **20-33行目**: GET /{id:int} - 詳細取得エンドポイント
- **25-26行目**: `context.Subscriptions.SingleOrDefaultAsync(s => s.Id == id && s.UserId == userId)` - ID+ユーザーIDでフィルタリング
- **27-30行目**: 存在チェックと200 OK返却
- **32行目**: 存在しない場合の404 Not Found返却

#### Step 3: データアクセス層を理解する

WebhooksContextを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | DbContext定義 |

### プログラム呼び出し階層図

```
GET /api/webhooks/{id}
    │
    ├─ ClaimsPrincipal.GetUserId()
    │      └─ JWTトークンからsubクレーム取得
    │
    └─ WebhooksContext.Subscriptions
           │
           └─ .SingleOrDefaultAsync(s => s.Id == id && s.UserId == userId)
                  │
                  ├─ 存在する場合 → TypedResults.Ok(subscription)
                  │
                  └─ 存在しない場合 → TypedResults.NotFound("...")
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URL Path /{id}     ───▶  WebhooksContext             ───▶ WebhookSubscription
                         .Subscriptions                    (JSON)
Authorization Header      .SingleOrDefaultAsync()          または
(Bearer Token)                 │                           404 Not Found
       │                       │
       ▼                       ▼
ClaimsPrincipal ──────▶ WHERE Id = @id
.GetUserId()            AND UserId = @userId
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | ソース | APIエンドポイント定義 |
| WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | ソース | エンティティモデル |
| WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | ソース | Webhookタイプ列挙型 |
| WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | ソース | DbContext |
