# 機能設計書 37-Webhook登録

## 概要

本ドキュメントは、eShopアプリケーションにおけるWebhook登録機能の設計仕様を定義するものである。本機能は、Webhooks.APIサービスが提供するWebhook管理機能の中核であり、新しいWebhookサブスクリプションを登録する機能を提供する。

### 本機能の処理概要

本機能は、認証済みユーザーが新しいWebhookサブスクリプションを登録する機能を提供する。RESTful APIエンドポイント（POST /api/webhooks）として実装され、Grant URLによる所有権検証を含むセキュアな登録プロセスを実現する。

**業務上の目的・背景**：外部システムとeShopの連携において、Webhookは重要な役割を果たす。ユーザーが自身のシステムにeShopからのイベント通知（注文支払完了、注文発送、価格変更）を受信するためには、Webhookサブスクリプションの登録が必要である。本機能は、セキュリティを確保しながら、この登録プロセスを提供する。特に、Grant URL検証により、悪意のある第三者が他人のサーバーにWebhook通知を送信させることを防止する。

**機能の利用シーン**：WebhookClientアプリケーションのWebhook追加画面から、OrderPaidイベントなどの通知を受け取るためのサブスクリプションを登録する際に使用される。

**主要な処理内容**：
1. 認証済みユーザーの確認
2. リクエストボディのバリデーション
3. Grant URL検証（No.40機能を呼び出し）
4. サブスクリプションエンティティの作成
5. データベースへの保存

**関連システム・外部連携**：WebhookClientアプリケーションから呼び出される。Grant URL検証のため、登録先の外部サーバーへのOPTIONSリクエストを送信する。

**権限による制御**：認証済みユーザーのみがアクセス可能。各ユーザーは自身のサブスクリプションのみ登録可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 22 | Webhook追加画面 | 主画面 | WebhooksClient.AddWebHookAsyncでWebhook登録 |

## 機能種別

CRUD操作（Create） / データ登録

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Url | string | Yes | Webhook送信先URL | 有効なURL形式（Uri.IsWellFormedUriString） |
| Token | string | No | 認証用トークン | - |
| Event | string | Yes | イベントタイプ | WebhookType列挙値（OrderPaid, OrderShipped, CatalogItemPriceChange） |
| GrantUrl | string | Yes | Grant検証用URL | 有効なURL形式、Urlと同一オリジン |

### WebhookSubscriptionRequestの構造

| フィールド名 | 型 | 説明 |
|-------------|-----|------|
| Url | string | Webhook送信先URL |
| Token | string | 認証用トークン（オプション） |
| Event | string | イベントタイプ文字列 |
| GrantUrl | string | Grant検証用URL |

### 認証情報

| 項目 | 説明 |
|-----|------|
| Authorization | Bearer {access_token} |
| User ID | JWTトークンのsubクレームから取得 |

### 入力データソース

- HTTPリクエストボディ: JSON形式のWebhookSubscriptionRequest
- HTTPリクエストヘッダー: Authorizationヘッダーからアクセストークン
- ClaimsPrincipal: トークンからユーザーID（sub）を抽出

## 出力仕様

### 出力データ

#### 成功時（201 Created）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Location | Header | 作成されたリソースのURI（/api/webhooks/{id}） |

#### 失敗時（400 Bad Request）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| message | string | エラーメッセージ |

### 出力先

- HTTPレスポンス: 201 Created + Locationヘッダー
- または 400 Bad Request + エラーメッセージ

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエストの受信
   └─ POST /api/webhooks

2. 認証・認可の確認
   └─ Authorizationヘッダーからトークンを検証

3. リクエストバリデーション
   └─ ValidateWebhookSubscriptionRequest()エンドポイントフィルター
   └─ Url, GrantUrlの形式チェック
   └─ Eventの列挙値チェック

4. Grant URL検証（機能No.40）
   └─ IGrantUrlTesterService.TestGrantUrl()呼び出し
   └─ 外部サーバーへOPTIONSリクエスト送信
   └─ トークン照合による所有権確認

5. サブスクリプション作成
   └─ WebhookSubscriptionエンティティ生成
   └─ Date = DateTime.UtcNow
   └─ UserId = user.GetUserId()

6. データベース保存
   └─ context.Add(subscription)
   └─ context.SaveChangesAsync()

7. レスポンス返却
   └─ Grant検証成功: 201 Created
   └─ Grant検証失敗: 400 Bad Request
```

### フローチャート

```mermaid
flowchart TD
    A[POST /api/webhooks] --> B[認証トークン検証]
    B --> C{認証成功?}
    C -->|No| D[401 Unauthorized]
    C -->|Yes| E[リクエストバリデーション]
    E --> F{バリデーション成功?}
    F -->|No| G[422 Validation Problem]
    F -->|Yes| H[Grant URL検証]
    H --> I{Grant検証成功?}
    I -->|No| J[400 Bad Request]
    I -->|Yes| K[サブスクリプション作成]
    K --> L[DB保存]
    L --> M[201 Created]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-37-01 | Grant URL検証必須 | 登録前にGrant URLの所有権を検証 | 全登録リクエスト |
| BR-37-02 | 同一オリジン | UrlとGrantUrlは同一オリジンである必要 | 全登録リクエスト |
| BR-37-03 | イベントタイプ制限 | 定義済みのWebhookTypeのみ登録可能 | 全登録リクエスト |
| BR-37-04 | 登録日時記録 | 登録日時はUTC時刻で記録 | 全登録処理 |
| BR-37-05 | ユーザー紐付け | サブスクリプションは登録ユーザーに紐付け | 全登録処理 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| サブスクリプション登録 | Subscriptions | INSERT | 新規サブスクリプション追加 |

### テーブル別操作詳細

#### Subscriptions

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | Id | 自動採番 | 主キー |
| INSERT | Type | Enum.Parse<WebhookType>(request.Event) | イベントタイプ |
| INSERT | Date | DateTime.UtcNow | 登録日時 |
| INSERT | DestUrl | request.Url | 送信先URL |
| INSERT | Token | request.Token | 認証トークン（nullable） |
| INSERT | UserId | user.GetUserId() | 所有者ID |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証トークンが無効または欠落 | 再ログインを促す |
| 400 | Bad Request | Grant URL検証失敗 | "Invalid grant URL: {url}" |
| 422 | Validation Problem | バリデーションエラー | 各フィールドのエラー詳細を返却 |
| 500 | Internal Server Error | データベースアクセスエラー | エラーログ出力、リトライ |

### リトライ仕様

- Grant URL検証の失敗はリトライ対象外（ユーザー設定の問題）
- データベースエラーはEntity Frameworkのリトライポリシーに従う

## トランザクション仕様

- SaveChangesAsync()内で暗黙的にトランザクション管理
- 保存失敗時は自動ロールバック

## パフォーマンス要件

- 登録処理: 3秒以内の応答（外部Grant URL検証を含む）
- Grant URL検証のタイムアウト設定が必要

## セキュリティ考慮事項

- Bearer Token認証による認可
- Grant URL検証によるWebhook送信先の所有権確認
- 同一オリジンポリシーによるセキュリティ強化
- X-eshop-whtokenヘッダーによるトークン照合
- HTTPS通信による暗号化

## 備考

- ASP.NET Core Minimal APIsを使用した実装
- IValidatableObjectインターフェースによるカスタムバリデーション
- ValidateWebhookSubscriptionRequestエンドポイントフィルターによるバリデーション

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、リクエスト/レスポンスのデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebhookSubscriptionRequest.cs | `src/Webhooks.API/Model/WebhookSubscriptionRequest.cs` | 登録リクエストの構造とバリデーション |
| 1-2 | WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | 登録されるエンティティの構造 |
| 1-3 | WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | 有効なイベントタイプ |

**読解のコツ**: WebhookSubscriptionRequestにIValidatableObjectが実装されており、Validate()メソッドでカスタムバリデーションが定義されている（10-27行目）。

#### Step 2: エントリーポイントを理解する

API定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | POSTエンドポイント定義 |

**主要処理フロー**:
- **35-64行目**: POST / - 登録エンドポイント
- **41行目**: `grantUrlTester.TestGrantUrl()` - Grant URL検証呼び出し
- **45-52行目**: サブスクリプションエンティティ作成
- **54-55行目**: DB保存
- **57行目**: 201 Created返却
- **61行目**: Grant検証失敗時400 Bad Request
- **64行目**: `.ValidateWebhookSubscriptionRequest()` - エンドポイントフィルター

#### Step 3: バリデーション層を理解する

バリデーションロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RouteHandlerBuilderExtensions.cs | `src/Webhooks.API/Extensions/RouteHandlerBuilderExtensions.cs` | エンドポイントフィルター実装 |

**主要処理フロー**:
- **5-25行目**: ValidateWebhookSubscriptionRequest - バリデーションフィルター
- **16行目**: webhookSubscriptionRequest.Validate() - カスタムバリデーション実行

#### Step 4: Grant URL検証を理解する

Grant URL検証サービスを確認する（機能No.40と関連）。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | IGrantUrlTesterService.cs | `src/Webhooks.API/Services/IGrantUrlTesterService.cs` | サービスインターフェース |
| 4-2 | GrantUrlTesterService.cs | `src/Webhooks.API/Services/GrantUrlTesterService.cs` | 検証実装 |

### プログラム呼び出し階層図

```
POST /api/webhooks
    │
    ├─ ValidateWebhookSubscriptionRequest (エンドポイントフィルター)
    │      │
    │      └─ WebhookSubscriptionRequest.Validate()
    │             ├─ Uri.IsWellFormedUriString(GrantUrl)
    │             ├─ Uri.IsWellFormedUriString(Url)
    │             └─ Enum.TryParse(Event, WebhookType)
    │
    ├─ IGrantUrlTesterService.TestGrantUrl()
    │      │
    │      ├─ CheckSameOrigin(urlHook, url)
    │      └─ HttpClient.SendAsync(OPTIONS request)
    │
    ├─ new WebhookSubscription()
    │
    ├─ WebhooksContext.Add(subscription)
    │
    └─ WebhooksContext.SaveChangesAsync()
```

### データフロー図

```
[入力]                        [処理]                         [出力]

WebhookSubscriptionRequest ───▶ ValidateWebhookSubscription ───▶ 422 (バリデーションエラー)
 ├─ Url                              │                           または
 ├─ Token                            ▼                           400 (Grant検証失敗)
 ├─ Event                      GrantUrlTesterService             または
 └─ GrantUrl                   .TestGrantUrl()                   201 Created
                                     │
Authorization Header                 ▼
(Bearer Token)               WebhookSubscription
       │                      作成・保存
       ▼                           │
ClaimsPrincipal ────────────────────┘
.GetUserId()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | ソース | APIエンドポイント定義 |
| WebhookSubscriptionRequest.cs | `src/Webhooks.API/Model/WebhookSubscriptionRequest.cs` | ソース | リクエストモデル・バリデーション |
| WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | ソース | エンティティモデル |
| WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | ソース | Webhookタイプ列挙型 |
| RouteHandlerBuilderExtensions.cs | `src/Webhooks.API/Extensions/RouteHandlerBuilderExtensions.cs` | ソース | バリデーションフィルター |
| IGrantUrlTesterService.cs | `src/Webhooks.API/Services/IGrantUrlTesterService.cs` | ソース | Grant検証インターフェース |
| GrantUrlTesterService.cs | `src/Webhooks.API/Services/GrantUrlTesterService.cs` | ソース | Grant検証実装 |
| WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | ソース | DbContext |
