# 機能設計書 38-Webhook削除

## 概要

本ドキュメントは、eShopアプリケーションにおけるWebhook削除機能の設計仕様を定義するものである。本機能は、Webhooks.APIサービスが提供するWebhook管理機能の一部であり、登録済みのWebhookサブスクリプションを削除する機能を提供する。

### 本機能の処理概要

本機能は、認証済みユーザーが自身で登録したWebhookサブスクリプションを削除する機能を提供する。RESTful APIエンドポイント（DELETE /api/webhooks/{id}）として実装され、セキュアな削除プロセスを実現する。

**業務上の目的・背景**：Webhook管理において、不要になったサブスクリプションを削除する機能は必須である。外部システムの廃止、URLの変更、イベント通知の不要化など、様々な理由でサブスクリプションの削除が必要になる。本機能は、ユーザーが自身のサブスクリプションのみを削除できるよう、所有者チェックを含めたセキュアな削除機能を提供する。

**機能の利用シーン**：WebhookClientアプリケーションのホーム画面で、登録済みWebhookの削除操作を行う際に使用される。また、API経由でプログラム的にサブスクリプションを管理する際にも使用される。

**主要な処理内容**：
1. 認証済みユーザーのIDとサブスクリプションIDを取得
2. WebhooksContextから指定IDのサブスクリプションを検索
3. ユーザーIDの一致を確認（所有者チェック）
4. サブスクリプションの削除とデータベース保存

**関連システム・外部連携**：WebhookClientアプリケーションから呼び出される。Identity.APIから発行されたアクセストークンによる認証が必要。

**権限による制御**：認証済みユーザーのみがアクセス可能。各ユーザーは自身のサブスクリプションのみ削除可能（他ユーザーのサブスクリプションは削除不可、存在しても404を返却）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 21 | ホーム画面 | 主画面 | 登録済みWebhookの削除操作 |

## 機能種別

CRUD操作（Delete） / データ削除

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 削除対象のサブスクリプションID | 正の整数 |

### 認証情報

| 項目 | 説明 |
|-----|------|
| Authorization | Bearer {access_token} |
| User ID | JWTトークンのsubクレームから取得 |

### 入力データソース

- URLパス: /api/webhooks/{id} からサブスクリプションID
- HTTPリクエストヘッダー: Authorizationヘッダーからアクセストークン
- ClaimsPrincipal: トークンからユーザーID（sub）を抽出

## 出力仕様

### 出力データ

#### 成功時（202 Accepted）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Location | Header | 削除されたリソースのURI（/api/webhooks/{id}） |

#### 失敗時（404 Not Found）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| message | string | "Subscriptions {id} not found" |

### 出力先

- HTTPレスポンス: 202 Accepted（削除成功）
- または 404 Not Found（サブスクリプションが存在しない、または他ユーザーの所有）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエストの受信
   └─ DELETE /api/webhooks/{id}

2. 認証・認可の確認
   └─ Authorizationヘッダーからトークンを検証
   └─ ClaimsPrincipalからユーザーIDを取得

3. データベースからサブスクリプション検索
   └─ WebhooksContext.Subscriptionsに対してクエリ
   └─ WHERE Id = @id AND UserId = @userId でフィルタリング
   └─ SingleOrDefaultAsync()で単一結果取得

4. 削除処理
   └─ サブスクリプションが存在する場合: context.Remove()
   └─ SaveChangesAsync()でデータベースに反映

5. レスポンス返却
   └─ 削除成功: 202 Accepted
   └─ サブスクリプション不存在: 404 Not Found
```

### フローチャート

```mermaid
flowchart TD
    A[DELETE /api/webhooks/id] --> B[認証トークン検証]
    B --> C{認証成功?}
    C -->|No| D[401 Unauthorized]
    C -->|Yes| E[ユーザーID取得]
    E --> F[DBからサブスクリプション検索]
    F --> G{サブスクリプション存在?<br>かつ所有者一致?}
    G -->|Yes| H[context.Remove実行]
    H --> I[SaveChangesAsync]
    I --> J[202 Accepted]
    G -->|No| K[404 Not Found]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-38-01 | ユーザースコープ | 各ユーザーは自身のサブスクリプションのみ削除可能 | 全削除リクエスト |
| BR-38-02 | 認証必須 | 有効なアクセストークンが必要 | 全削除リクエスト |
| BR-38-03 | 所有者チェック | IDが存在しても、所有者でない場合は404を返却 | セキュリティ対策 |
| BR-38-04 | 物理削除 | サブスクリプションは物理削除される（論理削除ではない） | 全削除処理 |
| BR-38-05 | 削除後の通知停止 | 削除後、該当Webhookへの通知は行われない | 削除完了後 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| サブスクリプション検索 | Subscriptions | SELECT | IDとユーザーIDで検索 |
| サブスクリプション削除 | Subscriptions | DELETE | 該当レコードを削除 |

### テーブル別操作詳細

#### Subscriptions

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Type, Date, DestUrl, Token, UserId | WHERE Id = @id AND UserId = @userId | 削除対象確認 |
| DELETE | - | WHERE Id = @id | 物理削除 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証トークンが無効または欠落 | 再ログインを促す |
| 404 | Not Found | サブスクリプションが存在しない、または他ユーザーの所有 | クライアントでエラー表示 |
| 500 | Internal Server Error | データベースアクセスエラー | エラーログ出力、リトライ |

### リトライ仕様

- クライアント側でリトライを実装
- サーバー側ではEntity Frameworkのリトライポリシーに従う

## トランザクション仕様

- SaveChangesAsync()内で暗黙的にトランザクション管理
- 削除失敗時は自動ロールバック

## パフォーマンス要件

- 削除処理: 500ms以内の応答
- 主キー検索 + DELETE操作による高速処理

## セキュリティ考慮事項

- Bearer Token認証による認可
- ユーザースコープによるデータ分離（他ユーザーのデータは404として扱う）
- HTTPS通信による暗号化
- ID推測攻撃対策（所有者チェックにより、IDを知っていても他ユーザーのデータは削除不可）

## 備考

- ASP.NET Core Minimal APIsを使用した実装
- 202 Acceptedは「削除リクエストを受け付けた」ことを示すステータスコード
- 物理削除のため、削除後の復元は不可

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、削除対象のデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | サブスクリプションエンティティの定義 |

**読解のコツ**: WebhookSubscriptionのIdフィールドが主キーとして使用され、削除時のキーとなる。

#### Step 2: エントリーポイントを理解する

API定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | DELETEエンドポイント定義 |

**主要処理フロー**:
- **66-82行目**: DELETE /{id:int} - 削除エンドポイント
- **71行目**: `user.GetUserId()` - ユーザーID取得
- **72行目**: `context.Subscriptions.SingleOrDefaultAsync(s => s.Id == id && s.UserId == userId)` - 所有者チェック付き検索
- **74-78行目**: 存在時の削除処理
- **76行目**: `context.Remove(subscription)` - 削除
- **77行目**: `context.SaveChangesAsync()` - DB反映
- **78行目**: `TypedResults.Accepted()` - 202返却
- **81行目**: 404 Not Found返却

#### Step 3: データアクセス層を理解する

WebhooksContextを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | DbContext定義 |

### プログラム呼び出し階層図

```
DELETE /api/webhooks/{id}
    │
    ├─ ClaimsPrincipal.GetUserId()
    │      └─ JWTトークンからsubクレーム取得
    │
    └─ WebhooksContext.Subscriptions
           │
           └─ .SingleOrDefaultAsync(s => s.Id == id && s.UserId == userId)
                  │
                  ├─ 存在する場合
                  │      │
                  │      ├─ context.Remove(subscription)
                  │      │
                  │      └─ context.SaveChangesAsync()
                  │             │
                  │             └─ TypedResults.Accepted()
                  │
                  └─ 存在しない場合
                         │
                         └─ TypedResults.NotFound("...")
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URL Path /{id}     ───▶  WebhooksContext             ───▶ 202 Accepted
                         .Subscriptions                    または
Authorization Header      .SingleOrDefaultAsync()          404 Not Found
(Bearer Token)                 │
       │                       ▼
       ▼               存在チェック
ClaimsPrincipal ──────▶ WHERE Id = @id
.GetUserId()            AND UserId = @userId
                               │
                               ▼
                        context.Remove()
                               │
                               ▼
                        SaveChangesAsync()
                        (DELETE実行)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | ソース | APIエンドポイント定義 |
| WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | ソース | エンティティモデル |
| WebhooksContext.cs | `src/Webhooks.API/Infrastructure/WebhooksContext.cs` | ソース | DbContext |
