# 機能設計書 39-Webhook送信

## 概要

本ドキュメントは、eShopアプリケーションにおけるWebhook送信機能の設計仕様を定義するものである。本機能は、Webhooks.APIサービスが提供するWebhook管理機能の中核であり、登録されたWebhookエンドポイントにイベント通知を送信する機能を提供する。

### 本機能の処理概要

本機能は、eShop内で発生したイベント（注文支払完了、注文発送、価格変更）を、登録済みのWebhookサブスクリプションに基づいて外部エンドポイントへ通知する。統合イベントハンドラーから呼び出され、複数の宛先への並列送信を実現する。

**業務上の目的・背景**：マイクロサービスアーキテクチャにおいて、外部システムとのリアルタイム連携は重要な要件である。eShopでは、注文の支払完了（OrderPaid）、注文発送（OrderShipped）、商品価格変更（CatalogItemPriceChange）などのビジネスイベントを外部システムに通知することで、倉庫管理システム、会計システム、マーケティングツールなどとの連携を実現する。本機能は、このイベント駆動型連携の基盤を提供する。

**機能の利用シーン**：統合イベントハンドラー（OrderStatusChangedToPaidIntegrationEventHandler、OrderStatusChangedToShippedIntegrationEventHandler等）からトリガーされ、該当イベントタイプを購読している全てのWebhookサブスクリプションに対して通知を送信する。

**主要な処理内容**：
1. 該当イベントタイプのサブスクリプション一覧を取得
2. WebhookDataオブジェクトの構築（イベントデータのシリアライズ）
3. 全サブスクリプションへの並列HTTP POST送信
4. 認証トークン（X-eshop-whtoken）のヘッダー付与

**関連システム・外部連携**：統合イベントバス（RabbitMQ）からトリガーされる。登録されたWebhookエンドポイント（外部システム）へHTTP POSTリクエストを送信する。

**権限による制御**：本機能は内部サービスとして動作し、直接的なエンドユーザーアクセスはない。統合イベントハンドラーからのみ呼び出される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面から直接呼び出されない（バックグラウンド処理） |

## 機能種別

データ連携 / イベント通知

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| receivers | IEnumerable<WebhookSubscription> | Yes | 送信先サブスクリプション一覧 | 空でも可 |
| data | WebhookData | Yes | 送信するWebhookデータ | null不可 |

### WebhookDataの構造

| フィールド名 | 型 | 説明 |
|-------------|-----|------|
| When | DateTime | イベント発生日時（UTC） |
| Payload | string | イベントデータ（JSONシリアライズ済み） |
| Type | string | イベントタイプ文字列 |

### 入力データソース

- 統合イベントハンドラー: イベントオブジェクト
- WebhooksRetriever: 該当タイプのサブスクリプション一覧

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTP POST | Request | 各Webhookエンドポイントへの通知 |

### HTTP POSTリクエストの構造

| 項目 | 値 |
|-----|-----|
| Method | POST |
| Content-Type | application/json |
| Body | WebhookData JSON |
| X-eshop-whtoken | 登録時のトークン（設定されている場合） |

### 出力先

- 各サブスクリプションのDestUrl（外部Webhookエンドポイント）

## 処理フロー

### 処理シーケンス

```
1. 統合イベントの受信
   └─ RabbitMQからイベントメッセージ受信
   └─ OrderStatusChangedToPaidIntegrationEventHandler等が起動

2. サブスクリプション取得
   └─ WebhooksRetriever.GetSubscriptionsOfType()呼び出し
   └─ 該当タイプの全サブスクリプション取得

3. WebhookData構築
   └─ new WebhookData(WebhookType, eventData)
   └─ イベントデータをJSONシリアライズ

4. 並列送信実行
   └─ WebhooksSender.SendAll()呼び出し
   └─ 各サブスクリプションへTask.WhenAll()で並列送信

5. 各送信処理
   └─ HttpRequestMessage構築
   └─ X-eshop-whtokenヘッダー付与
   └─ HttpClient.SendAsync()実行
```

### フローチャート

```mermaid
flowchart TD
    A[統合イベント受信] --> B[サブスクリプション取得]
    B --> C{サブスクリプションあり?}
    C -->|No| D[処理終了]
    C -->|Yes| E[WebhookData構築]
    E --> F[SendAll呼び出し]
    F --> G[並列タスク生成]
    G --> H[Task.WhenAll実行]
    H --> I[各宛先へPOST送信]
    I --> J[送信完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-39-01 | 並列送信 | 複数宛先への送信は並列で実行 | 2件以上のサブスクリプション |
| BR-39-02 | トークン付与 | サブスクリプションにトークンが設定されている場合、X-eshop-whtokenヘッダーを付与 | Token非null/空時 |
| BR-39-03 | JSONシリアライズ | イベントデータはJSON形式でシリアライズ | 全送信 |
| BR-39-04 | UTC日時 | イベント発生日時はUTC形式で送信 | 全送信 |
| BR-39-05 | Fire-and-Forget | 送信失敗時のリトライは現行実装では未対応 | 全送信 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| サブスクリプション取得 | Subscriptions | SELECT | タイプ別のサブスクリプション取得 |

### テーブル別操作詳細

#### Subscriptions

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Type, Date, DestUrl, Token, UserId | WHERE Type = @type | WebhooksRetriever経由 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | HTTP送信エラー | ネットワーク障害、タイムアウト | ログ出力、次の送信を継続 |
| - | 宛先不達 | Webhookエンドポイントがダウン | ログ出力、次の送信を継続 |
| - | タイムアウト | 宛先応答遅延 | ログ出力、次の送信を継続 |

### リトライ仕様

- 現行実装ではリトライ機能なし（Fire-and-Forget）
- 送信失敗時は他の宛先への送信に影響しない

## トランザクション仕様

- 読み取り専用操作（サブスクリプション取得）のため、トランザクション管理は不要
- 送信処理はHTTP通信のため、トランザクション対象外

## パフォーマンス要件

- 全送信完了: 宛先数 x 応答時間（並列化により最長応答時間に依存）
- HttpClientはIHttpClientFactoryで管理（接続プール）

## セキュリティ考慮事項

- X-eshop-whtokenヘッダーによる認証
- HTTPS通信による暗号化（宛先URLがHTTPSの場合）
- 送信データはイベント情報のみ（個人情報の最小化）

## 備考

- IHttpClientFactoryを使用したHttpClient管理
- 並列送信によるスループット向上
- 将来的にはリトライ機能の追加が望ましい

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、送信されるデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebhookData.cs | `src/Webhooks.API/Model/WebhookData.cs` | 送信データの構造 |
| 1-2 | WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | 送信先情報 |
| 1-3 | WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | イベントタイプ |

**読解のコツ**: WebhookDataのコンストラクタ（11-16行目）でイベントデータがJSONシリアライズされる点に注目。

#### Step 2: インターフェースを理解する

サービスインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | IWebhooksSender.cs | `src/Webhooks.API/Services/IWebhooksSender.cs` | 送信サービスインターフェース |
| 2-2 | IWebhooksRetriever.cs | `src/Webhooks.API/Services/IWebhooksRetriever.cs` | サブスクリプション取得インターフェース |

#### Step 3: 送信実装を理解する

WebhooksSenderの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebhooksSender.cs | `src/Webhooks.API/Services/WebhooksSender.cs` | 送信処理の実装 |

**主要処理フロー**:
- **3行目**: IHttpClientFactoryのインジェクション
- **5-11行目**: SendAll - 並列送信処理
- **7行目**: HttpClient生成
- **8行目**: WebhookDataのJSONシリアライズ
- **9行目**: 各サブスクリプションへのタスク生成
- **10行目**: Task.WhenAll - 並列実行
- **13-33行目**: OnSendData - 個別送信処理
- **15-20行目**: HttpRequestMessage構築
- **22-25行目**: トークンヘッダー付与
- **32行目**: HttpClient.SendAsync - 送信実行

#### Step 4: 統合イベントハンドラーを理解する

呼び出し元のイベントハンドラーを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | OrderStatusChangedToPaidIntegrationEventHandler.cs | `src/Webhooks.API/IntegrationEvents/OrderStatusChangedToPaidIntegrationEventHandler.cs` | OrderPaidイベントハンドラー |
| 4-2 | OrderStatusChangedToShippedIntegrationEventHandler.cs | `src/Webhooks.API/IntegrationEvents/OrderStatusChangedToShippedIntegrationEventHandler.cs` | OrderShippedイベントハンドラー |

**主要処理フロー（OrderStatusChangedToPaidIntegrationEventHandler）**:
- **8-17行目**: Handle - イベント処理
- **10行目**: `retriever.GetSubscriptionsOfType(WebhookType.OrderPaid)` - サブスクリプション取得
- **14行目**: `new WebhookData(WebhookType.OrderPaid, @event)` - データ構築
- **16行目**: `sender.SendAll(subscriptions, whook)` - 送信実行

### プログラム呼び出し階層図

```
統合イベントバス（RabbitMQ）
    │
    └─ IntegrationEventHandler.Handle()
           │
           ├─ IWebhooksRetriever.GetSubscriptionsOfType()
           │      └─ WebhooksContext.Subscriptions.Where(s => s.Type == type)
           │
           ├─ new WebhookData(WebhookType, eventData)
           │      └─ JsonSerializer.Serialize(data)
           │
           └─ IWebhooksSender.SendAll()
                  │
                  └─ Task.WhenAll(各サブスクリプションへの送信タスク)
                         │
                         └─ OnSendData()
                                │
                                ├─ new HttpRequestMessage(POST)
                                ├─ X-eshop-whtokenヘッダー追加
                                └─ HttpClient.SendAsync()
```

### データフロー図

```
[入力]                        [処理]                         [出力]

統合イベント            ───▶ IntegrationEventHandler  ───▶ HTTP POST Request
(RabbitMQ)                         │                       (各Webhookエンドポイント)
                                   ▼
                            WebhooksRetriever
                            .GetSubscriptionsOfType()
                                   │
                                   ▼
                            new WebhookData()
                            (JSONシリアライズ)
                                   │
                                   ▼
                            WebhooksSender.SendAll()
                                   │
                                   ▼
                            Task.WhenAll(並列送信)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebhooksSender.cs | `src/Webhooks.API/Services/WebhooksSender.cs` | ソース | Webhook送信サービス実装 |
| IWebhooksSender.cs | `src/Webhooks.API/Services/IWebhooksSender.cs` | ソース | 送信サービスインターフェース |
| WebhooksRetriever.cs | `src/Webhooks.API/Services/WebhooksRetriever.cs` | ソース | サブスクリプション取得サービス |
| IWebhooksRetriever.cs | `src/Webhooks.API/Services/IWebhooksRetriever.cs` | ソース | 取得サービスインターフェース |
| WebhookData.cs | `src/Webhooks.API/Model/WebhookData.cs` | ソース | 送信データモデル |
| WebhookSubscription.cs | `src/Webhooks.API/Model/WebhookSubscription.cs` | ソース | サブスクリプションモデル |
| WebhookType.cs | `src/Webhooks.API/Model/WebhookType.cs` | ソース | イベントタイプ列挙型 |
| OrderStatusChangedToPaidIntegrationEventHandler.cs | `src/Webhooks.API/IntegrationEvents/OrderStatusChangedToPaidIntegrationEventHandler.cs` | ソース | OrderPaidイベントハンドラー |
| OrderStatusChangedToShippedIntegrationEventHandler.cs | `src/Webhooks.API/IntegrationEvents/OrderStatusChangedToShippedIntegrationEventHandler.cs` | ソース | OrderShippedイベントハンドラー |
