# 機能設計書 41-カタログ画面

## 概要

本ドキュメントは、eShop WebAppにおけるカタログ画面（商品一覧表示画面）の機能設計を記載する。本画面はアプリケーションのメイン画面として、商品の一覧表示・フィルタリング・ページネーション機能を提供する。

### 本機能の処理概要

本機能は、eShopアプリケーションにおいてユーザーが商品を閲覧・検索するための中心的なUI機能である。ユーザーはブランドや商品タイプによるフィルタリングを行いながら、ページネーション付きで商品を閲覧できる。

**業務上の目的・背景**：ECサイトにおいて、ユーザーが効率的に商品を探し出せるようにすることは販売促進の観点から極めて重要である。本画面は、大量の商品カタログから目的の商品を素早く見つけ出すためのフィルタリング機能とページネーション機能を提供し、ユーザー体験を向上させる。また、ストリームレンダリングを採用することで、初期表示の高速化を実現している。

**機能の利用シーン**：ユーザーがeShopにアクセスした際の最初の画面として表示される。また、商品購入後にショッピングを継続する際にもこの画面に戻ってくる。ブランドや商品タイプで絞り込みながら、目的の商品を探す場面で使用される。

**主要な処理内容**：
1. 初期化時にCatalogServiceを使用して商品一覧を取得する（ページネーション付き）
2. CatalogSearchコンポーネントでブランド一覧・タイプ一覧を取得しフィルターUIを表示する
3. フィルター選択時にクエリパラメータを変更してページをリロードする
4. CatalogListItemコンポーネントで個々の商品を表示し、クリックで商品詳細画面へ遷移する
5. ページネーションUIで別ページへの移動リンクを提供する

**関連システム・外部連携**：Catalog.API（商品一覧取得、ブランド一覧取得、タイプ一覧取得のREST API）

**権限による制御**：本画面は認証不要で誰でもアクセス可能。商品閲覧自体にはログインは不要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | カタログ画面 | 主画面 | 商品一覧の表示・フィルタリング・ページネーション |
| 2 | 商品詳細画面 | 遷移先画面 | CatalogListItemクリックで商品詳細画面へ遷移 |

## 機能種別

画面表示機能（CRUD操作のRead）/ データ表示 / フィルタリング / ページネーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| page | int? | No | 表示ページ番号（1始まり） | 未指定時は1として扱う |
| brand | int? | No | フィルタリング対象のブランドID | 存在するブランドIDであること |
| type | int? | No | フィルタリング対象のタイプID | 存在するタイプIDであること |

### 入力データソース

- クエリパラメータ（URL）: page, brand, type
- Catalog.API: 商品データ、ブランドデータ、タイプデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| catalogResult | CatalogResult | 商品一覧情報（PageIndex, PageSize, Count, Data） |
| catalogResult.Data | List\<CatalogItem\> | 商品リスト |
| catalogBrands | IEnumerable\<CatalogBrand\> | ブランド一覧 |
| catalogItemTypes | IEnumerable\<CatalogItemType\> | タイプ一覧 |

### 出力先

- 画面表示: 商品カード一覧、フィルターUI、ページネーションリンク

## 処理フロー

### 処理シーケンス

```
1. 画面初期化（OnInitializedAsync）
   └─ ユーザーがルートURL "/"にアクセス

2. 商品一覧取得
   └─ CatalogService.GetCatalogItems(pageIndex, pageSize, brandId, typeId)を呼び出し
   └─ Catalog.API の GET /api/catalog/items にリクエスト

3. フィルター情報取得（CatalogSearchコンポーネント）
   └─ CatalogService.GetBrands() と CatalogService.GetTypes() を並列実行
   └─ Catalog.API の GET /api/catalog/catalogBrands, catalogTypes にリクエスト

4. 画面表示
   └─ フィルターUI表示（ブランド、タイプのタグ）
   └─ 商品カード一覧表示（CatalogListItem）
   └─ ページネーションリンク表示

5. ユーザーインタラクション
   └─ フィルター選択時: クエリパラメータを変更して画面遷移
   └─ 商品クリック時: 商品詳細画面へ遷移
   └─ ページリンククリック時: 該当ページへ遷移
```

### フローチャート

```mermaid
flowchart TD
    A[画面アクセス] --> B[OnInitializedAsync]
    B --> C[CatalogService.GetCatalogItems呼び出し]
    C --> D{データ取得成功?}
    D -->|Yes| E[商品一覧表示]
    D -->|No| F[Loading表示のまま]
    E --> G[CatalogSearchコンポーネント初期化]
    G --> H[GetBrands/GetTypes並列実行]
    H --> I[フィルターUI表示]
    I --> J[ユーザー操作待ち]
    J -->|フィルター選択| K[URLクエリパラメータ変更]
    K --> A
    J -->|商品クリック| L[商品詳細画面へ遷移]
    J -->|ページリンククリック| M[ページパラメータ変更]
    M --> A
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-41-01 | ページサイズ | 1ページあたり9件の商品を表示する | 常に適用 |
| BR-41-02 | ページ番号のデフォルト | ページ番号未指定時は1ページ目を表示 | pageパラメータ未指定時 |
| BR-41-03 | フィルターの初期状態 | フィルター未選択時は全商品を表示 | brand/typeパラメータ未指定時 |
| BR-41-04 | 複合フィルター | ブランドとタイプは同時にフィルター可能 | 両パラメータが指定された時 |

### 計算ロジック

- ページ数計算: `Math.Ceiling(1.0 * result.Count / PageSize)`
- APIへのページインデックス変換: `Page.GetValueOrDefault(1) - 1`（UIは1始まり、APIは0始まり）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品一覧取得 | Catalog | SELECT | 条件に合致する商品を取得（API経由） |
| ブランド一覧取得 | CatalogBrand | SELECT | 全ブランドを取得（API経由） |
| タイプ一覧取得 | CatalogType | SELECT | 全タイプを取得（API経由） |

※ 本画面はCatalog.APIを経由してデータを取得するため、直接のDB操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 通信エラー | Catalog.APIとの通信失敗 | Loading表示が継続される |
| - | データ未取得 | catalogResultがnull | "Loading..."メッセージを表示 |

### リトライ仕様

自動リトライは行わない。画面リロードでリトライ可能。

## トランザクション仕様

読み取り専用のため、トランザクション管理は不要。

## パフォーマンス要件

- ストリームレンダリング（@attribute [StreamRendering]）を使用して初期表示を高速化
- ブランド一覧とタイプ一覧の取得は並列実行（Task.WhenAll）
- ページサイズは9件に制限してレスポンスサイズを抑制

## セキュリティ考慮事項

- 認証不要の公開画面
- 入力パラメータはクエリパラメータのみで、SQLインジェクション等のリスクはAPI側で対処
- XSS対策はBlazorのデフォルトエスケープ機能で対応

## 備考

- 商品画像URLはIProductImageUrlProviderを通じて取得
- 本画面はアプリケーションのルート（"/"）にマッピングされている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、画面で使用されるデータ型を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | 商品データ、ブランド、タイプの型定義を理解する |

**読解のコツ**: C# recordを使用した不変データ型として定義されている。CatalogResult、CatalogBrand、CatalogItemTypeも同ファイルで定義。

**主要処理フロー**:
- **3-12行目**: CatalogItem record定義（Id, Name, Description, Price, PictureUrl, CatalogBrandId, CatalogBrand, CatalogTypeId, CatalogType）
- **14行目**: CatalogResult record定義（PageIndex, PageSize, Count, Data）
- **15行目**: CatalogBrand record定義（Id, Brand）
- **16行目**: CatalogItemType record定義（Id, Type）

#### Step 2: エントリーポイントを理解する

処理の起点となるRazorコンポーネント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Catalog.razor | `src/WebApp/Components/Pages/Catalog/Catalog.razor` | ページルーティング、クエリパラメータ、初期化処理 |

**主要処理フロー**:
- **1行目**: `@page "/"` - ルートURLへのマッピング
- **4行目**: `@attribute [StreamRendering]` - ストリームレンダリング有効化
- **38行目**: `const int PageSize = 9` - 1ページあたりの表示件数
- **40-47行目**: クエリパラメータの受け取り（Page, BrandId, ItemTypeId）
- **54-61行目**: OnInitializedAsyncでCatalogService.GetCatalogItemsを呼び出し
- **51-52行目**: GetVisiblePageIndexesでページネーション計算

#### Step 3: サービス層を理解する

APIとの通信を担当するサービス。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | HttpClientを使用したAPI呼び出し |

**主要処理フロー**:
- **9行目**: remoteServiceBaseUrlが"api/catalog/"
- **17-22行目**: GetCatalogItemsメソッド - ページネーション付き商品一覧取得
- **38-43行目**: GetBrandsメソッド - ブランド一覧取得
- **45-50行目**: GetTypesメソッド - タイプ一覧取得
- **52-66行目**: GetAllCatalogItemsUri - クエリパラメータ組み立て

#### Step 4: 子コンポーネントを理解する

画面を構成するサブコンポーネント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CatalogSearch.razor | `src/WebAppComponents/Catalog/CatalogSearch.razor` | フィルターUIの実装 |
| 4-2 | CatalogListItem.razor | `src/WebAppComponents/Catalog/CatalogListItem.razor` | 商品カードの表示 |

**CatalogSearch.razor主要処理フロー**:
- **55-62行目**: OnInitializedAsyncでブランド・タイプを並列取得
- **64-68行目**: BrandUriメソッド - ブランドフィルター用URLの生成
- **70-74行目**: TypeUriメソッド - タイプフィルター用URLの生成

**CatalogListItem.razor主要処理フロー**:
- **5行目**: ItemHelper.Urlで商品詳細画面へのリンクを生成
- **7行目**: ProductImages.GetProductImageUrlで商品画像URLを取得

### プログラム呼び出し階層図

```
Catalog.razor (エントリーポイント)
    │
    ├─ CatalogService.GetCatalogItems()
    │      └─ HttpClient.GetFromJsonAsync<CatalogResult>()
    │             └─ Catalog.API GET /api/catalog/items
    │
    ├─ CatalogSearch.razor (子コンポーネント)
    │      ├─ CatalogService.GetBrands()
    │      │      └─ HttpClient.GetFromJsonAsync<CatalogBrand[]>()
    │      │             └─ Catalog.API GET /api/catalog/catalogBrands
    │      │
    │      └─ CatalogService.GetTypes()
    │             └─ HttpClient.GetFromJsonAsync<CatalogItemType[]>()
    │                    └─ Catalog.API GET /api/catalog/catalogTypes
    │
    └─ CatalogListItem.razor (子コンポーネント) × N件
           └─ ProductImages.GetProductImageUrl()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

URLクエリパラメータ    ┌─────────────────┐
  page              ──▶│                 │
  brand             ──▶│  Catalog.razor  │──▶ 商品一覧HTML
  type              ──▶│                 │
                       └────────┬────────┘
                                │
                                ▼
                       ┌─────────────────┐
                       │ CatalogService  │
                       └────────┬────────┘
                                │
                                ▼
                       ┌─────────────────┐
Catalog.API          ──│   HttpClient    │──▶ JSON Response
/api/catalog/items     └─────────────────┘
/api/catalog/catalogBrands
/api/catalog/catalogTypes
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Catalog.razor | `src/WebApp/Components/Pages/Catalog/Catalog.razor` | ソース | メインページコンポーネント |
| CatalogSearch.razor | `src/WebAppComponents/Catalog/CatalogSearch.razor` | ソース | フィルターUIコンポーネント |
| CatalogListItem.razor | `src/WebAppComponents/Catalog/CatalogListItem.razor` | ソース | 商品カードコンポーネント |
| CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | ソース | API通信サービス |
| CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | ソース | データ型定義 |
| ICatalogService.cs | `src/WebAppComponents/Services/ICatalogService.cs` | ソース | サービスインターフェース |
| IProductImageUrlProvider.cs | `src/WebAppComponents/Item/IProductImageUrlProvider.cs` | ソース | 画像URL取得インターフェース |
