# 機能設計書 42-商品詳細画面

## 概要

本ドキュメントは、eShop WebAppにおける商品詳細画面の機能設計を記載する。本画面は指定された商品の詳細情報を表示し、買い物かごへの追加機能を提供する。

### 本機能の処理概要

本機能は、ユーザーが商品カタログから選択した商品の詳細情報（名前、説明、価格、ブランド、画像）を表示し、ログインユーザーに対して買い物かごへの商品追加機能を提供する画面である。未ログインユーザーにはログインを促すUIを表示する。

**業務上の目的・背景**：ECサイトにおいて、ユーザーが購入判断を行うためには商品の詳細情報が不可欠である。本画面は商品の画像、説明、価格、ブランド情報を一覧表示し、ユーザーの購入意思決定をサポートする。また、シームレスな購入体験を提供するため、画面から直接買い物かごへの追加が可能となっている。

**機能の利用シーン**：カタログ画面から商品をクリックした際に遷移する画面として使用される。ユーザーは詳細情報を確認し、購入を決定した場合は「Add to shopping bag」ボタンをクリックして買い物かごに追加する。未ログインユーザーはログイン画面へリダイレクトされる。

**主要な処理内容**：
1. URLパラメータから商品IDを取得し、CatalogServiceで商品詳細を取得
2. 商品画像をIProductImageUrlProvider経由で表示
3. ログイン状態を判定し、適切なボタンを表示
4. 「Add to shopping bag」クリック時にBasketState.AddAsyncでかごに追加
5. 未ログインユーザーの場合はログイン画面へリダイレクト
6. 商品が見つからない場合は404エラー画面を表示

**関連システム・外部連携**：Catalog.API（商品詳細取得）、Basket.API（gRPC経由で買い物かご操作）

**権限による制御**：商品の閲覧は認証不要。買い物かごへの追加はログイン必須（未ログイン時はログイン画面へリダイレクト）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品詳細画面 | 主画面 | 商品詳細の表示、かごへの追加 |
| 1 | カタログ画面 | 遷移元画面 | 商品一覧からの遷移 |
| 6 | ログイン画面 | 遷移先画面 | 未ログイン時のリダイレクト先 |
| 3 | カート画面 | 遷移先画面 | 「shopping bag」リンクからの遷移 |

## 機能種別

画面表示機能（CRUD操作のRead）/ データ更新（買い物かご追加）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| itemId | int | Yes | 表示する商品のID（URLパス） | 正の整数 |

### 入力データソース

- URLパス: /item/{itemId} から商品IDを取得
- Catalog.API: 商品詳細データ
- HttpContext: ユーザーの認証状態

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| item | CatalogItem? | 商品詳細情報 |
| item.Name | string | 商品名 |
| item.Description | string | 商品説明 |
| item.Price | decimal | 価格 |
| item.CatalogBrand | CatalogBrand | ブランド情報 |
| numInCart | int | 現在かごに入っている同商品の数量 |
| isLoggedIn | bool | ログイン状態 |
| notFound | bool | 商品が見つからなかった場合true |

### 出力先

- 画面表示: 商品画像、名前、説明、価格、ブランド、カート追加ボタン

## 処理フロー

### 処理シーケンス

```
1. 画面初期化（OnInitializedAsync）
   └─ URLから商品ID（ItemId）を取得

2. 認証状態確認
   └─ HttpContext.User.Identity?.IsAuthenticated で判定
   └─ isLoggedInフラグを設定

3. 商品詳細取得
   └─ CatalogService.GetCatalogItem(ItemId)を呼び出し
   └─ Catalog.API の GET /api/catalog/items/{id} にリクエスト

4. カート内数量取得（UpdateNumInCartAsync）
   └─ BasketState.GetBasketItemsAsync()でかご内容取得
   └─ 同一商品IDの数量をnumInCartに設定

5. 画面表示
   └─ 商品詳細情報を表示
   └─ ログイン状態に応じたボタンを表示
   └─ かご内数量があれば表示

6. ユーザーインタラクション
   └─ ログイン済みの場合: AddToCartAsyncでかごに追加
   └─ 未ログインの場合: ログイン画面へリダイレクト
```

### フローチャート

```mermaid
flowchart TD
    A[画面アクセス /item/{itemId}] --> B[OnInitializedAsync]
    B --> C{認証状態確認}
    C --> D[CatalogService.GetCatalogItem呼び出し]
    D --> E{商品取得成功?}
    E -->|Yes| F[UpdateNumInCartAsync]
    E -->|No - 404| G[notFound = true]
    G --> H[404エラー表示]
    F --> I[画面表示]
    I --> J{ユーザー操作}
    J -->|Add to cart クリック| K{isLoggedIn?}
    K -->|Yes| L[BasketState.AddAsync]
    L --> M[UpdateNumInCartAsync]
    M --> I
    K -->|No| N[ログイン画面へ遷移]
    J -->|shopping bag クリック| O[カート画面へ遷移]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-42-01 | カート追加認証 | 買い物かごへの追加はログインユーザーのみ可能 | Add to cartボタンクリック時 |
| BR-42-02 | 404ハンドリング | 存在しない商品IDの場合は404エラーを返す | HttpRequestException StatusCode.NotFound |
| BR-42-03 | 数量増加 | 既にかごにある商品を追加すると数量が1増加する | AddAsync呼び出し時 |
| BR-42-04 | 価格表示形式 | 価格は小数点以下2桁で表示 | 常に適用 |

### 計算ロジック

- かご内数量取得: `items.FirstOrDefault(row => row.ProductId == ItemId)?.Quantity ?? 0`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品詳細取得 | Catalog | SELECT | 指定IDの商品を取得（API経由） |
| かご取得 | Basket (Redis) | GET | ユーザーのかご内容を取得（gRPC経由） |
| かご更新 | Basket (Redis) | SET | 商品追加後のかご内容を保存（gRPC経由） |

※ 本画面はAPIを経由してデータを取得・更新するため、直接のDB操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | NotFound | 指定IDの商品が存在しない | "Not found"画面を表示 |
| - | 通信エラー | Catalog.APIとの通信失敗 | 画面表示されない状態が継続 |
| - | gRPCエラー | Basket.APIとの通信失敗 | かご操作が失敗 |

### リトライ仕様

自動リトライは行わない。画面リロードでリトライ可能。

## トランザクション仕様

- 商品詳細取得: 読み取り専用
- かご追加: BasketState.AddAsync内で一連の処理を実行（読み取り→更新）

## パフォーマンス要件

- 商品詳細は1件のみ取得のため軽量
- かご内容の取得はgRPCで高速通信

## セキュリティ考慮事項

- 商品閲覧は認証不要（公開情報）
- かご操作は認証済みユーザーのみ可能
- 未認証でのかご追加試行時は安全にログイン画面へリダイレクト
- AntiforgeryTokenでCSRF対策

## 備考

- `data-enhance="false"` 属性で商品リンクの拡張ナビゲーションを無効化
- フォームの `data-enhance="@isLoggedIn"` でログイン状態に応じた拡張機能を制御

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

画面で使用されるデータ型とサービスインターフェースを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | 商品データ型の構造 |
| 1-2 | BasketItem.cs | `src/WebApp/Services/BasketItem.cs` | かごアイテムの型定義 |

**読解のコツ**: CatalogItemはrecord型で不変データとして定義。BasketItemはクラスでQuantityプロパティが変更可能。

#### Step 2: エントリーポイントを理解する

メインのページコンポーネント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ItemPage.razor | `src/WebApp/Components/Pages/Item/ItemPage.razor` | ルーティング、認証確認、フォーム処理 |

**主要処理フロー**:
- **1行目**: `@page "/item/{itemId:int}"` - 商品IDをパスパラメータで受け取る
- **64-67行目**: 状態変数（item, numInCart, isLoggedIn, notFound）
- **75-88行目**: OnInitializedAsync - 初期化処理、404ハンドリング含む
- **79行目**: `isLoggedIn = HttpContext?.User.Identity?.IsAuthenticated == true`
- **90-103行目**: AddToCartAsync - かご追加処理、未認証時はログイン画面へ
- **105-109行目**: UpdateNumInCartAsync - かご内の同商品数量取得

#### Step 3: サービス層を理解する

バックエンドサービスとの連携。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | 商品詳細取得API呼び出し |
| 3-2 | BasketState.cs | `src/WebApp/Services/BasketState.cs` | かご状態管理 |

**CatalogService.cs主要処理フロー**:
- **11-15行目**: GetCatalogItemメソッド - 単一商品取得

**BasketState.cs主要処理フロー**:
- **21-24行目**: GetBasketItemsAsync - かご内容取得
- **33-56行目**: AddAsync - 商品追加（既存なら数量+1、新規なら追加）
- **53行目**: `_cachedBasket = null` - キャッシュ無効化
- **54行目**: `await basketService.UpdateBasketAsync(items)` - API呼び出し

#### Step 4: 認証・リダイレクト処理を理解する

未認証ユーザーの処理フロー。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | LogIn.razor | `src/WebApp/Components/Pages/User/LogIn.razor` | ログイン画面へのリダイレクト |

**主要処理フロー**:
- **19-20行目**: Urlメソッド - リダイレクトURL生成（returnUrlパラメータ付き）

### プログラム呼び出し階層図

```
ItemPage.razor (エントリーポイント)
    │
    ├─ OnInitializedAsync
    │      ├─ HttpContext.User.Identity?.IsAuthenticated (認証確認)
    │      │
    │      ├─ CatalogService.GetCatalogItem(ItemId)
    │      │      └─ HttpClient.GetFromJsonAsync<CatalogItem>()
    │      │             └─ Catalog.API GET /api/catalog/items/{id}
    │      │
    │      └─ UpdateNumInCartAsync
    │             └─ BasketState.GetBasketItemsAsync()
    │                    └─ BasketService.GetBasketAsync() (gRPC)
    │
    └─ AddToCartAsync (フォーム送信時)
           ├─ 未認証の場合
           │      └─ Nav.NavigateTo(Pages.User.LogIn.Url(Nav))
           │
           └─ 認証済みの場合
                  └─ BasketState.AddAsync(item)
                         ├─ FetchBasketItemsAsync (現在のかご取得)
                         └─ BasketService.UpdateBasketAsync (gRPC更新)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

URLパス /item/{id}  ───▶ ItemPage.razor
                              │
                              ▼
                    ┌─────────────────┐
Catalog.API       ──│ CatalogService  │──▶ CatalogItem (商品詳細)
GET /api/catalog/   │ .GetCatalogItem │
items/{id}          └─────────────────┘
                              │
                              ▼
                    ┌─────────────────┐
Basket.API        ──│   BasketState   │──▶ numInCart (かご内数量)
(gRPC)              │ GetBasketItems  │
                    └─────────────────┘
                              │
                              ▼
                    ┌─────────────────┐
フォーム送信      ──│   BasketState   │──▶ かご更新成功/失敗
Add to cart         │    .AddAsync    │
                    └─────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ItemPage.razor | `src/WebApp/Components/Pages/Item/ItemPage.razor` | ソース | メインページコンポーネント |
| CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | ソース | 商品取得API通信 |
| BasketState.cs | `src/WebApp/Services/BasketState.cs` | ソース | かご状態管理 |
| BasketService.cs | `src/WebApp/Services/BasketService.cs` | ソース | かごgRPCクライアント |
| CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | ソース | 商品データ型 |
| LogIn.razor | `src/WebApp/Components/Pages/User/LogIn.razor` | ソース | ログイン画面・URL生成 |
| IProductImageUrlProvider.cs | `src/WebAppComponents/Item/IProductImageUrlProvider.cs` | ソース | 画像URL取得インターフェース |
