# 機能設計書 43-買い物かご画面

## 概要

本ドキュメントは、eShop WebAppにおける買い物かご画面（CartPage）の機能設計を記載する。本画面は認証済みユーザーの買い物かごの内容を表示し、数量変更・削除・合計金額確認・チェックアウトへの遷移機能を提供する。

### 本機能の処理概要

本機能は、ユーザーが買い物かごに入れた商品の一覧表示、数量変更、削除、合計金額の計算を行い、チェックアウト画面への遷移を提供する画面である。ストリームレンダリングを使用して初期表示を高速化している。

**業務上の目的・背景**：ECサイトにおいて、買い物かごは購入前の最終確認と調整を行う重要な機能である。ユーザーは購入予定の商品と金額を確認し、数量の変更や不要な商品の削除を行える。この画面を経由してチェックアウト（注文確定）に進むため、購入フローの要となる。

**機能の利用シーン**：商品詳細画面で商品をかごに追加した後、またはヘッダーのカートアイコンからアクセスする。ユーザーは購入前にかごの内容を確認・調整し、問題なければチェックアウトに進む。

**主要な処理内容**：
1. 初期化時にBasketState.GetBasketItemsAsyncでかご内容を取得
2. 各商品の名前、価格、数量、小計を表示
3. 数量変更フォームから数量を更新（Basket.SetQuantityAsync）
4. 数量を0にすることで商品を削除
5. 合計金額と合計数量を計算・表示
6. 「Check out」ボタンでチェックアウト画面へ遷移

**関連システム・外部連携**：Basket.API（gRPC経由でかご操作）、Catalog.API（商品情報取得）

**権限による制御**：本画面は認証必須（@attribute [Authorize]）。未認証ユーザーはログイン画面にリダイレクトされる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | カート画面 | 主画面 | かご内容表示・数量変更・合計金額計算 |
| 2 | 商品詳細画面 | 遷移元画面 | 商品追加後にかご確認 |
| 4 | チェックアウト画面 | 遷移先画面 | 「Check out」ボタンで遷移 |
| 1 | カタログ画面 | 遷移先画面 | 「Continue shopping」リンクで遷移 |

## 機能種別

画面表示機能（CRUD操作のRead）/ データ更新（数量変更・削除）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| UpdateQuantityId | int? | No | 更新対象の商品ID | フォーム送信時に必須 |
| UpdateQuantityValue | int? | No | 更新後の数量 | 0以上の整数 |

### 入力データソース

- フォーム送信: UpdateQuantityId, UpdateQuantityValue
- BasketState: かご内容（BasketItem[]）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| basketItems | IReadOnlyCollection\<BasketItem\> | かご内の商品一覧 |
| TotalPrice | decimal? | 合計金額 |
| TotalQuantity | decimal? | 合計数量 |

### 出力先

- 画面表示: 商品一覧、数量入力フォーム、合計金額、チェックアウトボタン

## 処理フロー

### 処理シーケンス

```
1. 画面初期化（OnInitializedAsync）
   └─ BasketState.GetBasketItemsAsyncでかご内容を取得

2. 画面表示
   └─ かごが空の場合: 空メッセージを表示
   └─ かごに商品がある場合: 商品一覧、合計金額を表示

3. 数量変更（UpdateQuantityAsync）
   └─ フォームから UpdateQuantityId, UpdateQuantityValue を取得
   └─ Basket.SetQuantityAsync(id, quantity)で更新
   └─ かご内容を再取得して画面を更新

4. チェックアウトへ遷移
   └─ 「Check out」リンクで /checkout に遷移
```

### フローチャート

```mermaid
flowchart TD
    A[/cart アクセス] --> B{認証済み?}
    B -->|No| C[ログイン画面へリダイレクト]
    B -->|Yes| D[OnInitializedAsync]
    D --> E[BasketState.GetBasketItemsAsync]
    E --> F{かご空?}
    F -->|Yes| G[空メッセージ表示]
    F -->|No| H[商品一覧表示]
    H --> I[合計金額計算・表示]
    I --> J{ユーザー操作}
    J -->|Update クリック| K[UpdateQuantityAsync]
    K --> L[Basket.SetQuantityAsync]
    L --> M[GetBasketItemsAsync]
    M --> H
    J -->|Check out クリック| N[/checkout へ遷移]
    J -->|Continue shopping| O[/ へ遷移]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-43-01 | 認証必須 | 本画面へのアクセスは認証済みユーザーのみ | 常に適用 |
| BR-43-02 | 数量0で削除 | 数量を0に設定すると商品がかごから削除される | SetQuantityAsync時 |
| BR-43-03 | 小計計算 | 小計 = 単価 × 数量 | 各商品行 |
| BR-43-04 | 合計金額 | 合計 = 全商品の（単価 × 数量）の合計 | 常に適用 |
| BR-43-05 | 価格表示形式 | 価格は小数点以下2桁で表示 | 常に適用 |

### 計算ロジック

- 合計金額: `basketItems.Sum(i => i.Quantity * i.UnitPrice)`
- 合計数量: `basketItems.Sum(i => i.Quantity)`
- 商品小計: `item.UnitPrice * quantity`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| かご取得 | Basket (Redis) | GET | ユーザーのかご内容を取得（gRPC経由） |
| かご更新 | Basket (Redis) | SET | 数量変更後のかご内容を保存（gRPC経由） |

※ BasketStateが内部でCatalogServiceを呼び出して商品詳細を取得している。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証アクセス | ログイン画面へリダイレクト |
| - | gRPCエラー | Basket.APIとの通信失敗 | かご取得・更新が失敗 |

### リトライ仕様

自動リトライは行わない。画面リロードでリトライ可能。

## トランザクション仕様

- かご取得: 読み取り専用
- かご更新: BasketState.SetQuantityAsync内で一連の処理を実行（読み取り→更新）

## パフォーマンス要件

- ストリームレンダリング（@attribute [StreamRendering]）を使用して初期表示を高速化
- かご内容のキャッシュ機能あり

## セキュリティ考慮事項

- 認証必須（@attribute [Authorize]）
- AntiforgeryTokenでCSRF対策
- ユーザーは自分のかごのみ操作可能（買い手IDで識別）

## 備考

- `data-enhance` 属性でBlazorのフォーム拡張を有効化
- CurrentOrPendingQuantityメソッドで更新中の数量を優先表示（楽観的UI）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

かごアイテムの型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BasketItem.cs | `src/WebApp/Services/BasketItem.cs` | かごアイテムの型定義 |

**読解のコツ**: BasketItemはクラスでQuantityプロパティが変更可能。ProductId, ProductName, UnitPrice, Quantityを持つ。

#### Step 2: エントリーポイントを理解する

メインのページコンポーネント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CartPage.razor | `src/WebApp/Components/Pages/Cart/CartPage.razor` | ルーティング、認証、フォーム処理 |

**主要処理フロー**:
- **1行目**: `@page "/cart"` - カート画面のルーティング
- **5行目**: `@attribute [StreamRendering]` - ストリームレンダリング有効化
- **6行目**: `@attribute [Authorize]` - 認証必須
- **86行目**: basketItems - かご内商品のコレクション
- **94-97行目**: OnInitializedAsync - 初期化でかご内容取得
- **99-100行目**: TotalPrice, TotalQuantity - 計算プロパティ
- **104-107行目**: CurrentOrPendingQuantity - 更新中の数量を優先表示
- **109-115行目**: UpdateQuantityAsync - 数量更新処理

#### Step 3: サービス層を理解する

かご状態管理サービス。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | BasketState.cs | `src/WebApp/Services/BasketState.cs` | かご状態管理・API連携 |

**主要処理フロー**:
- **21-24行目**: GetBasketItemsAsync - 認証済みユーザーのかご取得
- **58-76行目**: SetQuantityAsync - 数量変更処理
- **63-65行目**: 数量 > 0 の場合は数量を更新
- **66-69行目**: 数量 <= 0 の場合はアイテムを削除
- **117-148行目**: FetchBasketItemsAsync - かご内容をCatalog情報と結合して取得

### プログラム呼び出し階層図

```
CartPage.razor (エントリーポイント)
    │
    ├─ OnInitializedAsync
    │      └─ BasketState.GetBasketItemsAsync()
    │             └─ FetchBasketItemsAsync()
    │                    ├─ BasketService.GetBasketAsync() (gRPC)
    │                    └─ CatalogService.GetCatalogItems(ids)
    │                           └─ Catalog.API GET /api/catalog/items/by
    │
    ├─ TotalPrice (計算プロパティ)
    │      └─ basketItems.Sum(i => i.Quantity * i.UnitPrice)
    │
    └─ UpdateQuantityAsync (フォーム送信時)
           └─ BasketState.SetQuantityAsync(id, quantity)
                  ├─ FetchBasketItemsAsync (現在のかご取得)
                  ├─ 数量変更または削除
                  └─ BasketService.UpdateBasketAsync (gRPC更新)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

Basket.API (gRPC)   ───▶ BasketState
                              │
                              ▼
                    ┌─────────────────┐
                    │ FetchBasketItems│──▶ basketItems
                    └────────┬────────┘        │
                             │                 ▼
Catalog.API         ◀───────┘         ┌─────────────────┐
GET /api/catalog/                      │   CartPage.razor │──▶ 画面表示
items/by                               └────────┬────────┘
                                                │
                                                ▼
フォーム送信        ───▶ UpdateQuantityAsync
UpdateQuantityId              │
UpdateQuantityValue           ▼
                    ┌─────────────────┐
                    │  SetQuantityAsync│──▶ Basket.API更新
                    └─────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CartPage.razor | `src/WebApp/Components/Pages/Cart/CartPage.razor` | ソース | メインページコンポーネント |
| BasketState.cs | `src/WebApp/Services/BasketState.cs` | ソース | かご状態管理 |
| BasketService.cs | `src/WebApp/Services/BasketService.cs` | ソース | かごgRPCクライアント |
| BasketItem.cs | `src/WebApp/Services/BasketItem.cs` | ソース | かごアイテム型定義 |
| CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | ソース | 商品情報取得 |
| IProductImageUrlProvider.cs | `src/WebAppComponents/Item/IProductImageUrlProvider.cs` | ソース | 画像URL取得インターフェース |
