# 機能設計書 46-AIチャットボット

## 概要

本ドキュメントは、eShop WebAppにおけるAIチャットボット機能の設計を記載する。本機能はAzure OpenAI（またはOpenAI互換サービス）を使用した対話型AIアシスタントで、商品検索・推薦やカート操作の支援を提供する。

### 本機能の処理概要

本機能は、Microsoft.Extensions.AIを使用したAIチャットボット機能である。ユーザーは自然言語で商品に関する質問や要望を伝えることができ、AIが商品検索、カート追加、カート内容確認などの操作を支援する。Function Callingを活用して、AIが適切なツール（関数）を呼び出し、実際のサービス連携を行う。

**業務上の目的・背景**：ECサイトにおいて、ユーザーが目的の商品を見つけることは購買行動の重要なステップである。AIチャットボットを導入することで、ユーザーは自然言語で「スキーに適した服を探して」のような曖昧な要望を伝えることができ、AIがセマンティック検索を活用して関連商品を提案する。これにより、従来のキーワード検索では見つけにくい商品も発見しやすくなる。

**機能の利用シーン**：画面右下のチャットボタンをクリックするとチャットパネルが開く。ユーザーは商品に関する質問（例：「防水のジャケットはありますか？」）やカート操作の依頼（例：「その商品をカートに追加して」）を自然言語で入力する。AIは質問に回答し、必要に応じて商品検索やカート操作を実行する。

**主要な処理内容**：
1. チャット画面の初期化でIChatClientの存在を確認
2. ユーザーメッセージを受信し、ChatStateに追加
3. ChatClientを通じてAIに問い合わせ（Function Calling対応）
4. AIが必要に応じてツール関数を呼び出し（SearchCatalog, AddToCart等）
5. AIの応答をチャット画面に表示

**関連システム・外部連携**：Azure OpenAI / OpenAI API、Catalog.API（セマンティック検索）、Basket.API（カート操作）

**権限による制御**：チャット自体は認証不要で利用可能。ただし、カートへの追加などの操作は認証が必要（未認証時はエラーメッセージを返す）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | チャットボット | 主画面 | AIチャット機能 |
| 1 | カタログ画面 | 連携画面 | 商品検索結果の表示 |
| 3 | カート画面 | 連携画面 | カート追加後の確認 |

## 機能種別

AI対話機能 / 検索支援 / カート操作支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| messageToSend | string | Yes | ユーザーの入力メッセージ | 空白でないこと |

### 入力データソース

- ユーザー入力: テキストエリアからのメッセージ
- 認証情報: ClaimsPrincipal（ユーザー情報取得用）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Messages | IList\<ChatMessage\> | チャット履歴（User/Assistant） |
| response.Text | string | AIからの応答テキスト |

### 出力先

- 画面表示: チャットメッセージ一覧

## 処理フロー

### 処理シーケンス

```
1. 画面初期化（OnInitializedAsync）
   └─ IChatClientの存在確認
   └─ ChatStateの初期化（システムプロンプト設定）

2. ユーザーメッセージ送信（SendMessageAsync）
   └─ メッセージをMessagesに追加
   └─ ChatClient.GetResponseAsyncを呼び出し

3. AI応答生成
   └─ AIがFunction Callingを判断
   └─ 必要に応じてツール関数を実行
      - GetUserInfo: ユーザー情報取得
      - SearchCatalog: 商品検索
      - AddToCart: カート追加
      - GetCartContents: カート内容取得

4. 応答表示
   └─ AI応答をMessagesに追加
   └─ 画面更新（StateHasChanged）
```

### フローチャート

```mermaid
flowchart TD
    A[チャットボット開く] --> B{IChatClient利用可能?}
    B -->|No| C[設定不足メッセージ表示]
    B -->|Yes| D[ChatState初期化]
    D --> E[初期挨拶表示]
    E --> F[メッセージ入力待ち]
    F --> G[SendMessageAsync]
    G --> H[Messages.Add UserMessage]
    H --> I[ChatClient.GetResponseAsync]
    I --> J{Function Call必要?}
    J -->|Yes| K[ツール関数実行]
    K --> L{どの関数?}
    L -->|SearchCatalog| M[CatalogService.GetCatalogItemsWithSemanticRelevance]
    L -->|AddToCart| N[BasketState.AddAsync]
    L -->|GetCartContents| O[BasketState.GetBasketItemsAsync]
    L -->|GetUserInfo| P[ClaimsPrincipal参照]
    M --> Q[結果をAIに返す]
    N --> Q
    O --> Q
    P --> Q
    Q --> I
    J -->|No| R[応答テキスト取得]
    R --> S[Messages.Add AssistantMessage]
    S --> T[画面更新]
    T --> F
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | 話題制限 | AIはAdventureWorksに関する話題のみ回答 | 常に適用 |
| BR-46-02 | 簡潔な応答 | 必要な場合のみ詳細な回答を提供 | 常に適用 |
| BR-46-03 | カート追加認証 | カート追加は認証済みユーザーのみ可能 | AddToCart実行時 |
| BR-46-04 | セマンティック検索 | 商品検索はベクトル埋め込みを使用 | SearchCatalog実行時 |
| BR-46-05 | 検索結果上限 | 1回の検索で最大8件まで取得 | SearchCatalog実行時 |

### 計算ロジック

特になし（AIモデルが応答生成を担当）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品検索 | Catalog | SELECT | セマンティック検索で関連商品を取得（API経由） |
| カート追加 | Basket (Redis) | SET | 商品をカートに追加（gRPC経由） |
| カート参照 | Basket (Redis) | GET | カート内容を取得（gRPC経由） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 設定不足 | IChatClientが未設定 | 設定不足メッセージを表示 |
| - | AI応答エラー | ChatClient呼び出し失敗 | "unexpected error"メッセージ表示 |
| Unauthenticated | 認証エラー | 未認証でカート追加試行 | エラーメッセージ返却 |
| - | 通信エラー | Catalog.API通信失敗 | エラーメッセージをAIに返却 |

### リトライ仕様

自動リトライは行わない。ユーザーが再度メッセージを送信してリトライ。

## トランザクション仕様

各ツール関数呼び出しは独立した操作。トランザクション管理は各サービス側で実施。

## パフォーマンス要件

- InteractiveServerRenderMode（prerender: false）で即時対話可能
- セマンティック検索は8件に制限してレスポンス時間を短縮
- 非同期処理でUIブロッキングを防止

## セキュリティ考慮事項

- カート操作は認証済みユーザーのみ可能
- ユーザー情報（住所、メール等）はClaimsPrincipalから取得
- AIへの入力はサニタイズ不要（AIサービス側で対処）
- システムプロンプトで話題を制限し、不適切な応答を防止

## 備考

- IChatClientが設定されていない場合（useOpenAI = false）はチャットボット機能が無効化される
- MessageProcessor.AllowImagesでテキスト内の画像URLを処理
- JSモジュールでスクロール制御・Enterキー送信を実装

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

チャットメッセージとAI関連の型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ChatMessage | Microsoft.Extensions.AI | ChatRole（User/Assistant/System）とTextを持つ |

**読解のコツ**: Microsoft.Extensions.AIはAI統合の標準化ライブラリ。ChatMessageはロールとテキストを持つシンプルな構造。

#### Step 2: UIコンポーネントを理解する

チャットボットのRazorコンポーネント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Chatbot.razor | `src/WebApp/Components/Chatbot/Chatbot.razor` | チャットUI、メッセージ送信処理 |

**主要処理フロー**:
- **1行目**: `@rendermode @(new InteractiveServerRenderMode(prerender: false))` - プリレンダリング無効
- **54-66行目**: OnInitializedAsync - IChatClient取得とChatState初期化
- **68-79行目**: SendMessageAsync - メッセージ送信処理
- **81-91行目**: OnAfterRenderAsync - JSモジュール初期化

#### Step 3: チャット状態管理を理解する

AIとの対話ロジックとFunction Calling。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ChatState.cs | `src/WebApp/Components/Chatbot/ChatState.cs` | AIツール定義、メッセージ処理 |

**主要処理フロー**:
- **19-65行目**: コンストラクタ - ツール関数の登録、システムプロンプト設定
- **41-47行目**: ChatOptions.Tools - 4つのツール関数を登録
- **52-60行目**: システムプロンプト - AdventureWorks専用AIとして振る舞う指示
- **69-93行目**: AddUserMessageAsync - ユーザーメッセージ追加とAI応答取得
- **96-115行目**: GetUserInfo - ユーザー情報取得ツール
- **117-134行目**: SearchCatalog - 商品検索ツール（セマンティック検索）
- **136-153行目**: AddToCart - カート追加ツール
- **155-167行目**: GetCartContents - カート内容取得ツール

#### Step 4: ツール関数の詳細を理解する

各AIツールの実装詳細。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | GetCatalogItemsWithSemanticRelevance |
| 4-2 | BasketState.cs | `src/WebApp/Services/BasketState.cs` | AddAsync, GetBasketItemsAsync |

**CatalogService.cs主要処理フロー**:
- **31-36行目**: GetCatalogItemsWithSemanticRelevance - セマンティック検索API呼び出し

### プログラム呼び出し階層図

```
Chatbot.razor (エントリーポイント)
    │
    ├─ OnInitializedAsync
    │      ├─ ServiceProvider.GetService<IChatClient>()
    │      └─ new ChatState(...)
    │             └─ AIFunctionFactory.Create (ツール登録)
    │
    └─ SendMessageAsync (メッセージ送信時)
           └─ ChatState.AddUserMessageAsync(message)
                  │
                  ├─ Messages.Add(UserMessage)
                  │
                  └─ ChatClient.GetResponseAsync(Messages, ChatOptions)
                         │
                         ├─ [Function Call] GetUserInfo
                         │      └─ ClaimsPrincipal参照
                         │
                         ├─ [Function Call] SearchCatalog
                         │      └─ CatalogService.GetCatalogItemsWithSemanticRelevance()
                         │             └─ Catalog.API GET /api/catalog/items/withsemanticrelevance
                         │
                         ├─ [Function Call] AddToCart
                         │      ├─ CatalogService.GetCatalogItem(itemId)
                         │      └─ BasketState.AddAsync(item)
                         │
                         └─ [Function Call] GetCartContents
                                └─ BasketState.GetBasketItemsAsync()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

ユーザーメッセージ  ───▶ ChatState.AddUserMessageAsync
                              │
                              ▼
                    ┌─────────────────┐
                    │   IChatClient   │
                    │ (Azure OpenAI)  │
                    └────────┬────────┘
                             │
          ┌──────────────────┼──────────────────┐
          ▼                  ▼                  ▼
    ┌───────────┐     ┌───────────┐     ┌───────────┐
    │GetUserInfo│     │SearchCatalog│   │AddToCart  │
    └─────┬─────┘     └─────┬─────┘     └─────┬─────┘
          │                 │                 │
          ▼                 ▼                 ▼
    ClaimsPrincipal  Catalog.API       BasketState
                     /withsemanticrelevance
          │                 │                 │
          └────────┬────────┴────────┬────────┘
                   ▼                 ▼
             JSON結果をAIに返却 → AI応答生成
                                     │
                                     ▼
                              チャット画面表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Chatbot.razor | `src/WebApp/Components/Chatbot/Chatbot.razor` | ソース | チャットUIコンポーネント |
| ChatState.cs | `src/WebApp/Components/Chatbot/ChatState.cs` | ソース | AI対話ロジック・ツール定義 |
| Chatbot.razor.js | `src/WebApp/Components/Chatbot/Chatbot.razor.js` | ソース | JSスクロール・キー処理 |
| ShowChatbotButton.razor | `src/WebApp/Components/Chatbot/ShowChatbotButton.razor` | ソース | チャット表示ボタン |
| MessageProcessor.cs | `src/WebApp/Components/Chatbot/MessageProcessor.cs` | ソース | メッセージ処理ユーティリティ |
| CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | ソース | 商品検索API |
| BasketState.cs | `src/WebApp/Services/BasketState.cs` | ソース | カート状態管理 |
