# 機能設計書 47-ログイン画面

## 概要

本ドキュメントは、eShop WebAppにおけるログイン画面の機能設計を記載する。本画面はユーザー認証のエントリーポイントとして機能し、OpenID Connect経由でIdentity.APIへリダイレクトする。

### 本機能の処理概要

本機能は、eShop WebAppにおけるユーザー認証を処理するリダイレクト画面である。[Authorize]属性が付与されているため、未認証ユーザーがアクセスすると自動的にOpenID Connect認証フローが開始される。認証完了後はreturnUrlパラメータで指定された元のページにリダイレクトされる。

**業務上の目的・背景**：ECサイトにおいて、ユーザー認証は購入処理やパーソナライズされた機能（注文履歴、買い物かご等）を提供するために必要不可欠である。本機能はOpenID Connect認証を使用し、Identity.APIにユーザー認証を委譲することで、セキュアな認証フローを実現している。

**機能の利用シーン**：
- 商品詳細画面で未ログインユーザーが「Add to cart」をクリックした場合
- 買い物かご画面やチェックアウト画面に未認証でアクセスした場合
- ユーザーメニューの「Sign in」リンクをクリックした場合

**主要な処理内容**：
1. [Authorize]属性によりOpenID Connect認証チャレンジが自動開始
2. ユーザーはIdentity.APIのログイン画面にリダイレクト
3. 認証成功後、WebAppのログイン画面に戻る
4. returnUrlがあれば元のページへ、なければルートへリダイレクト

**関連システム・外部連携**：Identity.API（OpenID Connect認証プロバイダー）

**権限による制御**：本画面自体が認証を要求する（[Authorize]属性）。これにより認証フローがトリガーされる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 6 | ログイン画面 | 主画面 | 認証リダイレクト処理 |
| 2 | 商品詳細画面 | 遷移元画面 | 未認証でカート追加時 |
| 3 | カート画面 | 遷移元画面 | 未認証アクセス時 |
| 4 | チェックアウト画面 | 遷移元画面 | 未認証アクセス時 |

## 機能種別

認証処理 / リダイレクト処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| returnUrl | string? | No | 認証後のリダイレクト先URL | 相対URLのみ許可 |

### 入力データソース

- クエリパラメータ: returnUrl
- Identity.API: 認証結果（Cookie）

## 出力仕様

### 出力データ

本画面は直接的なデータ出力を行わず、リダイレクト処理のみを行う。

| 項目名 | 型 | 説明 |
|--------|-----|------|
| リダイレクト | - | returnUrlまたはルート（"/"）への遷移 |

### 出力先

- 画面遷移: 認証後のリダイレクト先

## 処理フロー

### 処理シーケンス

```
1. 画面アクセス
   └─ /user/login?returnUrl={url} でアクセス

2. 認証チェック（[Authorize]属性）
   └─ 未認証の場合: OpenID Connect認証チャレンジ
   └─ Identity.APIログイン画面へリダイレクト

3. Identity.APIでの認証
   └─ ユーザーがログイン情報を入力
   └─ 認証成功: コールバックURLでWebAppに戻る

4. OnInitializedAsync
   └─ returnUrlを取得（未指定時は"/"）
   └─ 絶対URLの場合は"/"にフォールバック
   └─ Nav.NavigateToでリダイレクト
```

### フローチャート

```mermaid
flowchart TD
    A[/user/login?returnUrl=xxx アクセス] --> B{認証済み?}
    B -->|No| C[OpenID Connect認証チャレンジ]
    C --> D[Identity.APIログイン画面]
    D --> E[ユーザーがログイン]
    E --> F{認証成功?}
    F -->|No| G[エラー表示/再試行]
    G --> D
    F -->|Yes| H[WebApp /user/login にコールバック]
    B -->|Yes| I[OnInitializedAsync]
    H --> I
    I --> J[returnUrl取得]
    J --> K{returnUrl指定あり?}
    K -->|No| L[returnUrl = "/"]
    K -->|Yes| M{絶対URL?}
    M -->|Yes| L
    M -->|No| N[returnUrlをそのまま使用]
    L --> O[Nav.NavigateTo]
    N --> O
    O --> P[リダイレクト先画面]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-47-01 | 認証必須 | 本画面へのアクセスには認証が必要 | 常に適用 |
| BR-47-02 | 相対URLのみ | returnUrlは相対URLのみ許可、絶対URLは"/"に変換 | リダイレクト時 |
| BR-47-03 | デフォルトリダイレクト | returnUrl未指定時はルート（"/"）へリダイレクト | returnUrl未指定時 |
| BR-47-04 | OpenID Connect | 認証はOpenID Connectプロトコルを使用 | 認証時 |

### 計算ロジック

- URL生成: `$"user/login?returnUrl={Uri.EscapeDataString(nav.ToBaseRelativePath(nav.Uri))}"`
- 絶対URL判定: `new Uri(returnUrl, UriKind.RelativeOrAbsolute).IsAbsoluteUri`

## データベース操作仕様

### 操作別データベース影響一覧

本画面は直接的なデータベース操作を行わない。認証処理はIdentity.API側で実行。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | - |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 認証失敗 | Identity.APIでのログイン失敗 | Identity.API側でエラー表示 |
| - | 通信エラー | Identity.APIとの通信失敗 | ASP.NET Core認証ミドルウェアが処理 |

### リトライ仕様

認証失敗時はIdentity.API側で再試行を促す。

## トランザクション仕様

トランザクション管理は不要（リダイレクト処理のみ）。

## パフォーマンス要件

- リダイレクト処理のため即座に応答
- 認証処理の遅延はIdentity.APIに依存

## セキュリティ考慮事項

- OpenID Connectによるセキュアな認証フロー
- returnUrlは相対URLのみ許可（オープンリダイレクト攻撃対策）
- Cookie認証でセッション管理
- HTTPS必須（認証トークンの保護）

## 備考

- 本画面は実際にはUIを表示せず、リダイレクト処理のみを行う
- Url()静的メソッドで他のコンポーネントからログインURLを生成可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

ログイン画面のRazorコンポーネント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | LogIn.razor | `src/WebApp/Components/Pages/User/LogIn.razor` | 認証リダイレクトロジック |

**主要処理フロー**:
- **1行目**: `@page "/user/login"` - ルーティング
- **3行目**: `@attribute [Authorize]` - 認証必須（これが認証フローをトリガー）
- **5-6行目**: ReturnUrlパラメータ（クエリパラメータから取得）
- **11-17行目**: OnInitializedAsync - リダイレクト処理
- **13行目**: `ReturnUrl ?? "/"` - デフォルト値設定
- **14行目**: `new Uri(returnUrl, UriKind.RelativeOrAbsolute)` - URI解析
- **15行目**: `url.IsAbsoluteUri ? "/" : returnUrl` - 絶対URLの場合は"/"に変換
- **19-20行目**: Urlメソッド - 他コンポーネントからログインURLを生成

#### Step 2: 認証設定を理解する

ASP.NET Core認証ミドルウェアの設定。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | HostingExtensions.cs | `src/WebApp/Extensions/HostingExtensions.cs` | 認証サービス設定 |

**読解のコツ**: AddOpenIdConnect()でIdentity.APIとの連携を設定。認証チャレンジ時に自動的にIdentity.APIにリダイレクトされる。

#### Step 3: ログインURL生成の使用箇所を理解する

他のコンポーネントからのログイン画面呼び出し。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ItemPage.razor | `src/WebApp/Components/Pages/Item/ItemPage.razor` | 未認証時のログインリダイレクト |
| 3-2 | UserMenu.razor | `src/WebApp/Components/Layout/UserMenu.razor` | ログインリンク |

**ItemPage.razor主要処理フロー**:
- **94行目**: `Nav.NavigateTo(Pages.User.LogIn.Url(Nav))` - ログインURLへ遷移

**UserMenu.razor主要処理フロー**:
- **23行目**: `href="@Pages.User.LogIn.Url(Nav)"` - ログインリンク生成

### プログラム呼び出し階層図

```
未認証状態でのアクセス
    │
    ├─ [Authorize]属性による認証チェック
    │      └─ ASP.NET Core認証ミドルウェア
    │             └─ OpenID Connect認証チャレンジ
    │                    └─ Identity.APIへリダイレクト
    │
    └─ 認証成功後のコールバック
           └─ LogIn.razor OnInitializedAsync
                  ├─ returnUrl取得
                  ├─ URL検証（絶対URL→"/"）
                  └─ Nav.NavigateTo(returnUrl)

他コンポーネントからのログイン呼び出し
    │
    └─ Pages.User.LogIn.Url(Nav)
           └─ "user/login?returnUrl={currentUrl}"
```

### データフロー図

```
[入力]                    [処理]                           [出力]

未認証アクセス      ───▶ [Authorize]属性
                              │
                              ▼
                    ┌─────────────────┐
                    │ 認証ミドルウェア │
                    └────────┬────────┘
                             │
                             ▼
                    ┌─────────────────┐
Identity.API      ◀─│ OIDC Challenge  │
ログイン画面         └─────────────────┘
     │
     │ 認証成功
     ▼
returnUrl       ───▶ OnInitializedAsync
                              │
                              ▼
                    ┌─────────────────┐
                    │ URL検証・変換   │
                    └────────┬────────┘
                             │
                             ▼
                    Nav.NavigateTo ──▶ リダイレクト先
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| LogIn.razor | `src/WebApp/Components/Pages/User/LogIn.razor` | ソース | ログインリダイレクト処理 |
| UserMenu.razor | `src/WebApp/Components/Layout/UserMenu.razor` | ソース | ログイン/ログアウトUI |
| ItemPage.razor | `src/WebApp/Components/Pages/Item/ItemPage.razor` | ソース | ログインリダイレクト呼び出し元 |
| HostingExtensions.cs | `src/WebApp/Extensions/HostingExtensions.cs` | ソース | 認証設定 |
