# 機能設計書 5-商品名検索

## 概要

本ドキュメントは、eShopアプリケーションにおける商品名検索機能の設計仕様を定義する。商品名の前方一致検索により、該当する商品をページネーション付きで取得するREST APIエンドポイントを提供する。

### 本機能の処理概要

**業務上の目的・背景**：ECサイトにおいて、ユーザーが商品名から直接商品を検索できる機能は、目的の商品を素早く見つけるために重要である。前方一致検索により、商品名の一部を入力するだけで関連商品を絞り込むことができる。

**機能の利用シーン**：カタログ画面での商品名検索、検索ボックスからのクイック検索、AIチャットボットでの商品名ベース検索など、商品名をキーワードとして検索する場面で利用される。

**主要な処理内容**：
1. パスパラメータから検索する商品名を取得
2. 前方一致（StartsWith）で該当商品をフィルタリング
3. ページネーション付きで商品一覧を返却

**関連システム・外部連携**：PostgreSQLデータベースからEntity Framework Coreを通じてデータを取得する。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | カタログ画面 | 参照画面 | 商品名フィルタリングで使用 |
| 8 | チャットボット | 参照画面 | 商品名ベースの検索で使用 |

## 機能種別

CRUD操作（Read）/ 検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | string | Yes | 検索する商品名 | 最小長1文字（minlength(1)） |
| PageSize | int | No | 1ページあたりの取得件数 | デフォルト値: 10 |
| PageIndex | int | No | 取得するページのインデックス | デフォルト値: 0 |

### 入力データソース

- URLパスパラメータ + クエリパラメータ: `/api/catalog/items/by/{name}?pageSize={pageSize}&pageIndex={pageIndex}`

## 出力仕様

### 出力データ

**正常時（200 OK）: PaginatedItems&lt;CatalogItem&gt;**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PageIndex | int | 現在のページインデックス |
| PageSize | int | 1ページあたりの件数 |
| Count | long | 総件数（検索条件適用後） |
| Data | IEnumerable&lt;CatalogItem&gt; | 商品データの配列 |

### 出力先

- HTTP Response Body（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ パスパラメータからnameを取得
   └─ クエリパラメータからPaginationRequestを取得

2. GetAllItemsメソッド呼び出し
   └─ nameパラメータを渡して内部委譲

3. クエリ構築
   └─ Where(c => c.Name.StartsWith(name))

4. データ取得
   ├─ LongCountAsyncで総件数取得
   └─ OrderBy/Skip/Take/ToListAsyncでページデータ取得

5. レスポンス生成
   └─ PaginatedItems<CatalogItem>を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[name取得]
    B --> C[PaginationRequest取得]
    C --> D[GetAllItems呼び出し]
    D --> E[StartsWith条件構築]
    E --> F[総件数カウント]
    F --> G[名前順ソート]
    G --> H[Skip/Take適用]
    H --> I[データ取得]
    I --> J[PaginatedItems生成]
    J --> K[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 前方一致検索 | 商品名の前方一致で検索する（StartsWith） | 常時 |
| BR-002 | 最小文字数 | 検索文字列は最低1文字必要 | 常時 |
| BR-003 | ソート順 | 商品名の昇順でソートする | 常時 |
| BR-004 | 大文字小文字 | データベースの照合順序に依存 | 常時 |

### 計算ロジック

- 内部的にGetAllItemsメソッドを呼び出し、nameパラメータのみを指定（type, brandはnull）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品名検索 | Catalog | SELECT | 名前前方一致で商品を取得 |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE Name LIKE '{name}%' | StartsWith → LIKE演算子に変換 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | nameが空文字または未指定 | ルーティングでminlength(1)制約 |
| 400 | BadRequest | パラメータ形式が不正 | ProblemDetails形式でエラー詳細を返却 |
| 500 | InternalServerError | データベース接続エラー等 | サーバーエラーとして処理 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。

## パフォーマンス要件

- Nameカラムにインデックスが設定されており、前方一致検索の性能を確保
- ページネーションにより大量検索結果でも効率的に処理

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- SQLインジェクション対策: Entity Framework CoreのLINQクエリによりパラメータ化

## 備考

- V1 APIのみで提供（V2ではGetAllItemsのnameパラメータを使用）
- 内部実装はGetAllItemsメソッドへの委譲で実現

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 商品エンティティ、Nameプロパティ |
| 1-2 | PaginatedItems.cs | `src/Catalog.API/Model/PaginatedItems.cs` | ページネーション結果構造 |
| 1-3 | PaginationRequest.cs | `src/Catalog.API/Model/PaginationRequest.cs` | ページネーションリクエスト |

**読解のコツ**: Nameプロパティは[Required]属性付きで必須。50文字の最大長制限あり（EntityConfiguration）。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemsByNameメソッドとGetAllItemsへの委譲 |

**主要処理フロー**:
1. **41-45行目**: `/items/by/{name:minlength(1)}` エンドポイント定義（V1のみ）
2. **194-200行目**: `GetItemsByName` メソッド実装
3. **199行目**: GetAllItemsへの委譲（nameのみ指定、type/brandはnull）
4. **138行目**: StartsWith条件の構築（GetAllItems内）

#### Step 3: 内部実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetAllItemsメソッドの詳細 |

**主要処理フロー**:
- **136-139行目**: name指定時のStartsWith条件追加
- **149-150行目**: LongCountAsyncで総件数取得
- **152-156行目**: OrderBy/Skip/Take/ToListAsyncでページデータ取得

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items/by/{name})
    │
    └─ CatalogApi.GetItemsByName()
           │
           └─ CatalogApi.GetAllItems(name, type=null, brand=null)
                  │
                  └─ CatalogServices.Context (CatalogContext)
                         └─ DbSet<CatalogItem>.CatalogItems
                                ├─ Where(c => c.Name.StartsWith(name))
                                ├─ LongCountAsync()
                                ├─ OrderBy(c => c.Name)
                                ├─ Skip(pageSize * pageIndex)
                                ├─ Take(pageSize)
                                └─ ToListAsync()
                                       │
                                       └─ PaginatedItems<CatalogItem>
```

### データフロー図

```
[入力]                       [処理]                              [出力]

HTTPリクエスト               CatalogApi.GetItemsByName()         HTTPレスポンス
  │                              │                                  │
  ├─ name ─────────────────────▶│                                  │
  ├─ PageSize ─────────────────▶│   GetAllItems()                  │
  └─ PageIndex ────────────────▶│      │                           │
                                 │      ▼                           │
                                 │   PostgreSQL                     │
                                 │   WHERE Name LIKE 'name%'        │
                                 │      │                           │
                                 │      ▼                           │
                                 └─▶ PaginatedItems ───────────────▶ JSON
                                      ├─ PageIndex
                                      ├─ PageSize
                                      ├─ Count
                                      └─ Data[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| PaginatedItems.cs | `src/Catalog.API/Model/PaginatedItems.cs` | ソース | ページネーション結果モデル |
| PaginationRequest.cs | `src/Catalog.API/Model/PaginationRequest.cs` | ソース | ページネーションリクエストモデル |
| CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | ソース | Nameインデックス設定 |
