# 機能設計書 53-統合イベント発行

## 概要

本ドキュメントは、eShopシステムにおける「統合イベント発行」機能の設計仕様を定義する。この機能は、マイクロサービス間でイベントを発行するための共通インターフェースを提供する。

### 本機能の処理概要

本機能は、eShopのマイクロサービスアーキテクチャにおいて、サービス間の非同期通信を実現するためのイベント発行機能である。IEventBusインターフェースを通じて、任意のIntegrationEventを他のマイクロサービスへ発行する。

**業務上の目的・背景**：マイクロサービスアーキテクチャでは、各サービスが独立して動作するため、サービス間でデータの整合性を保つ必要がある。本機能は、イベント駆動アーキテクチャを採用し、サービス間の疎結合な連携を実現する。例えば、商品価格が変更された際に他サービスへ通知したり、注文ステータスが変更された際に関連サービスへ通知する。

**機能の利用シーン**：
- 商品登録・更新時にカタログサービスから他サービスへ通知
- 注文ステータス変更時に注文サービスから支払い・配送サービスへ通知
- 在庫確認完了時に在庫サービスから注文サービスへ通知

**主要な処理内容**：
1. IntegrationEventインスタンスを生成
2. IEventBus.PublishAsyncメソッドを呼び出し
3. 実装クラス（RabbitMQEventBus等）がメッセージブローカーへイベントを送信
4. OpenTelemetryによるトレーシング情報の伝搬

**関連システム・外部連携**：RabbitMQメッセージブローカーを使用してイベントを配信する。

**権限による制御**：サービス間の内部通信のため、特定の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし（バックエンド処理） |

## 機能種別

データ連携 / イベント発行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| event | IntegrationEvent | Yes | 発行するイベントオブジェクト | null不可 |

### 入力データソース

各マイクロサービスの業務処理から呼び出される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | Task | 非同期タスク（成功時に完了） |

### 出力先

RabbitMQメッセージブローカーのExchange（eshop_event_bus）へメッセージを発行。

## 処理フロー

### 処理シーケンス

```
1. PublishAsync呼び出し
   └─ IntegrationEventを受け取る

2. ルーティングキー生成
   └─ イベント型名をルーティングキーとして使用

3. チャネル作成
   └─ RabbitMQ接続からチャネルを作成

4. Exchange宣言
   └─ "eshop_event_bus"をdirect型で宣言

5. メッセージシリアライズ
   └─ JSONでシリアライズしてUTF8バイト配列に変換

6. OpenTelemetryコンテキスト注入
   └─ トレーシング情報をメッセージヘッダーに追加

7. メッセージ発行
   └─ BasicPublishAsyncでメッセージを送信
   └─ DeliveryMode.Persistentで永続化
```

### フローチャート

```mermaid
flowchart TD
    A[PublishAsync呼び出し] --> B[ルーティングキー生成]
    B --> C[チャネル作成]
    C --> D[Exchange宣言]
    D --> E[メッセージシリアライズ]
    E --> F[OpenTelemetryコンテキスト注入]
    F --> G{リトライパイプライン}
    G --> H[BasicPublishAsync]
    H --> I{成功?}
    I -->|Yes| J[完了]
    I -->|No| K{リトライ回数内?}
    K -->|Yes| G
    K -->|No| L[例外スロー]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-53-1 | 永続化 | メッセージはDeliveryMode.Persistentで永続化される | 常時 |
| BR-53-2 | リトライ | 指数バックオフでリトライ（デフォルト10回） | 接続エラー時 |
| BR-53-3 | ルーティング | イベント型名がルーティングキーとして使用される | 常時 |
| BR-53-4 | トレーシング | OpenTelemetryコンテキストがメッセージヘッダーに伝搬 | 常時 |

### 計算ロジック

リトライ間隔の計算: `TimeSpan.FromSeconds(Math.Pow(2, attemptNumber))`

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを直接操作しない。メッセージブローカーへのメッセージ送信のみ。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | BrokerUnreachableException | RabbitMQ接続不可 | リトライ後、例外をスロー |
| - | SocketException | ネットワークエラー | リトライ後、例外をスロー |
| - | InvalidOperationException | RabbitMQ接続未確立 | 例外をスロー |

### リトライ仕様

- リトライ対象: BrokerUnreachableException, SocketException
- リトライ回数: デフォルト10回（EventBusOptions.RetryCountで設定可能）
- リトライ間隔: 指数バックオフ（2^attempt秒）

## トランザクション仕様

本機能自体はトランザクション管理を行わない。呼び出し元でトランザクションと連携する場合は、IntegrationEventLogサービスを使用する。

## パフォーマンス要件

- メッセージ発行: 100ms以内
- スループット: 1000メッセージ/秒以上

## セキュリティ考慮事項

- RabbitMQへの接続は認証情報を使用
- メッセージ内容は暗号化されない（内部ネットワーク前提）
- TLS接続はオプションで設定可能

## 備考

- Pollyライブラリを使用したレジリエンスパイプラインを実装
- OpenTelemetry対応によりトレーシングが可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、統合イベントの基底クラスを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IntegrationEvent.cs | `src/EventBus/Events/IntegrationEvent.cs` | イベントの基底クラス（Id, CreationDate） |

**読解のコツ**: recordとして定義。コンストラクタで新しいGuidとUTC日時を自動設定。

**主要処理フロー**:
- **5-9行目**: コンストラクタでId（Guid）とCreationDate（DateTime.UtcNow）を初期化
- **11-15行目**: JsonInclude属性でシリアライズ対象を指定

#### Step 2: インターフェースを理解する

イベントバスのインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | IEventBus.cs | `src/EventBus/Abstractions/IEventBus.cs` | PublishAsyncメソッドの定義 |

**読解のコツ**: シンプルなインターフェース。PublishAsync(IntegrationEvent @event)のみを定義。

#### Step 3: 実装クラスを理解する

RabbitMQを使用した実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RabbitMQEventBus.cs | `src/EventBusRabbitMQ/RabbitMQEventBus.cs` | PublishAsyncメソッドの実装 |

**主要処理フロー**:
- **20行目**: Exchange名は"eshop_event_bus"固定
- **22行目**: ResiliencePipelineでリトライポリシーを構築
- **31-111行目**: PublishAsyncメソッドの実装
- **33行目**: イベント型名をルーティングキーとして使用
- **40行目**: チャネルを作成
- **47-49行目**: Exchangeをdirect型で宣言
- **51行目**: SerializeMessageでJSONシリアライズ
- **75-78行目**: DeliveryMode.Persistentを設定
- **86行目**: OpenTelemetryコンテキストを注入
- **97-102行目**: BasicPublishAsyncでメッセージ発行
- **302-320行目**: CreateResiliencePipelineでリトライポリシーを構築

#### Step 4: 設定オプションを理解する

イベントバスの設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | EventBusOptions.cs | `src/EventBusRabbitMQ/EventBusOptions.cs` | SubscriptionClientName, RetryCountの設定 |

**主要処理フロー**:
- **5行目**: SubscriptionClientNameでキュー名を設定
- **6行目**: RetryCountでリトライ回数を設定（デフォルト10）

### プログラム呼び出し階層図

```
呼び出し元サービス（Catalog.API, Ordering.API等）
    │
    └─ IEventBus.PublishAsync(IntegrationEvent)
           │
           └─ RabbitMQEventBus.PublishAsync
                  │
                  ├─ CreateChannelAsync
                  │
                  ├─ ExchangeDeclareAsync
                  │
                  ├─ SerializeMessage
                  │      └─ JsonSerializer.SerializeToUtf8Bytes
                  │
                  ├─ Propagator.Inject（OpenTelemetryコンテキスト）
                  │
                  └─ BasicPublishAsync
                         │
                         └─ RabbitMQ Exchange
```

### データフロー図

```
[入力]                       [処理]                           [出力]

IntegrationEvent ───▶ RabbitMQEventBus.PublishAsync ───▶ RabbitMQ Exchange
                           │
                           ├─ JSONシリアライズ
                           ├─ OpenTelemetryコンテキスト注入
                           └─ リトライポリシー適用
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| IEventBus.cs | `src/EventBus/Abstractions/IEventBus.cs` | ソース | イベントバスインターフェース |
| IntegrationEvent.cs | `src/EventBus/Events/IntegrationEvent.cs` | ソース | 統合イベント基底クラス |
| RabbitMQEventBus.cs | `src/EventBusRabbitMQ/RabbitMQEventBus.cs` | ソース | RabbitMQ実装 |
| EventBusOptions.cs | `src/EventBusRabbitMQ/EventBusOptions.cs` | ソース | イベントバス設定 |
| RabbitMQTelemetry.cs | `src/EventBusRabbitMQ/RabbitMQTelemetry.cs` | ソース | OpenTelemetry連携 |
| EventBusSubscriptionInfo.cs | `src/EventBus/Abstractions/EventBusSubscriptionInfo.cs` | ソース | サブスクリプション情報管理 |
