# 機能設計書 55-RabbitMQ連携

## 概要

本ドキュメントは、eShopシステムにおける「RabbitMQ連携」機能の設計仕様を定義する。この機能は、RabbitMQを使用したメッセージブローカー連携機能を提供する。

### 本機能の処理概要

本機能は、eShopのマイクロサービスアーキテクチャにおいて、RabbitMQをメッセージブローカーとして使用し、サービス間の非同期メッセージングを実現する機能である。IEventBusインターフェースの具体的な実装を提供する。

**業務上の目的・背景**：マイクロサービス間の疎結合な通信を実現するため、メッセージブローカーが必要である。RabbitMQは高可用性、信頼性、柔軟なルーティング機能を提供し、eShopの要件に適している。本機能は、RabbitMQとの接続管理、メッセージの送受信、OpenTelemetryによる分散トレーシングを統合的に提供する。

**機能の利用シーン**：
- 各マイクロサービス起動時にRabbitMQ接続を確立
- イベント発行時にRabbitMQ Exchangeへメッセージを送信
- イベント購読時にRabbitMQ Queueからメッセージを受信

**主要な処理内容**：
1. AddRabbitMqEventBusでDI登録とRabbitMQ接続設定
2. RabbitMQEventBusのシングルトン登録
3. IHostedServiceとしてアプリ起動時にコンシューマー開始
4. OpenTelemetryによる分散トレーシング対応

**関連システム・外部連携**：RabbitMQメッセージブローカーと接続する。

**権限による制御**：RabbitMQ接続には認証情報（ユーザー名・パスワード）が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし（インフラ機能） |

## 機能種別

インフラストラクチャ / メッセージブローカー連携

## 入力仕様

### 入力パラメータ

#### DI登録時

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| connectionName | string | Yes | Aspire接続名 | null/空不可 |

#### 設定オプション

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| SubscriptionClientName | string | Yes | キュー名として使用 | - |
| RetryCount | int | No | リトライ回数（デフォルト10） | 0以上 |

### 入力データソース

appsettings.jsonまたは環境変数から設定を読み込む。

```json
{
  "EventBus": {
    "SubscriptionClientName": "ordering-api",
    "RetryCount": 10
  }
}
```

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| IEventBus | インターフェース | メッセージ発行機能を提供 |
| IHostedService | インターフェース | バックグラウンドでメッセージ受信 |

### 出力先

RabbitMQ Exchange（eshop_event_bus）およびQueue。

## 処理フロー

### 処理シーケンス

```
1. AddRabbitMqEventBus呼び出し（DI設定）
   └─ AddRabbitMQClientでAspire接続を追加
   └─ OpenTelemetryトレーシングを設定
   └─ EventBusOptionsを設定セクションからバインド
   └─ RabbitMQTelemetryをシングルトン登録
   └─ RabbitMQEventBusをIEventBusとして登録
   └─ IHostedServiceとして同インスタンスを登録

2. アプリケーション起動時
   └─ IHostedService.StartAsyncが呼び出される
   └─ RabbitMQ接続を取得
   └─ コンシューマーチャネルを作成
   └─ Exchange宣言（eshop_event_bus, direct型）
   └─ Queue宣言（durable: true）
   └─ 登録済みイベント型でQueueをExchangeにバインド
   └─ AsyncEventingBasicConsumerでメッセージ受信開始

3. メッセージ発行時
   └─ IEventBus.PublishAsyncを呼び出し
   └─ 新しいチャネルを作成
   └─ Exchange宣言
   └─ メッセージをJSONシリアライズ
   └─ OpenTelemetryコンテキストを注入
   └─ BasicPublishAsyncで送信
```

### フローチャート

```mermaid
flowchart TD
    A[AddRabbitMqEventBus] --> B[AddRabbitMQClient]
    B --> C[OpenTelemetry設定]
    C --> D[EventBusOptions設定]
    D --> E[RabbitMQEventBus登録]
    E --> F[IHostedService登録]

    G[アプリ起動] --> H[StartAsync]
    H --> I[RabbitMQ接続取得]
    I --> J[Exchange宣言]
    J --> K[Queue宣言]
    K --> L[QueueBind]
    L --> M[Consumer開始]

    N[PublishAsync] --> O[チャネル作成]
    O --> P[Exchange宣言]
    P --> Q[シリアライズ]
    Q --> R[OpenTelemetry注入]
    R --> S[BasicPublish]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-55-1 | Exchange型 | direct型Exchangeを使用 | 常時 |
| BR-55-2 | Exchange名 | "eshop_event_bus"固定 | 常時 |
| BR-55-3 | 永続化 | Queue、メッセージともにdurable | 常時 |
| BR-55-4 | シングルトン | RabbitMQEventBusはシングルトン | 常時 |
| BR-55-5 | バックグラウンド起動 | LongRunningタスクで接続処理 | 常時 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを操作しない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | BrokerUnreachableException | RabbitMQサーバー接続不可 | リトライ後、ログ出力 |
| - | SocketException | ネットワークエラー | リトライ後、ログ出力 |
| - | InvalidOperationException | 接続未確立でPublish | 例外をスロー |

### リトライ仕様

- 発行時: Pollyによる指数バックオフリトライ
- 受信時: リトライなし（メッセージはAck）

## トランザクション仕様

RabbitMQ自体はトランザクションをサポートするが、本実装では使用していない。

## パフォーマンス要件

- 接続確立: 5秒以内
- メッセージ発行: 100ms以内
- スループット: 1000メッセージ/秒以上

## セキュリティ考慮事項

- RabbitMQ接続にはユーザー名・パスワードによる認証が必要
- TLS接続はオプションで設定可能
- 接続情報はAspire接続文字列で管理

## 備考

- .NET Aspireとの統合によりRabbitMQ接続管理を簡素化
- OpenTelemetryによる分散トレーシングで可観測性を向上

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、設定オプションとテレメトリクラスを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | EventBusOptions.cs | `src/EventBusRabbitMQ/EventBusOptions.cs` | キュー名とリトライ回数の設定 |
| 1-2 | RabbitMQTelemetry.cs | `src/EventBusRabbitMQ/RabbitMQTelemetry.cs` | OpenTelemetryのActivitySourceとPropagator |

**読解のコツ**:
- EventBusOptionsはシンプルな設定クラス
- RabbitMQTelemetryはOpenTelemetry連携用。ActivitySourceNameが"EventBusRabbitMQ"

**主要処理フロー（RabbitMQTelemetry.cs）**:
- **8行目**: ActivitySourceName = "EventBusRabbitMQ"
- **10行目**: ActivitySourceインスタンス生成
- **11行目**: DefaultTextMapPropagatorを使用

#### Step 2: DI登録を理解する

依存性注入の設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RabbitMqDependencyInjectionExtensions.cs | `src/EventBusRabbitMQ/RabbitMqDependencyInjectionExtensions.cs` | AddRabbitMqEventBusメソッド |

**主要処理フロー**:
- **15行目**: 設定セクション名は"EventBus"
- **17-40行目**: AddRabbitMqEventBusメソッド
- **21行目**: AspireのAddRabbitMQClientで接続を追加
- **24-28行目**: OpenTelemetryトレーシングを設定
- **31行目**: EventBusOptionsを設定からバインド
- **34行目**: RabbitMQTelemetryをシングルトン登録
- **35行目**: RabbitMQEventBusをIEventBusとして登録
- **37行目**: IHostedServiceとして同じインスタンスを登録

#### Step 3: 実装クラスを理解する

RabbitMQEventBusの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RabbitMQEventBus.cs | `src/EventBusRabbitMQ/RabbitMQEventBus.cs` | IEventBus、IHostedServiceの実装 |

**主要処理フロー**:
- **13-18行目**: プライマリコンストラクタで依存性注入
- **20行目**: ExchangeName = "eshop_event_bus"
- **22行目**: ResiliencePipelineでリトライポリシー構築
- **31-111行目**: PublishAsyncメソッド
- **226-295行目**: StartAsyncメソッド（IHostedService）
- **229-292行目**: バックグラウンドでRabbitMQ接続開始
- **235行目**: IConnectionをDIから取得
- **246行目**: コンシューマーチャネル作成
- **254-256行目**: Exchange宣言
- **258-263行目**: Queue宣言（durable: true）
- **279-285行目**: イベント型ごとにQueueBind

### プログラム呼び出し階層図

```
Program.cs / Host構築
    │
    └─ AddRabbitMqEventBus(connectionName)
           │
           ├─ AddRabbitMQClient（Aspire）
           │
           ├─ OpenTelemetry設定
           │      └─ AddSource("EventBusRabbitMQ")
           │
           ├─ Configure<EventBusOptions>
           │
           ├─ AddSingleton<RabbitMQTelemetry>
           │
           ├─ AddSingleton<IEventBus, RabbitMQEventBus>
           │
           └─ AddSingleton<IHostedService>（同インスタンス）

アプリケーション実行時
    │
    ├─ IHostedService.StartAsync
    │      │
    │      └─ RabbitMQEventBus
    │             ├─ IConnection取得
    │             ├─ ExchangeDeclare
    │             ├─ QueueDeclare
    │             ├─ QueueBind
    │             └─ BasicConsume
    │
    └─ IEventBus.PublishAsync
           │
           └─ RabbitMQEventBus
                  ├─ CreateChannel
                  ├─ ExchangeDeclare
                  └─ BasicPublish
```

### データフロー図

```
[設定]                    [DI登録]                        [実行時]

appsettings.json ───▶ EventBusOptions ───▶ RabbitMQEventBus
  EventBus:                                      │
    SubscriptionClientName                       ├─ 発行 ───▶ RabbitMQ Exchange
    RetryCount                                   │
                                                 └─ 受信 ◀─── RabbitMQ Queue
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| RabbitMQEventBus.cs | `src/EventBusRabbitMQ/RabbitMQEventBus.cs` | ソース | IEventBus・IHostedService実装 |
| RabbitMqDependencyInjectionExtensions.cs | `src/EventBusRabbitMQ/RabbitMqDependencyInjectionExtensions.cs` | ソース | DI登録拡張メソッド |
| EventBusOptions.cs | `src/EventBusRabbitMQ/EventBusOptions.cs` | ソース | 設定オプション |
| RabbitMQTelemetry.cs | `src/EventBusRabbitMQ/RabbitMQTelemetry.cs` | ソース | OpenTelemetry連携 |
| IEventBus.cs | `src/EventBus/Abstractions/IEventBus.cs` | ソース | インターフェース定義 |
