# 機能設計書 56-イベントログ保存

## 概要

本ドキュメントは、eShopシステムにおける「イベントログ保存」機能の設計仕様を定義する。この機能は、統合イベントをデータベースに永続化する機能を提供する。

### 本機能の処理概要

本機能は、マイクロサービスから発行される統合イベントをRDBMSに永続化する機能である。これにより、イベント発行の信頼性を向上させ、トランザクションと連携したイベント発行を可能にする。

**業務上の目的・背景**：マイクロサービスアーキテクチャでは、ローカルトランザクションの完了とイベント発行の原子性を保証することが課題である。本機能は、業務データの更新とイベントログの保存を同一トランザクションで行うことで、トランザクショナルアウトボックスパターンを実現する。これにより、データの整合性を保ちながら確実にイベントを発行できる。

**機能の利用シーン**：
- 商品価格更新時に、更新トランザクションと同時にProductPriceChangedIntegrationEventを保存
- 注文作成時に、注文トランザクションと同時にOrderStartedIntegrationEventを保存

**主要な処理内容**：
1. IntegrationEventとトランザクションIDを受け取る
2. IntegrationEventLogEntryエンティティを生成
3. 同一トランザクション内でログテーブルに保存
4. 保存後、バックグラウンドプロセスがイベントを発行

**関連システム・外部連携**：各マイクロサービスのEntity Framework Coreコンテキストと連携する。

**権限による制御**：内部サービス間の処理のため、特定の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし（バックエンド処理） |

## 機能種別

データ永続化 / トランザクショナルアウトボックス

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| event | IntegrationEvent | Yes | 保存するイベント | null不可 |
| transaction | IDbContextTransaction | Yes | 現在のトランザクション | null不可 |

### 入力データソース

各マイクロサービスの業務処理から呼び出される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | Task | 非同期タスク（成功時に完了） |

### 出力先

IntegrationEventLogテーブル。

## 処理フロー

### 処理シーケンス

```
1. SaveEventAsync呼び出し
   └─ IntegrationEventとトランザクションを受け取る

2. バリデーション
   └─ transactionがnullの場合、ArgumentNullExceptionをスロー

3. IntegrationEventLogEntry生成
   └─ EventId: イベントのId
   └─ CreationTime: イベントのCreationDate
   └─ EventTypeName: イベント型のFullName
   └─ Content: イベントをJSONシリアライズ
   └─ State: NotPublished
   └─ TimesSent: 0
   └─ TransactionId: トランザクションのId

4. トランザクション共有
   └─ UseTransactionで同一トランザクションを使用

5. エンティティ追加
   └─ DbSet<IntegrationEventLogEntry>.Add

6. 保存
   └─ SaveChangesAsync
```

### フローチャート

```mermaid
flowchart TD
    A[SaveEventAsync呼び出し] --> B{transactionがnull?}
    B -->|Yes| C[ArgumentNullException]
    B -->|No| D[IntegrationEventLogEntry生成]
    D --> E[JSONシリアライズ]
    E --> F[UseTransaction]
    F --> G[DbSet.Add]
    G --> H[SaveChangesAsync]
    H --> I[完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-56-1 | 同一トランザクション | 業務データと同一トランザクションでイベントログを保存 | 常時 |
| BR-56-2 | 初期状態 | 保存時のStateはNotPublished | 常時 |
| BR-56-3 | 送信回数初期化 | 保存時のTimesSentは0 | 常時 |
| BR-56-4 | JSONシリアライズ | イベント内容はインデント付きJSONで保存 | 常時 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| イベント保存 | IntegrationEventLog | INSERT | イベントログエントリを追加 |

### テーブル別操作詳細

#### IntegrationEventLog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | EventId | @event.Id | イベントの一意識別子 |
| INSERT | EventTypeName | @event.GetType().FullName | イベント型の完全修飾名 |
| INSERT | Content | JSONシリアライズ結果 | インデント付きJSON |
| INSERT | State | NotPublished (0) | 初期状態 |
| INSERT | TimesSent | 0 | 送信回数 |
| INSERT | CreationTime | @event.CreationDate | イベント作成日時 |
| INSERT | TransactionId | transaction.TransactionId | トランザクションID |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | transactionがnull | 例外をスロー |
| - | DbUpdateException | DB保存失敗 | 例外をスロー（呼び出し元でロールバック） |

### リトライ仕様

本機能自体はリトライを行わない。呼び出し元でトランザクションごとリトライする。

## トランザクション仕様

- 呼び出し元のトランザクションを共有（UseTransaction）
- 業務データの保存とイベントログの保存が同一トランザクションで実行
- トランザクションがコミットされるまでイベントログは永続化されない

## パフォーマンス要件

- 保存処理: 50ms以内

## セキュリティ考慮事項

- イベント内容はプレーンテキストのJSONで保存
- 機密情報を含むイベントの場合は暗号化を検討

## 備考

- トランザクショナルアウトボックスパターンの実装
- イベント発行は別のバックグラウンドプロセスで実行

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、イベントログエントリのエンティティを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IntegrationEventLogEntry.cs | `src/IntegrationEventLogEF/IntegrationEventLogEntry.cs` | イベントログのエンティティ定義 |
| 1-2 | EventStateEnum.cs | `src/IntegrationEventLogEF/EventStateEnum.cs` | イベント状態の列挙型 |

**読解のコツ（IntegrationEventLogEntry.cs）**:
- コンストラクタでイベントをJSONシリアライズして保存
- EventTypeShortNameはEventTypeNameからドット区切りの最後の部分を取得

**主要処理フロー（IntegrationEventLogEntry.cs）**:
- **7-8行目**: JSONシリアライズオプション（インデント付き、大文字小文字区別なし）
- **11-20行目**: コンストラクタでイベント情報を設定
- **16行目**: JSONシリアライズしてContentに保存
- **17行目**: State = NotPublished
- **22-33行目**: プロパティ定義
- **35-39行目**: DeserializeJsonContentでJSONからイベントを復元

**主要処理フロー（EventStateEnum.cs）**:
- **5行目**: NotPublished = 0（未発行）
- **6行目**: InProgress = 1（発行中）
- **7行目**: Published = 2（発行完了）
- **8行目**: PublishedFailed = 3（発行失敗）

#### Step 2: インターフェースを理解する

サービスインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | IIntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IIntegrationEventLogService.cs` | SaveEventAsyncメソッドの定義 |

**主要処理フロー**:
- **6行目**: SaveEventAsync(IntegrationEvent, IDbContextTransaction)

#### Step 3: 実装クラスを理解する

サービス実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IntegrationEventLogService.cs` | SaveEventAsyncの実装 |

**主要処理フロー**:
- **3-5行目**: ジェネリッククラスでDbContextを型パラメータとして受け取る
- **10-17行目**: コンストラクタでイベント型一覧を取得
- **34-44行目**: SaveEventAsyncメソッド
- **36行目**: transactionのnullチェック
- **38行目**: IntegrationEventLogEntryを生成
- **40行目**: UseTransactionで同一トランザクションを使用
- **41行目**: DbSetにエンティティを追加
- **43行目**: SaveChangesAsyncで保存

### プログラム呼び出し階層図

```
呼び出し元サービス（Catalog.API, Ordering.API等）
    │
    └─ IIntegrationEventLogService.SaveEventAsync
           │
           ├─ IntegrationEventLogEntry生成
           │      ├─ EventId, CreationTime設定
           │      ├─ EventTypeName設定
           │      ├─ JsonSerializer.Serialize
           │      └─ State = NotPublished
           │
           ├─ DbContext.UseTransaction
           │
           ├─ DbSet<IntegrationEventLogEntry>.Add
           │
           └─ DbContext.SaveChangesAsync
```

### データフロー図

```
[入力]                          [処理]                              [出力]

IntegrationEvent ───▶ IntegrationEventLogService ───▶ IntegrationEventLogテーブル
+ Transaction              ├─ JSONシリアライズ                       │
                           ├─ エンティティ生成                        ├─ EventId
                           └─ 同一トランザクションで保存               ├─ Content (JSON)
                                                                      ├─ State
                                                                      └─ TransactionId
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| IIntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IIntegrationEventLogService.cs` | ソース | サービスインターフェース |
| IntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IntegrationEventLogService.cs` | ソース | サービス実装 |
| IntegrationEventLogEntry.cs | `src/IntegrationEventLogEF/IntegrationEventLogEntry.cs` | ソース | イベントログエンティティ |
| EventStateEnum.cs | `src/IntegrationEventLogEF/EventStateEnum.cs` | ソース | イベント状態列挙型 |
| IntegrationLogExtensions.cs | `src/IntegrationEventLogEF/IntegrationLogExtensions.cs` | ソース | EF Coreモデル設定 |
