# 機能設計書 57-イベントログ取得

## 概要

本ドキュメントは、eShopシステムにおける「イベントログ取得」機能の設計仕様を定義する。この機能は、保存済みの統合イベントを取得する機能を提供する。

### 本機能の処理概要

本機能は、IntegrationEventLogテーブルから未発行のイベントを取得し、イベントバス経由で発行するために使用される機能である。トランザクショナルアウトボックスパターンの発行部分を担う。

**業務上の目的・背景**：SaveEventAsyncで保存されたイベントログは、トランザクションのコミット後に実際にメッセージブローカーへ発行される必要がある。本機能は、特定のトランザクションIDに紐づく未発行イベントを取得し、発行処理に引き渡す役割を担う。

**機能の利用シーン**：
- 業務トランザクションのコミット後、保存されたイベントを取得して発行
- バックグラウンドワーカーが定期的に未発行イベントをポーリング

**主要な処理内容**：
1. トランザクションIDを指定してRetrieveEventLogsPendingToPublishAsyncを呼び出し
2. State = NotPublishedのイベントをトランザクションIDでフィルタリング
3. CreationTime順にソート
4. JSONからIntegrationEventにデシリアライズして返却

**関連システム・外部連携**：各マイクロサービスのEntity Framework Coreコンテキストと連携する。

**権限による制御**：内部サービス間の処理のため、特定の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし（バックエンド処理） |

## 機能種別

データ取得 / トランザクショナルアウトボックス

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| transactionId | Guid | Yes | 対象トランザクションID | 有効なGuid |

### 入力データソース

IntegrationEventLogテーブルから取得。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | IEnumerable<IntegrationEventLogEntry> | 未発行イベントログエントリのコレクション |

各IntegrationEventLogEntryには以下が含まれる：

| 項目名 | 型 | 説明 |
|--------|-----|------|
| EventId | Guid | イベントの一意識別子 |
| EventTypeName | string | イベント型の完全修飾名 |
| Content | string | イベントのJSON文字列 |
| State | EventStateEnum | イベントの状態 |
| TimesSent | int | 送信試行回数 |
| CreationTime | DateTime | イベント作成日時 |
| TransactionId | Guid | トランザクションID |
| IntegrationEvent | IntegrationEvent | デシリアライズされたイベント |

### 出力先

呼び出し元サービスへ返却。

## 処理フロー

### 処理シーケンス

```
1. RetrieveEventLogsPendingToPublishAsync呼び出し
   └─ トランザクションIDを受け取る

2. データベースクエリ
   └─ TransactionId = 指定値
   └─ State = NotPublished
   └─ ToListAsyncで取得

3. 結果チェック
   └─ 結果が0件の場合、空のコレクションを返却

4. ソート・デシリアライズ
   └─ CreationTimeで昇順ソート
   └─ 各エントリでDeserializeJsonContentを実行
   └─ イベント型を動的に解決してデシリアライズ

5. 結果返却
   └─ IntegrationEventLogEntryのコレクションを返却
```

### フローチャート

```mermaid
flowchart TD
    A[RetrieveEventLogsPendingToPublishAsync] --> B[DBクエリ実行]
    B --> C{結果あり?}
    C -->|No| D[空コレクション返却]
    C -->|Yes| E[CreationTime順ソート]
    E --> F[イベント型解決]
    F --> G[JSONデシリアライズ]
    G --> H[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-57-1 | 未発行フィルタ | State = NotPublishedのイベントのみ取得 | 常時 |
| BR-57-2 | トランザクションスコープ | 指定されたトランザクションIDに紐づくイベントのみ取得 | 常時 |
| BR-57-3 | 時系列順序 | CreationTimeの昇順でソートして返却 | 常時 |
| BR-57-4 | 動的型解決 | イベント型名からTypeを動的に解決してデシリアライズ | 常時 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| イベント取得 | IntegrationEventLog | SELECT | 未発行イベントを取得 |

### テーブル別操作詳細

#### IntegrationEventLog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | TransactionId | = @transactionId | トランザクションIDで絞り込み |
| SELECT | State | = NotPublished (0) | 未発行のみ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | DbException | DB接続エラー | 例外をスロー |
| - | JsonException | デシリアライズ失敗 | 例外をスロー |

### リトライ仕様

本機能自体はリトライを行わない。呼び出し元で必要に応じてリトライする。

## トランザクション仕様

読み取り専用のクエリのため、特別なトランザクション制御は行わない。

## パフォーマンス要件

- 取得処理: 100ms以内
- 対象件数: 1トランザクションあたり通常1〜10件程度

## セキュリティ考慮事項

- イベント内容はプレーンテキストのJSONで保存されている
- 適切なアクセス制御でDBへのアクセスを制限

## 備考

- トランザクショナルアウトボックスパターンの取得部分
- 発行処理は呼び出し元で実行

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースを理解する

まず、サービスインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IIntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IIntegrationEventLogService.cs` | RetrieveEventLogsPendingToPublishAsyncメソッドの定義 |

**主要処理フロー**:
- **5行目**: RetrieveEventLogsPendingToPublishAsync(Guid transactionId)

#### Step 2: 実装クラスを理解する

サービス実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | IntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IntegrationEventLogService.cs` | RetrieveEventLogsPendingToPublishAsyncの実装 |

**主要処理フロー**:
- **10-17行目**: コンストラクタでアプリケーション内のIntegrationEvent派生型を収集
- **19-32行目**: RetrieveEventLogsPendingToPublishAsyncメソッド
- **21-23行目**: LINQクエリでTransactionIdとStateでフィルタリング
- **25行目**: 結果が0件の場合は空コレクションを返却
- **27-28行目**: CreationTime順ソート、DeserializeJsonContentを実行
- **28行目**: イベント型名から対応するTypeを取得してデシリアライズ

#### Step 3: デシリアライズ処理を理解する

エンティティのデシリアライズメソッドを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IntegrationEventLogEntry.cs | `src/IntegrationEventLogEF/IntegrationEventLogEntry.cs` | DeserializeJsonContentメソッド |

**主要処理フロー**:
- **24-26行目**: EventTypeShortNameはEventTypeNameの最後のドット区切り部分
- **35-39行目**: DeserializeJsonContentメソッド
- **37行目**: JsonSerializer.Deserializeで動的にデシリアライズ
- **38行目**: 自身を返却（流暢なインターフェース）

### プログラム呼び出し階層図

```
呼び出し元サービス（Catalog.API, Ordering.API等）
    │
    └─ IIntegrationEventLogService.RetrieveEventLogsPendingToPublishAsync
           │
           ├─ DbSet<IntegrationEventLogEntry>.Where
           │      ├─ TransactionId == transactionId
           │      └─ State == NotPublished
           │
           ├─ ToListAsync
           │
           ├─ OrderBy(CreationTime)
           │
           └─ Select(DeserializeJsonContent)
                  │
                  ├─ EventTypeShortName取得
                  │
                  ├─ _eventTypes.FirstOrDefault
                  │
                  └─ JsonSerializer.Deserialize
```

### データフロー図

```
[入力]                          [処理]                              [出力]

transactionId (Guid) ───▶ IntegrationEventLogService ───▶ IEnumerable<IntegrationEventLogEntry>
                              │                                     │
                              ├─ DBクエリ                           ├─ IntegrationEvent（デシリアライズ済み）
                              ├─ ソート                             ├─ EventId
                              └─ デシリアライズ                      └─ その他メタデータ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| IIntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IIntegrationEventLogService.cs` | ソース | サービスインターフェース |
| IntegrationEventLogService.cs | `src/IntegrationEventLogEF/Services/IntegrationEventLogService.cs` | ソース | サービス実装 |
| IntegrationEventLogEntry.cs | `src/IntegrationEventLogEF/IntegrationEventLogEntry.cs` | ソース | イベントログエンティティ・デシリアライズ |
| EventStateEnum.cs | `src/IntegrationEventLogEF/EventStateEnum.cs` | ソース | イベント状態列挙型 |
