# 機能設計書 58-埋め込みベクトル生成

## 概要

本ドキュメントは、eShopシステムにおける「埋め込みベクトル生成」機能の設計仕様を定義する。この機能は、商品情報からAI埋め込みベクトルを生成する機能を提供する。

### 本機能の処理概要

本機能は、商品の名前と説明文からAI（IEmbeddingGenerator）を使用してベクトル埋め込みを生成する機能である。生成されたベクトルはPostgreSQLのpgvector拡張を使用してデータベースに保存され、セマンティック検索に使用される。

**業務上の目的・背景**：従来のキーワード検索では、ユーザーの検索意図と商品情報のマッチングが不十分な場合がある。本機能は、商品情報を意味的なベクトル表現に変換することで、「軽いハイキング用の靴」のような自然言語クエリで関連商品を検索できるセマンティック検索を実現する。

**機能の利用シーン**：
- 商品登録時に埋め込みベクトルを自動生成
- 商品更新時に埋め込みベクトルを再生成
- AIチャットボットでの商品推薦

**主要な処理内容**：
1. CatalogItemから名前と説明文を結合したテキストを生成
2. IEmbeddingGeneratorでベクトル埋め込みを生成
3. 384次元に切り詰めてPgvector.Vector型で返却

**関連システム・外部連携**：Azure OpenAIまたはOllamaのEmbedding APIを使用する。

**権限による制御**：内部サービスの処理のため、特定の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし（バックエンド処理） |

## 機能種別

AI処理 / ベクトル埋め込み生成

## 入力仕様

### 入力パラメータ

#### GetEmbeddingAsync(CatalogItem)

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| item | CatalogItem | Yes | 商品エンティティ | null不可 |

#### GetEmbeddingAsync(string)

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| text | string | Yes | 埋め込み対象テキスト | null/空不可 |

### 入力データソース

CatalogItemエンティティまたは任意のテキスト文字列。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | Vector? | 384次元のベクトル埋め込み（AI無効時はnull） |

### 出力先

呼び出し元サービスへ返却。CatalogItemのEmbeddingプロパティに保存される。

## 処理フロー

### 処理シーケンス

```
1. GetEmbeddingAsync(CatalogItem)呼び出し
   └─ CatalogItemToStringで「{Name} {Description}」形式のテキストを生成
   └─ GetEmbeddingAsync(string)に委譲

2. GetEmbeddingAsync(string)呼び出し
   └─ IsEnabledチェック
   └─ falseの場合はnullを返却

3. ベクトル生成
   └─ Stopwatch.GetTimestampで計測開始
   └─ _embeddingGenerator.GenerateVectorAsyncを呼び出し
   └─ 結果を先頭384次元に切り詰め

4. ログ出力
   └─ LogLevel.Traceで生成時間とテキストを出力

5. 結果返却
   └─ new Vector(embedding)で返却
```

### フローチャート

```mermaid
flowchart TD
    A[GetEmbeddingAsync呼び出し] --> B{CatalogItem?}
    B -->|Yes| C[CatalogItemToString]
    C --> D[GetEmbeddingAsync/string/]
    B -->|No| D
    D --> E{IsEnabled?}
    E -->|No| F[null返却]
    E -->|Yes| G[GenerateVectorAsync]
    G --> H[384次元に切り詰め]
    H --> I[Vectorインスタンス生成]
    I --> J[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-58-1 | 次元数制限 | 埋め込みベクトルは384次元に切り詰め | 常時 |
| BR-58-2 | テキスト形式 | 「{Name} {Description}」形式でテキスト化 | CatalogItem入力時 |
| BR-58-3 | 無効時の動作 | AI機能が無効の場合はnullを返却 | IsEnabled = false |

### 計算ロジック

テキスト変換: `$"{item.Name} {item.Description}"`
次元切り詰め: `embedding[0..EmbeddingDimensions]`（EmbeddingDimensions = 384）

## データベース操作仕様

### 操作別データベース影響一覧

本機能自体はデータベースを操作しない。生成されたベクトルは呼び出し元でCatalogItemに保存される。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | Embedding API呼び出し失敗 | 例外をスロー |

### リトライ仕様

本機能自体はリトライを行わない。呼び出し元でリトライする。

## トランザクション仕様

外部API呼び出しのため、トランザクション管理は行わない。

## パフォーマンス要件

- 単一テキストの埋め込み生成: 1秒以内

## セキュリティ考慮事項

- Embedding APIへの接続は認証情報（APIキー等）が必要
- 商品情報は外部APIに送信されるため、機密情報は含めない

## 備考

- Microsoft.Extensions.AIのIEmbeddingGeneratorインターフェースを使用
- Pgvectorライブラリを使用してPostgreSQLのpgvector拡張と連携

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースを理解する

まず、サービスインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | AI機能のインターフェース定義 |

**主要処理フロー**:
- **8行目**: IsEnabledプロパティ（AI機能有効/無効）
- **11行目**: GetEmbeddingAsync(string text)
- **14行目**: GetEmbeddingAsync(CatalogItem item)
- **17行目**: GetEmbeddingsAsync(IEnumerable<CatalogItem> item)

#### Step 2: 実装クラスを理解する

サービス実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | 埋め込み生成の実装 |

**主要処理フロー**:
- **9行目**: EmbeddingDimensions = 384（次元数定数）
- **10行目**: IEmbeddingGenerator<string, Embedding<float>>をDI注入
- **17-22行目**: コンストラクタ
- **25行目**: IsEnabled => _embeddingGenerator is not null
- **28-31行目**: GetEmbeddingAsync(CatalogItem)
- **30行目**: CatalogItemToStringでテキスト化してから処理
- **55-73行目**: GetEmbeddingAsync(string)
- **61行目**: GenerateVectorAsyncでベクトル生成
- **62行目**: [0..EmbeddingDimensions]で384次元に切り詰め
- **69行目**: new Vector(embedding)で返却
- **75行目**: CatalogItemToString = `$"{item.Name} {item.Description}"`

### プログラム呼び出し階層図

```
呼び出し元（CatalogApi等）
    │
    └─ ICatalogAI.GetEmbeddingAsync(CatalogItem)
           │
           ├─ CatalogItemToString
           │      └─ "{Name} {Description}"
           │
           └─ GetEmbeddingAsync(string)
                  │
                  ├─ IsEnabled チェック
                  │
                  ├─ IEmbeddingGenerator.GenerateVectorAsync
                  │
                  ├─ [0..384] 次元切り詰め
                  │
                  └─ new Vector(embedding)
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CatalogItem ───▶ CatalogAI.GetEmbeddingAsync ───▶ Vector?
  ├─ Name                  │                           │
  └─ Description           ├─ テキスト化                ├─ 384次元ベクトル
                           ├─ Embedding API呼び出し     └─ または null
                           └─ 次元切り詰め

または

string text ───▶ CatalogAI.GetEmbeddingAsync ───▶ Vector?
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | ソース | AI機能インターフェース |
| CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | ソース | 埋め込み生成実装 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティ（Embeddingプロパティ） |
