# 機能設計書 59-一括埋め込み生成

## 概要

本ドキュメントは、eShopシステムにおける「一括埋め込み生成」機能の設計仕様を定義する。この機能は、複数商品の埋め込みベクトルを一括生成する機能を提供する。

### 本機能の処理概要

本機能は、複数のCatalogItemから一括でAI埋め込みベクトルを生成する機能である。単一アイテムごとに処理するよりも効率的にベクトル生成を行い、初期データ投入やバッチ処理に適している。

**業務上の目的・背景**：カタログの初期データ投入時や既存商品への一括埋め込み付与時に、商品を1つずつ処理するのは非効率である。本機能は、複数商品を一括でEmbedding APIに送信することで、API呼び出し回数を削減し、処理時間を短縮する。

**機能の利用シーン**：
- カタログデータベースの初期セットアップ時
- 既存商品への一括埋め込みベクトル付与
- バッチ処理による定期的なベクトル更新

**主要な処理内容**：
1. IEnumerable<CatalogItem>を受け取り、各アイテムをテキスト化
2. IEmbeddingGenerator.GenerateAsyncで一括ベクトル生成
3. 各結果を384次元に切り詰めてVector型に変換
4. IReadOnlyList<Vector>として返却

**関連システム・外部連携**：Azure OpenAIまたはOllamaのEmbedding APIを使用する。

**権限による制御**：内部サービスの処理のため、特定の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし（バックエンド処理） |

## 機能種別

AI処理 / バッチベクトル埋め込み生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| items | IEnumerable<CatalogItem> | Yes | 商品エンティティのコレクション | null不可、空不可 |

### 入力データソース

CatalogItemエンティティのコレクション。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | IReadOnlyList<Vector>? | 384次元のベクトル埋め込みのリスト（AI無効時はnull） |

### 出力先

呼び出し元サービスへ返却。各CatalogItemのEmbeddingプロパティに保存される。

## 処理フロー

### 処理シーケンス

```
1. GetEmbeddingsAsync呼び出し
   └─ IEnumerable<CatalogItem>を受け取る

2. 有効性チェック
   └─ IsEnabledをチェック
   └─ falseの場合はnullを返却

3. テキスト化
   └─ items.Select(CatalogItemToString)で各アイテムをテキスト化

4. 一括ベクトル生成
   └─ Stopwatch.GetTimestampで計測開始
   └─ _embeddingGenerator.GenerateAsyncを呼び出し

5. 結果変換
   └─ 各埋め込みを384次元に切り詰め
   └─ new Vector()でVector型に変換
   └─ ToList()でリスト化

6. ログ出力
   └─ LogLevel.Traceで生成件数と処理時間を出力

7. 結果返却
   └─ IReadOnlyList<Vector>として返却
```

### フローチャート

```mermaid
flowchart TD
    A[GetEmbeddingsAsync呼び出し] --> B{IsEnabled?}
    B -->|No| C[null返却]
    B -->|Yes| D[items.Select/CatalogItemToString/]
    D --> E[GenerateAsync]
    E --> F[各結果を384次元に切り詰め]
    F --> G[Vector型に変換]
    G --> H[ToList]
    H --> I[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-59-1 | 次元数制限 | 各埋め込みベクトルは384次元に切り詰め | 常時 |
| BR-59-2 | テキスト形式 | 「{Name} {Description}」形式でテキスト化 | 常時 |
| BR-59-3 | 無効時の動作 | AI機能が無効の場合はnullを返却 | IsEnabled = false |
| BR-59-4 | 順序保持 | 入力順序と出力順序は対応 | 常時 |

### 計算ロジック

テキスト変換: `items.Select(CatalogItemToString)` => `items.Select(i => $"{i.Name} {i.Description}")`
次元切り詰め: `embeddings.Select(m => new Vector(m.Vector[0..EmbeddingDimensions]))`

## データベース操作仕様

### 操作別データベース影響一覧

本機能自体はデータベースを操作しない。生成されたベクトルは呼び出し元で各CatalogItemに保存される。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | Embedding API呼び出し失敗 | 例外をスロー |

### リトライ仕様

本機能自体はリトライを行わない。呼び出し元でリトライする。

## トランザクション仕様

外部API呼び出しのため、トランザクション管理は行わない。

## パフォーマンス要件

- 一括埋め込み生成: 100件で10秒以内

## セキュリティ考慮事項

- Embedding APIへの接続は認証情報（APIキー等）が必要
- 商品情報は外部APIに送信されるため、機密情報は含めない

## 備考

- 単一アイテム処理より効率的（API呼び出し回数削減）
- 大量データ処理時はバッチサイズを考慮

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースを理解する

まず、サービスインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | GetEmbeddingsAsyncメソッドの定義 |

**主要処理フロー**:
- **17行目**: GetEmbeddingsAsync(IEnumerable<CatalogItem> item)

#### Step 2: 実装クラスを理解する

サービス実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | GetEmbeddingsAsyncの実装 |

**主要処理フロー**:
- **34-52行目**: GetEmbeddingsAsyncメソッド
- **36-37行目**: IsEnabledチェック
- **38行目**: Stopwatch.GetTimestampで計測開始
- **40行目**: items.Select(CatalogItemToString)でテキスト化
- **40行目**: _embeddingGenerator.GenerateAsyncで一括生成
- **41行目**: 各結果をVector型に変換、384次元に切り詰め
- **43-46行目**: LogLevel.Traceで生成件数と処理時間をログ
- **48行目**: resultsを返却
- **75行目**: CatalogItemToString = `$"{item.Name} {item.Description}"`

### プログラム呼び出し階層図

```
呼び出し元（CatalogApi - DbSeeder等）
    │
    └─ ICatalogAI.GetEmbeddingsAsync(IEnumerable<CatalogItem>)
           │
           ├─ IsEnabled チェック
           │
           ├─ items.Select(CatalogItemToString)
           │      └─ 各アイテムを"{Name} {Description}"に変換
           │
           ├─ IEmbeddingGenerator.GenerateAsync
           │      └─ 一括でEmbedding API呼び出し
           │
           ├─ Select(m => new Vector(m.Vector[0..384]))
           │      └─ 各結果を384次元に切り詰めてVector化
           │
           └─ ToList()
                  └─ IReadOnlyList<Vector>として返却
```

### データフロー図

```
[入力]                                [処理]                              [出力]

IEnumerable<CatalogItem> ───▶ CatalogAI.GetEmbeddingsAsync ───▶ IReadOnlyList<Vector>?
  ├─ Item1 (Name, Desc)           │                                   │
  ├─ Item2 (Name, Desc)           ├─ 一括テキスト化                    ├─ Vector1 (384次元)
  └─ ...                          ├─ Embedding API一括呼び出し         ├─ Vector2 (384次元)
                                  └─ 次元切り詰め・変換                └─ ...
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | ソース | AI機能インターフェース |
| CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | ソース | 一括埋め込み生成実装 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティ |
