# 機能設計書 6-AI商品検索

## 概要

本ドキュメントは、eShopアプリケーションにおけるAI商品検索機能の設計仕様を定義する。AIベクトル埋め込み（Embedding）を使用したセマンティック検索により、検索テキストに意味的に関連する商品をページネーション付きで取得するREST APIエンドポイントを提供する。

### 本機能の処理概要

**業務上の目的・背景**：従来のキーワード検索では、ユーザーが正確な商品名を知らない場合や、曖昧な表現で検索した場合に適切な結果を返せない。AI埋め込みを使用したセマンティック検索により、検索テキストの「意味」を理解し、関連性の高い商品を推薦することで、ユーザーの商品発見体験を向上させる。

**機能の利用シーン**：AIチャットボットでの商品推薦、自然言語での商品検索（「夏に着られる涼しい服」など）、類似商品の検索など、キーワード完全一致では対応できない高度な検索ニーズに対応する。

**主要な処理内容**：
1. 検索テキストを受け取り、AI埋め込みベクトルを生成
2. 商品の埋め込みベクトルとのコサイン距離を計算
3. 距離が近い（類似度が高い）順に商品を返却
4. AIが無効な場合は商品名検索にフォールバック

**関連システム・外部連携**：Azure OpenAI等のEmbedding生成サービス、PostgreSQLのpgvector拡張機能を使用。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | チャットボット | 主画面 | ChatStateでセマンティック検索による商品推薦 |

## 機能種別

AI検索 / セマンティック検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| text | string | Yes | 検索テキスト | V1: minlength(1)、V2: [Required, MinLength(1)] |
| PageSize | int | No | 1ページあたりの取得件数 | デフォルト値: 10 |
| PageIndex | int | No | 取得するページのインデックス | デフォルト値: 0 |

### 入力データソース

- V1: URLパスパラメータ + クエリパラメータ: `/api/catalog/items/withsemanticrelevance/{text}`
- V2: クエリパラメータ: `/api/catalog/items/withsemanticrelevance?text={text}`

## 出力仕様

### 出力データ

**正常時（200 OK）: PaginatedItems&lt;CatalogItem&gt;**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PageIndex | int | 現在のページインデックス |
| PageSize | int | 1ページあたりの件数 |
| Count | long | 総件数 |
| Data | IEnumerable&lt;CatalogItem&gt; | 商品データの配列（類似度順） |

### 出力先

- HTTP Response Body（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ パス/クエリパラメータからtextを取得

2. AI有効性チェック
   └─ IsEnabled == false の場合、商品名検索にフォールバック

3. 埋め込みベクトル生成
   ├─ CatalogAI.GetEmbeddingAsync(text)でベクトル生成
   └─ vector == null の場合、商品名検索にフォールバック

4. セマンティック検索実行
   ├─ Embedding != null の商品のみを対象
   ├─ CosineDistance(vector)でコサイン距離を計算
   └─ 距離が近い順（類似度が高い順）にソート

5. ページネーション適用
   ├─ Skip/Takeでページデータ取得
   └─ デバッグログ出力（LogLevel.Debug有効時）

6. レスポンス生成
   └─ PaginatedItems<CatalogItem>を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[text取得]
    B --> C{AI有効?}
    C -->|No| D[商品名検索フォールバック]
    C -->|Yes| E[埋め込みベクトル生成]
    E --> F{vector != null?}
    F -->|No| D
    F -->|Yes| G[総件数カウント]
    G --> H{Debug有効?}
    H -->|Yes| I[距離付きでクエリ実行]
    H -->|No| J[通常クエリ実行]
    I --> K[距離ログ出力]
    K --> L[PaginatedItems生成]
    J --> L
    D --> L
    L --> M[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | フォールバック動作 | AIが無効または埋め込み生成失敗時は商品名検索にフォールバック | AI無効時/ベクトル生成失敗時 |
| BR-002 | Embeddingフィルタ | Embeddingがnullの商品は検索対象外 | AI検索実行時 |
| BR-003 | コサイン距離ソート | コサイン距離が小さい順（類似度が高い順）にソート | AI検索実行時 |
| BR-004 | デバッグログ | LogLevel.Debug有効時は距離情報をログ出力 | Debug有効時 |

### 計算ロジック

- **コサイン距離**: `c.Embedding!.CosineDistance(vector)` - pgvectorの組み込み関数を使用
- **埋め込み次元**: 384次元（EmbeddingDimensions定数）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| セマンティック検索 | Catalog | SELECT | コサイン距離でソートして取得 |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE Embedding IS NOT NULL ORDER BY Embedding <=> {vector} | pgvector演算子使用 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | textが空文字または未指定 | ルーティング/バリデーションで制約 |
| 400 | BadRequest | パラメータ形式が不正 | ProblemDetails形式でエラー詳細を返却 |
| 500 | InternalServerError | AI埋め込み生成エラー | フォールバックで商品名検索を実行 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。AI埋め込み生成失敗時は商品名検索へのフォールバックで対応。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。

## パフォーマンス要件

- pgvector拡張によりベクトル検索を効率的に実行
- 埋め込みがnullの商品を除外することでインデックス効率を向上
- AI埋め込み生成はIEmbeddingGenerator経由で非同期実行

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- 検索テキストはAIサービスに送信される（プライバシー考慮が必要な場合あり）

## 備考

- AI機能が無効な場合や埋め込み生成に失敗した場合は、自動的に商品名検索（No.5）にフォールバック
- 商品登録時にEmbeddingが設定されていない商品は検索対象外

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | Embeddingプロパティ（Vector型、JsonIgnore） |
| 1-2 | ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | AI機能のインターフェース定義 |

**読解のコツ**: EmbeddingプロパティはPgvector.Vector型で、384次元のベクトルを格納。JsonIgnore属性により通常のJSON応答には含まれない。

#### Step 2: AI機能の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | 埋め込みベクトル生成の実装 |

**主要処理フロー**:
1. **9行目**: EmbeddingDimensions = 384（ベクトル次元数）
2. **25行目**: IsEnabledプロパティ（_embeddingGenerator != null）
3. **55-73行目**: GetEmbeddingAsync(string text)メソッド
4. **61行目**: GenerateVectorAsyncでベクトル生成
5. **62行目**: 384次元に切り詰め
6. **69行目**: new Vector(embedding)でPgvector型に変換

#### Step 3: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemsBySemanticRelevanceメソッド |

**主要処理フロー**:
1. **53-57行目**: V1エンドポイント `/items/withsemanticrelevance/{text}`
2. **60-64行目**: V2エンドポイント `/items/withsemanticrelevance`
3. **237-289行目**: `GetItemsBySemanticRelevance` メソッド実装
4. **245-248行目**: AI無効時のフォールバック
5. **251行目**: GetEmbeddingAsync(text)でベクトル生成
6. **253-256行目**: ベクトル生成失敗時のフォールバック
7. **267-272行目**: Debug時の距離付きクエリ
8. **280-285行目**: 通常クエリ（CosineDistanceでソート）

#### Step 4: データベース設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | pgvector拡張の有効化 |
| 4-2 | CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | Embeddingカラム設定 |

**主要処理フロー**:
- **CatalogContext.cs 20行目**: `builder.HasPostgresExtension("vector")`
- **EntityConfiguration 13-14行目**: `HasColumnType("vector(384)")`

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items/withsemanticrelevance)
    │
    ├─ CatalogApi.GetItemsBySemanticRelevanceV1() [V1の場合]
    │      └─ CatalogApi.GetItemsBySemanticRelevance()
    │
    └─ CatalogApi.GetItemsBySemanticRelevance() [V2の場合]
           │
           ├─ CatalogAI.IsEnabled チェック
           │      └─ false → GetItemsByName() フォールバック
           │
           ├─ CatalogAI.GetEmbeddingAsync(text)
           │      ├─ IEmbeddingGenerator.GenerateVectorAsync()
           │      └─ null → GetItemsByName() フォールバック
           │
           └─ CatalogContext.CatalogItems
                  ├─ Where(c => c.Embedding != null)
                  ├─ OrderBy(c => c.Embedding.CosineDistance(vector))
                  ├─ Skip/Take
                  └─ ToListAsync()
                         │
                         └─ PaginatedItems<CatalogItem>
```

### データフロー図

```
[入力]                       [処理]                              [出力]

HTTPリクエスト               GetItemsBySemanticRelevance()       HTTPレスポンス
  │                              │                                  │
  └─ text ─────────────────────▶│                                  │
     "涼しい夏服"                │                                  │
                                 │   AI Embedding Service           │
                                 │   (Azure OpenAI等)               │
                                 │      │                           │
                                 │      ▼                           │
                                 │   Vector[384]                    │
                                 │      │                           │
                                 │      ▼                           │
                                 │   PostgreSQL (pgvector)          │
                                 │   ORDER BY cosine_distance       │
                                 │      │                           │
                                 │      ▼                           │
                                 └─▶ PaginatedItems ───────────────▶ JSON
                                      ├─ PageIndex
                                      ├─ PageSize
                                      ├─ Count
                                      └─ Data[] (類似度順)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | Embeddingプロパティ |
| ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | ソース | AIインターフェース |
| CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | ソース | AI実装（埋め込み生成） |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | pgvector拡張有効化 |
| CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | ソース | vector(384)カラム設定 |
