# 機能設計書 60-HybridAppカタログ

## 概要

本ドキュメントは、eShopシステムにおける「HybridAppカタログ」機能の設計仕様を定義する。この機能は、MAUI Blazor Hybridアプリでのカタログ表示機能を提供する。

### 本機能の処理概要

本機能は、.NET MAUI Blazor Hybridアプリケーションにおいて、商品カタログを一覧表示する画面コンポーネントである。WebAppと同様のBlazorコンポーネントを使用しつつ、モバイル/デスクトップアプリとしてネイティブに動作する。

**業務上の目的・背景**：eShopは複数のプラットフォーム（Web、モバイル、デスクトップ）で商品カタログを提供する必要がある。MAUI Blazor Hybridを採用することで、Blazorコンポーネントを再利用しながらネイティブアプリを開発でき、開発効率とユーザー体験を両立する。

**機能の利用シーン**：
- スマートフォン（iOS/Android）でのカタログ閲覧
- デスクトップ（Windows/Mac）でのカタログ閲覧
- オフライン環境でのカタログ確認（将来的な拡張）

**主要な処理内容**：
1. CatalogService経由でCatalog.APIから商品一覧を取得
2. ブランド・タイプによるフィルタリング機能を提供
3. ページネーション機能を提供
4. 商品詳細画面への遷移リンクを表示

**関連システム・外部連携**：Catalog.APIのREST APIエンドポイントを使用する。

**権限による制御**：カタログ表示は認証不要（公開情報）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 25 | カタログ画面（HybridApp） | 主画面 | CatalogServiceで商品一覧を取得・表示 |
| 26 | 商品詳細画面（HybridApp） | 遷移先画面 | CatalogListItemクリックで商品詳細画面へ遷移 |

## 機能種別

画面表示 / データ取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| page | int? | No | ページ番号（クエリパラメータ） | 1以上 |
| brand | int? | No | ブランドID（クエリパラメータ） | 有効なブランドID |
| type | int? | No | タイプID（クエリパラメータ） | 有効なタイプID |

### 入力データソース

Catalog.API経由でデータを取得。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| catalogResult | CatalogResult | 商品一覧データ |
| catalogResult.Data | CatalogItem[] | 商品アイテムの配列 |
| catalogResult.Count | int | 総件数 |

### 出力先

MAUI Blazor Hybridアプリ画面に表示。

## 処理フロー

### 処理シーケンス

```
1. ページ初期化（OnParametersSetAsync）
   └─ クエリパラメータからPage, BrandId, ItemTypeIdを取得

2. データ取得
   └─ CatalogService.GetCatalogItemsを呼び出し
   └─ ページインデックス、ページサイズ（9）、ブランド、タイプを指定

3. 画面表示
   └─ ローディング中：「Loading...」を表示
   └─ データ取得後：CatalogListItemコンポーネントで各商品を表示

4. ページネーション
   └─ GetVisiblePageIndexesで表示するページ番号を計算
   └─ NavLinkで各ページへのリンクを生成

5. フィルタリング（CatalogSearchコンポーネント）
   └─ CatalogService.GetBrandsでブランド一覧取得
   └─ CatalogService.GetTypesでタイプ一覧取得
   └─ NavLinkでフィルタリング用リンクを生成
```

### フローチャート

```mermaid
flowchart TD
    A[ページ初期化] --> B[クエリパラメータ解析]
    B --> C[CatalogService.GetCatalogItems]
    C --> D{データ取得成功?}
    D -->|Yes| E[CatalogListItemで表示]
    D -->|No| F[Loading表示継続]
    E --> G[ページネーション表示]

    H[CatalogSearch初期化] --> I[GetBrands / GetTypes]
    I --> J[フィルタリングUI表示]
    J --> K{フィルタ選択?}
    K -->|Yes| L[クエリパラメータ更新]
    L --> A
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-60-1 | ページサイズ | 1ページあたり9件表示 | 常時 |
| BR-60-2 | デフォルトページ | pageパラメータ未指定時は1ページ目を表示 | 常時 |
| BR-60-3 | フィルタリングリセット | フィルタ変更時はページを1にリセット | フィルタ変更時 |
| BR-60-4 | API呼び出し | api-version=2.0を使用 | 常時 |

### 計算ロジック

ページ数計算: `Math.Ceiling(1.0 * result.Count / PageSize)`
ページインデックス: `Page.GetValueOrDefault(1) - 1`（0ベース）

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを直接操作しない。Catalog.API経由でデータを取得する。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | API経由でデータ取得 |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | HttpRequestException | API接続エラー | ローディング状態を継続 |

### リトライ仕様

特になし。

## トランザクション仕様

読み取り専用のため、トランザクション管理は行わない。

## パフォーマンス要件

- 画面初期表示: 2秒以内
- API応答時間: 1秒以内

## セキュリティ考慮事項

- カタログ情報は公開データのため、認証不要
- HTTPS通信でAPIと接続

## 備考

- .NET MAUI Blazor Hybrid技術により、Blazorコンポーネントをネイティブアプリで再利用
- カート機能は現在コメントアウト（将来的な実装予定）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、共通の型定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogResult | `src/WebAppComponents/Catalog/CatalogResult.cs` | 商品一覧のレスポンス型 |
| 1-2 | CatalogItem | `src/WebAppComponents/Catalog/CatalogItem.cs` | 商品アイテムの型 |

**読解のコツ**: WebAppComponentsプロジェクトで共通型を定義し、WebApp/HybridAppで共有。

#### Step 2: サービス層を理解する

APIクライアントの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogService.cs | `src/HybridApp/Services/CatalogService.cs` | Catalog.APIとの通信 |

**主要処理フロー**:
- **10行目**: remoteServiceBaseUrl = "api/catalog/"
- **19-24行目**: GetCatalogItemsメソッド
- **21行目**: GetAllCatalogItemsUriでクエリ文字列を構築
- **22行目**: httpClient.GetFromJsonAsyncでAPI呼び出し
- **40-45行目**: GetBrandsメソッド
- **47-52行目**: GetTypesメソッド
- **54-67行目**: GetAllCatalogItemsUriでフィルタパラメータを組み立て

#### Step 3: 画面コンポーネントを理解する

Blazorコンポーネントの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Catalog.razor | `src/HybridApp/Components/Pages/Catalog/Catalog.razor` | メインカタログ画面 |
| 3-2 | CatalogSearch.razor | `src/HybridApp/Components/Pages/Catalog/CatalogSearch.razor` | フィルタリングコンポーネント |

**主要処理フロー（Catalog.razor）**:
- **1行目**: @page "/"でルートページ
- **37行目**: PageSize = 9（定数）
- **39-46行目**: クエリパラメータの定義（Page, BrandId, ItemTypeId）
- **50-51行目**: GetVisiblePageIndexesでページ番号を計算
- **53-59行目**: OnParametersSetAsyncでデータ取得

**主要処理フロー（CatalogSearch.razor）**:
- **55-62行目**: OnInitializedAsyncでブランド・タイプを並列取得
- **64-68行目**: BrandUriでブランドフィルタ用URLを生成
- **70-74行目**: TypeUriでタイプフィルタ用URLを生成

### プログラム呼び出し階層図

```
Catalog.razor（メイン画面）
    │
    ├─ CatalogSearch.razor（フィルタ）
    │      │
    │      ├─ CatalogService.GetBrands
    │      │      └─ GET /api/catalog/catalogBrands
    │      │
    │      └─ CatalogService.GetTypes
    │             └─ GET /api/catalog/catalogTypes
    │
    ├─ CatalogService.GetCatalogItems
    │      └─ GET /api/catalog/items?...
    │
    └─ CatalogListItem（各商品）
           └─ NavLinkで商品詳細へ遷移
```

### データフロー図

```
[入力]                       [処理]                           [出力]

クエリパラメータ ───▶ Catalog.razor ───▶ 商品一覧表示
  ├─ page                    │                    │
  ├─ brand                   ├─ CatalogService    ├─ CatalogListItem
  └─ type                    │                    └─ ページネーション
                             │
                             └─ Catalog.API
                                   │
                                   └─ GET /api/catalog/items
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Catalog.razor | `src/HybridApp/Components/Pages/Catalog/Catalog.razor` | ソース | カタログメイン画面 |
| CatalogSearch.razor | `src/HybridApp/Components/Pages/Catalog/CatalogSearch.razor` | ソース | フィルタリングUI |
| CatalogService.cs | `src/HybridApp/Services/CatalogService.cs` | ソース | APIクライアント |
| ItemPage.razor | `src/HybridApp/Components/Pages/Item/ItemPage.razor` | ソース | 商品詳細画面 |
| CatalogListItem | `src/WebAppComponents/Catalog/CatalogListItem.razor` | ソース | 商品アイテムコンポーネント（共有） |
| MainLayout.razor | `src/HybridApp/Components/Layout/MainLayout.razor` | ソース | レイアウト |
