# 機能設計書 61-HybridApp商品詳細

## 概要

本ドキュメントは、eShopシステムのMAUI Blazor Hybridアプリケーションにおける商品詳細表示機能の設計仕様を定義する。本機能はモバイル・デスクトップアプリにおいて、ユーザーが選択した商品の詳細情報を表示するUI機能である。

### 本機能の処理概要

本機能は、MAUI Blazor Hybridテクノロジーを使用して、iOS、Android、macOS、Windows各プラットフォームのネイティブアプリ上で商品詳細情報を表示する機能である。Blazorコンポーネントを活用し、クロスプラットフォームで統一されたUI/UXを提供する。

**業務上の目的・背景**：eコマースプラットフォームにおいて、ユーザーが購入前に商品の詳細な情報（名前、説明、価格、ブランド、画像）を確認することは購買意思決定において不可欠である。本機能は、モバイルデバイスやデスクトップアプリケーションを通じて、いつでもどこでも商品情報にアクセスできる利便性を提供し、オムニチャネル戦略の一環としてユーザー体験を向上させる。WebアプリケーションとUI/UXを統一することで、ブランドの一貫性を維持しつつ、ネイティブアプリならではのオフラインアクセスやプッシュ通知などの拡張性を確保する。

**機能の利用シーン**：本機能は以下のシーンで利用される。
- カタログ画面で商品一覧を閲覧中に、特定の商品をタップ/クリックして詳細を確認する場面
- URLディープリンクから直接商品詳細ページにアクセスする場面
- 商品の価格、説明、ブランド情報を確認して購入を検討する場面

**主要な処理内容**：
1. ルートパラメータからitemIdを取得し、バリデーションを行う
2. CatalogServiceを通じてCatalog.APIから商品情報を非同期取得する
3. 取得した商品情報（名前、説明、価格、ブランド）を画面に描画する
4. ProductImageUrlProviderを通じて商品画像URLを生成し表示する
5. 商品が見つからない場合は404エラー画面を表示する

**関連システム・外部連携**：
- Catalog.API: RESTful APIを通じて商品詳細データを取得（GET /api/catalog/items/{id}）
- Mobile BFF: ローカルホスト経由でAPIにアクセス（Android: 10.0.2.2:11632、その他: localhost:11632）

**権限による制御**：
- 現時点では認証・認可は実装されていない（コメントアウト済み）
- 商品詳細の閲覧は全ユーザー（匿名ユーザー含む）に許可されている
- カート追加機能は将来的にログインユーザーのみに制限される設計

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 26 | 商品詳細画面 | 主画面 | MAUI Blazor Hybrid用商品詳細表示機能 |
| 25 | カタログ画面 | 遷移元画面 | CatalogListItemクリックで商品詳細画面へ遷移 |

## 機能種別

- 参照処理（READ）
- 画面表示機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| itemId | int | Yes | 表示する商品の一意識別子 | 正の整数であること。URLルートパラメータとして受け取る |

### 入力データソース

- URLルートパラメータ: `/item/{itemId:int}` 形式でアクセス
- Catalog.API: 商品詳細情報をREST APIから取得

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Name | string | 商品名（PageTitle、ヘッダーにも表示） |
| Description | string | 商品の詳細説明文 |
| Price | decimal | 商品価格（$形式で小数点2桁表示） |
| CatalogBrand.Brand | string | ブランド名（サブタイトルとしても表示） |
| 商品画像 | URL | ProductImageUrlProviderで生成された画像URL |

### 出力先

- Blazor Razorコンポーネント（ItemPage.razor）によるUI描画
- BlazorWebView経由でネイティブアプリ画面に表示

## 処理フロー

### 処理シーケンス

```
1. ページ初期化（OnInitializedAsync）
   └─ Blazorコンポーネントがルートパラメータ ItemId を受け取る

2. 商品情報取得
   └─ CatalogService.GetCatalogItem(ItemId) を呼び出し
      └─ HTTP GET リクエストを Catalog.API に送信
         └─ URI: api/catalog/items/{id}?api-version=2.0

3. レスポンス処理
   └─ 成功: CatalogItem オブジェクトを item 変数に格納
   └─ 404エラー: notFound フラグを true に設定

4. UI描画
   └─ item が存在: 商品詳細情報を表示
      └─ PageTitle, SectionContent（タイトル/サブタイトル）を設定
      └─ 商品画像、説明、ブランド、価格を表示
   └─ notFound が true: "Not found" エラーメッセージを表示
```

### フローチャート

```mermaid
flowchart TD
    A[ページアクセス /item/{itemId}] --> B[OnInitializedAsync開始]
    B --> C[CatalogService.GetCatalogItem呼び出し]
    C --> D{APIレスポンス}
    D -->|成功| E[itemに商品情報を格納]
    D -->|404エラー| F[notFound = true]
    E --> G[商品詳細表示]
    G --> H[PageTitle設定]
    H --> I[ヘッダー情報設定]
    I --> J[商品画像表示]
    J --> K[商品説明・価格表示]
    K --> L[終了]
    F --> M[Not Found画面表示]
    M --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-61-01 | 商品ID検証 | 商品IDは1以上の正の整数である必要がある | URLルートパラメータ受け取り時 |
| BR-61-02 | 存在チェック | 指定されたIDの商品が存在しない場合はエラー表示 | API応答が404の場合 |
| BR-61-03 | 価格表示形式 | 価格は小数点2桁のドル表記で表示（例: $12.99） | 商品価格表示時 |
| BR-61-04 | ブランド表示 | ブランド情報がある場合はヘッダーサブタイトルとして表示 | 商品詳細表示時 |

### 計算ロジック

- 価格表示: `item.Price.ToString("0.00")` で小数点2桁に整形
- 画像URL生成: `{MobileBffHost}api/catalog/items/{productId}/pic?api-version=2.0`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品取得 | CatalogItems | SELECT | 指定IDの商品情報を取得 |
| ブランド取得 | CatalogBrands | SELECT | 商品に紐づくブランド情報をIncludeで取得 |

### テーブル別操作詳細

#### CatalogItems（Catalog.API経由）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Name, Description, Price, PictureFileName, CatalogBrandId, CatalogTypeId | WHERE Id = @itemId | 単一レコード取得 |

#### CatalogBrands（Include結合）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Brand | CatalogItems.CatalogBrandId = CatalogBrands.Id | Eager Loading |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | NotFound | 指定されたIDの商品が存在しない | notFoundフラグをtrueに設定し、エラーメッセージを表示 |
| HttpRequestException | ネットワークエラー | API接続に失敗した場合 | 現状は未処理（例外がスローされる） |

### リトライ仕様

- 現時点ではリトライ処理は実装されていない
- ネットワーク障害時はユーザーによるページ再読み込みが必要

## トランザクション仕様

- 本機能は参照のみのため、トランザクション管理は不要
- Catalog.API側でDbContextによるクエリが実行される

## パフォーマンス要件

| 要件項目 | 目標値 | 備考 |
|---------|--------|------|
| API応答時間 | 500ms以下 | ローカルネットワーク経由 |
| 画像読み込み | 1秒以下 | 画像サイズに依存 |
| 初期表示 | 1秒以下 | アプリ内ナビゲーション |

## セキュリティ考慮事項

| 項目 | 内容 |
|------|------|
| 認証 | 現時点では未実装（将来的にOIDC認証を予定） |
| 認可 | 商品詳細閲覧は全ユーザーに許可 |
| データ保護 | 商品情報は公開情報のため特別な保護は不要 |
| 通信 | 開発環境ではHTTP、本番環境ではHTTPS使用を推奨 |
| API保護 | APIバージョニング（api-version=2.0）で互換性を管理 |

## 備考

- 本機能は現在、カート追加機能がコメントアウトされており、閲覧のみの機能となっている
- 将来的にはBasketState統合により、商品をカートに追加する機能が有効化される予定
- モバイルBFFのホストアドレスは、プラットフォーム判定により動的に決定される
  - Android: `http://10.0.2.2:11632/`（エミュレータからホストマシンへのアクセス用）
  - その他（iOS、macOS、Windows）: `http://localhost:11632/`
- WebAppComponentsプロジェクトとの共有コンポーネント（CatalogItem、IProductImageUrlProvider等）を使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、商品データの構造とサービスインターフェースを理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | CatalogItem recordの構造（Id, Name, Description, Price, CatalogBrand等）を把握 |
| 1-2 | ICatalogService.cs | `src/WebAppComponents/Services/ICatalogService.cs` | GetCatalogItemメソッドのシグネチャを確認 |
| 1-3 | IProductImageUrlProvider.cs | `src/WebAppComponents/Services/IProductImageUrlProvider.cs` | 画像URL生成インターフェースの仕様を確認 |

**読解のコツ**: C# recordは不変のデータ転送オブジェクトで、プロパティベースの等値比較が自動実装される。CatalogItemは7つのプロパティを持つシンプルなDTOである。

#### Step 2: エントリーポイントを理解する

MAUIアプリケーションの起動フローとBlazor統合を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | MauiProgram.cs | `src/HybridApp/MauiProgram.cs` | DIコンテナへのサービス登録、HttpClient設定、MobileBffHost定義 |
| 2-2 | MainPage.xaml | `src/HybridApp/MainPage.xaml` | BlazorWebViewの定義とルートコンポーネント設定 |
| 2-3 | Routes.razor | `src/HybridApp/Components/Routes.razor` | Blazorルーターの設定とデフォルトレイアウト |

**主要処理フロー**:
1. **9行目（MauiProgram.cs）**: MobileBffHostがプラットフォーム別に設定される
2. **30行目（MauiProgram.cs）**: CatalogServiceがHttpClientと共にDI登録される
3. **31行目（MauiProgram.cs）**: ProductImageUrlProviderがSingletonとして登録される
4. **8行目（MainPage.xaml）**: BlazorWebViewがwwwroot/index.htmlをホストページとして使用

#### Step 3: 商品詳細画面コンポーネントを理解する

ItemPage.razorの実装詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ItemPage.razor | `src/HybridApp/Components/Pages/Item/ItemPage.razor` | メインの商品詳細表示ロジック |

**主要処理フロー**:
- **1行目**: `@page "/item/{itemId:int}"` でルート定義、itemIdはint型制約
- **3行目**: CatalogServiceの依存性注入
- **6行目**: IProductImageUrlProviderの依存性注入
- **41行目**: CatalogItem?型のitem変数定義（null許容）
- **44行目**: notFoundフラグ（商品未発見時のUI制御用）
- **46-47行目**: ItemIdパラメータ定義（ルートパラメータからバインド）
- **52-65行目**: OnInitializedAsyncでの非同期データ取得
  - **57行目**: CatalogService.GetCatalogItem(ItemId)呼び出し
  - **60-64行目**: HttpStatusCode.NotFound例外のキャッチとnotFoundフラグ設定
- **8-31行目**: 商品情報が存在する場合のUI描画
  - **10行目**: PageTitleに商品名を設定
  - **11行目**: ヘッダータイトルに商品名を設定
  - **12行目**: ヘッダーサブタイトルにブランド名を設定
  - **15行目**: ProductImages.GetProductImageUrl(item.Id)で画像URL生成
  - **22行目**: 価格を$0.00形式で表示
- **32-38行目**: 商品が見つからない場合のエラーUI

#### Step 4: サービス層を理解する

CatalogServiceとProductImageUrlProviderの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CatalogService.cs | `src/HybridApp/Services/CatalogService.cs` | ICatalogServiceの実装、API呼び出しロジック |
| 4-2 | ProductImageUrlProvider.cs | `src/HybridApp/Services/ProductImageUrlProvider.cs` | 商品画像URL生成ロジック |
| 4-3 | CatalogJsonContext.cs | `src/HybridApp/Services/CatalogJsonContext.cs` | System.Text.Json用のソース生成コンテキスト |

**主要処理フロー**:
- **8行目（CatalogService.cs）**: Primary Constructorでの HttpClient受け取り
- **10行目（CatalogService.cs）**: remoteServiceBaseUrl = "api/catalog/"
- **13-17行目（CatalogService.cs）**: GetCatalogItemメソッド
  - **15行目**: URI構築（items/{id}?api-version=2.0）
  - **16行目**: GetFromJsonAsyncでJSON逆シリアル化（CatalogJsonContext使用）
- **7-8行目（ProductImageUrlProvider.cs）**: GetProductImageUrlの実装
  - MobileBffHost + "api/catalog/items/{productId}/pic?api-version=2.0"形式

#### Step 5: バックエンドAPI（参考）を理解する

フロントエンドが呼び出すCatalog.APIのエンドポイントを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemByIdエンドポイントの実装 |
| 5-2 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | バックエンド側のCatalogItemエンティティ |

**主要処理フロー**:
- **36-40行目（CatalogApi.cs）**: `/items/{id:int}`エンドポイント定義
- **171-191行目（CatalogApi.cs）**: GetItemByIdメソッド
  - **176-180行目**: ID検証（0以下はBadRequest）
  - **183行目**: CatalogItemsテーブルからInclude(CatalogBrand)でEager Loading
  - **185-188行目**: null時はNotFound返却
  - **190行目**: Ok(item)で商品情報返却
- **46-50行目（CatalogApi.cs）**: `/items/{id:int}/pic`エンドポイント（画像取得）
- **205-224行目（CatalogApi.cs）**: GetItemPictureByIdメソッド
  - **210行目**: CatalogItemsから商品取得
  - **217行目**: ファイルパス構築
  - **223行目**: PhysicalFileで画像ファイル返却

### プログラム呼び出し階層図

```
MauiProgram.CreateMauiApp()
    │
    ├─ builder.Services.AddHttpClient<CatalogService>()
    │      └─ HttpClient with BaseAddress = MobileBffHost
    │
    └─ builder.Services.AddSingleton<IProductImageUrlProvider, ProductImageUrlProvider>()

MainPage.xaml (BlazorWebView)
    │
    └─ Routes.razor (Router)
           │
           └─ ItemPage.razor (/item/{itemId:int})
                  │
                  ├─ OnInitializedAsync()
                  │      │
                  │      └─ CatalogService.GetCatalogItem(itemId)
                  │             │
                  │             └─ HttpClient.GetFromJsonAsync()
                  │                    │
                  │                    └─ GET api/catalog/items/{id}?api-version=2.0
                  │                           │
                  │                           └─ [Catalog.API] CatalogApi.GetItemById()
                  │                                  └─ CatalogContext.CatalogItems.Include(CatalogBrand)
                  │
                  └─ ProductImages.GetProductImageUrl(item.Id)
                         │
                         └─ {MobileBffHost}api/catalog/items/{id}/pic?api-version=2.0
```

### データフロー図

```
[入力]                    [処理]                           [出力]

URL                       ItemPage.razor                   UI画面
/item/{itemId}  ────────▶ [ルートパラメータ解析]
                                │
                                ▼
                          OnInitializedAsync()
                                │
                                ▼
                          CatalogService
                          .GetCatalogItem()
                                │
                          ┌─────▼─────┐
HttpClient ──────────────▶ HTTP GET  │
                          │ Request  │
                          └─────┬────┘
                                │
                                ▼
                          [Catalog.API]
                          CatalogApi
                          .GetItemById()
                                │
                          ┌─────▼─────┐
                          │ PostgreSQL │
                          │ CatalogItems│
                          │ CatalogBrands│
                          └─────┬─────┘
                                │
                          ┌─────▼─────┐
CatalogItem ◀────────────│ JSON      │
                          │ Response  │
                          └───────────┘
                                │
                                ▼
                          [StateHasChanged]
                                │
                          ┌─────▼─────┐
                          │ Blazor    │────────────▶ 商品詳細画面
                          │ Rendering │              - 商品名
                          └───────────┘              - 説明
                                │                    - 価格
                                │                    - ブランド
ProductImageUrl ◀───────────────┘                    - 商品画像
{Host}/api/catalog/items/{id}/pic
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ItemPage.razor | `src/HybridApp/Components/Pages/Item/ItemPage.razor` | ソース | 商品詳細表示のメインコンポーネント |
| CatalogService.cs | `src/HybridApp/Services/CatalogService.cs` | ソース | Catalog.APIとの通信サービス |
| ProductImageUrlProvider.cs | `src/HybridApp/Services/ProductImageUrlProvider.cs` | ソース | 商品画像URL生成サービス |
| CatalogJsonContext.cs | `src/HybridApp/Services/CatalogJsonContext.cs` | ソース | JSONシリアライゼーションコンテキスト |
| MauiProgram.cs | `src/HybridApp/MauiProgram.cs` | ソース | アプリケーション設定・DI登録 |
| MainPage.xaml | `src/HybridApp/MainPage.xaml` | XAML | BlazorWebViewのホストページ |
| MainPage.xaml.cs | `src/HybridApp/MainPage.xaml.cs` | ソース | MainPageのコードビハインド |
| Routes.razor | `src/HybridApp/Components/Routes.razor` | ソース | Blazorルーター設定 |
| MainLayout.razor | `src/HybridApp/Components/Layout/MainLayout.razor` | ソース | 共通レイアウトコンポーネント |
| HeaderBar.razor | `src/HybridApp/Components/Layout/HeaderBar.razor` | ソース | ヘッダーコンポーネント |
| _Imports.razor | `src/HybridApp/Components/_Imports.razor` | ソース | グローバルusingディレクティブ |
| CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | ソース（共有） | 商品データモデル（record型） |
| ICatalogService.cs | `src/WebAppComponents/Services/ICatalogService.cs` | ソース（共有） | カタログサービスインターフェース |
| IProductImageUrlProvider.cs | `src/WebAppComponents/Services/IProductImageUrlProvider.cs` | ソース（共有） | 画像URLプロバイダーインターフェース |
| ItemHelper.cs | `src/WebAppComponents/Item/ItemHelper.cs` | ソース（共有） | 商品URLヘルパー |
| CatalogListItem.razor | `src/WebAppComponents/Catalog/CatalogListItem.razor` | ソース（共有） | カタログ一覧アイテムコンポーネント（遷移元） |
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース（API） | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース（API） | APIエンティティモデル |
| HybridApp.csproj | `src/HybridApp/HybridApp.csproj` | 設定 | プロジェクト設定・依存関係 |
