# 機能設計書 7-タイプ別商品取得

## 概要

本ドキュメントは、eShopアプリケーションにおけるタイプ別商品取得機能の設計仕様を定義する。商品タイプIDとオプションのブランドIDを指定して、該当する商品をページネーション付きで取得するREST APIエンドポイントを提供する。

### 本機能の処理概要

**業務上の目的・背景**：ECサイトにおいて、ユーザーが特定カテゴリ（タイプ）の商品を閲覧したい場合に、効率的に商品を絞り込んで表示する機能である。さらにブランドで絞り込むことで、より細かなフィルタリングが可能になる。

**機能の利用シーン**：カタログ画面でのタイプ別フィルタリング、カテゴリナビゲーションからの商品一覧表示、ブランド×タイプの複合フィルタなど、商品をカテゴリ別に閲覧する場面で利用される。

**主要な処理内容**：
1. パスパラメータからタイプIDとオプションのブランドIDを取得
2. 内部でGetAllItemsメソッドを呼び出してフィルタリング
3. ページネーション付きで商品一覧を返却

**関連システム・外部連携**：PostgreSQLデータベースからEntity Framework Coreを通じてデータを取得する。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | カタログ画面 | 参照画面 | タイプフィルタリングで使用 |

## 機能種別

CRUD操作（Read）/ フィルタ検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| typeId | int | Yes | 商品タイプID | パスパラメータとして必須 |
| brandId | int? | No | ブランドID（オプション） | nullの場合はブランドフィルタなし |
| PageSize | int | No | 1ページあたりの取得件数 | デフォルト値: 10 |
| PageIndex | int | No | 取得するページのインデックス | デフォルト値: 0 |

### 入力データソース

- URLパスパラメータ + クエリパラメータ: `/api/catalog/items/type/{typeId}/brand/{brandId}?pageSize={pageSize}&pageIndex={pageIndex}`
- brandIdはオプショナルパスパラメータ

## 出力仕様

### 出力データ

**正常時（200 OK）: PaginatedItems&lt;CatalogItem&gt;**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PageIndex | int | 現在のページインデックス |
| PageSize | int | 1ページあたりの件数 |
| Count | long | 総件数（フィルタ条件適用後） |
| Data | IEnumerable&lt;CatalogItem&gt; | 商品データの配列 |

### 出力先

- HTTP Response Body（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ パスパラメータからtypeIdとbrandIdを取得

2. GetAllItemsメソッド呼び出し
   └─ typeId、brandIdパラメータを渡して内部委譲（nameはnull）

3. クエリ構築
   ├─ Where(c => c.CatalogTypeId == typeId)
   └─ brandId指定時: Where(c => c.CatalogBrandId == brandId)

4. データ取得
   ├─ LongCountAsyncで総件数取得
   └─ OrderBy/Skip/Take/ToListAsyncでページデータ取得

5. レスポンス生成
   └─ PaginatedItems<CatalogItem>を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[typeId取得]
    B --> C[brandId取得]
    C --> D[GetAllItems呼び出し]
    D --> E[CatalogTypeId条件構築]
    E --> F{brandId指定?}
    F -->|Yes| G[CatalogBrandId条件追加]
    F -->|No| H[総件数カウント]
    G --> H
    H --> I[名前順ソート]
    I --> J[Skip/Take適用]
    J --> K[データ取得]
    K --> L[PaginatedItems生成]
    L --> M[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | タイプフィルタ必須 | typeIdは必須パラメータとして指定 | 常時 |
| BR-002 | ブランドフィルタオプション | brandIdはnull許容、指定時のみフィルタ適用 | brandId指定時 |
| BR-003 | ソート順 | 商品名の昇順でソートする | 常時 |

### 計算ロジック

- 内部的にGetAllItemsメソッドを呼び出し、typeId/brandIdパラメータを指定（nameはnull）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| タイプ別検索 | Catalog | SELECT | タイプID/ブランドIDでフィルタして取得 |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE CatalogTypeId = {typeId} AND (CatalogBrandId = {brandId} OR brandId IS NULL) | Embeddingは除外（JsonIgnore） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | パラメータ形式が不正 | ProblemDetails形式でエラー詳細を返却 |
| 500 | InternalServerError | データベース接続エラー等 | サーバーエラーとして処理 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。

## パフォーマンス要件

- CatalogTypeId、CatalogBrandIdによるフィルタリングでインデックスを活用
- ページネーションにより大量検索結果でも効率的に処理

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- SQLインジェクション対策: Entity Framework CoreのLINQクエリによりパラメータ化

## 備考

- V1 APIのみで提供
- 存在しないtypeIdやbrandIdを指定した場合は空の結果が返却される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | CatalogTypeId、CatalogBrandIdプロパティ |
| 1-2 | CatalogType.cs | `src/Catalog.API/Model/CatalogType.cs` | タイプエンティティの構造 |
| 1-3 | CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ブランドエンティティの構造 |

**読解のコツ**: CatalogItemにはCatalogTypeId（int）とCatalogBrandId（int）の外部キーがあり、それぞれナビゲーションプロパティも定義されている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemsByBrandAndTypeIdメソッド |

**主要処理フロー**:
1. **67-71行目**: `/items/type/{typeId}/brand/{brandId?}` エンドポイント定義（V1のみ）
2. **291-299行目**: `GetItemsByBrandAndTypeId` メソッド実装
3. **298行目**: GetAllItemsへの委譲（name=null, type=typeId, brand=brandId）

#### Step 3: 内部実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetAllItemsメソッドのtype/brandフィルタ |

**主要処理フロー**:
- **140-143行目**: type指定時のCatalogTypeId条件追加
- **144-147行目**: brand指定時のCatalogBrandId条件追加

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items/type/{typeId}/brand/{brandId})
    │
    └─ CatalogApi.GetItemsByBrandAndTypeId()
           │
           └─ CatalogApi.GetAllItems(name=null, type=typeId, brand=brandId)
                  │
                  └─ CatalogServices.Context (CatalogContext)
                         └─ DbSet<CatalogItem>.CatalogItems
                                ├─ Where(c => c.CatalogTypeId == typeId)
                                ├─ Where(c => c.CatalogBrandId == brandId) [brandId指定時]
                                ├─ LongCountAsync()
                                ├─ OrderBy(c => c.Name)
                                ├─ Skip(pageSize * pageIndex)
                                ├─ Take(pageSize)
                                └─ ToListAsync()
                                       │
                                       └─ PaginatedItems<CatalogItem>
```

### データフロー図

```
[入力]                       [処理]                              [出力]

HTTPリクエスト               GetItemsByBrandAndTypeId()          HTTPレスポンス
  │                              │                                  │
  ├─ typeId ───────────────────▶│                                  │
  ├─ brandId (optional) ───────▶│   GetAllItems()                  │
  ├─ PageSize ─────────────────▶│      │                           │
  └─ PageIndex ────────────────▶│      ▼                           │
                                 │   PostgreSQL                     │
                                 │   WHERE CatalogTypeId = x        │
                                 │   AND CatalogBrandId = y         │
                                 │      │                           │
                                 │      ▼                           │
                                 └─▶ PaginatedItems ───────────────▶ JSON
                                      ├─ PageIndex
                                      ├─ PageSize
                                      ├─ Count
                                      └─ Data[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティ（外部キー含む） |
| CatalogType.cs | `src/Catalog.API/Model/CatalogType.cs` | ソース | タイプエンティティ |
| CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ソース | ブランドエンティティ |
| PaginatedItems.cs | `src/Catalog.API/Model/PaginatedItems.cs` | ソース | ページネーション結果モデル |
