# 機能設計書 8-ブランド別商品取得

## 概要

本ドキュメントは、eShopアプリケーションにおけるブランド別商品取得機能の設計仕様を定義する。ブランドIDのみを指定して、該当するブランドの商品をページネーション付きで取得するREST APIエンドポイントを提供する。タイプによるフィルタリングは行わず、全タイプの商品を対象とする。

### 本機能の処理概要

**業務上の目的・背景**：ECサイトにおいて、ユーザーが特定ブランドの商品を閲覧したい場合に、効率的に商品を絞り込んで表示する機能である。ブランドへの関心が高いユーザーに対して、ブランド単位での商品閲覧体験を提供する。

**機能の利用シーン**：カタログ画面でのブランド別フィルタリング、ブランドページからの商品一覧表示、ブランド検索結果の表示など、特定ブランドの商品を閲覧する場面で利用される。

**主要な処理内容**：
1. パスパラメータからブランドIDを取得
2. 内部でGetAllItemsメソッドを呼び出してブランドフィルタリング
3. ページネーション付きで商品一覧を返却

**関連システム・外部連携**：PostgreSQLデータベースからEntity Framework Coreを通じてデータを取得する。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | カタログ画面 | 参照画面 | ブランドフィルタリングで使用 |

## 機能種別

CRUD操作（Read）/ フィルタ検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| brandId | int? | No | ブランドID（オプション） | nullの場合は全ブランド対象 |
| PageSize | int | No | 1ページあたりの取得件数 | デフォルト値: 10 |
| PageIndex | int | No | 取得するページのインデックス | デフォルト値: 0 |

### 入力データソース

- URLパスパラメータ + クエリパラメータ: `/api/catalog/items/type/all/brand/{brandId}?pageSize={pageSize}&pageIndex={pageIndex}`

## 出力仕様

### 出力データ

**正常時（200 OK）: PaginatedItems&lt;CatalogItem&gt;**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PageIndex | int | 現在のページインデックス |
| PageSize | int | 1ページあたりの件数 |
| Count | long | 総件数（フィルタ条件適用後） |
| Data | IEnumerable&lt;CatalogItem&gt; | 商品データの配列 |

### 出力先

- HTTP Response Body（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ パスパラメータからbrandIdを取得

2. GetAllItemsメソッド呼び出し
   └─ brandIdパラメータのみを渡して内部委譲（name=null, type=null）

3. クエリ構築
   └─ brandId指定時: Where(c => c.CatalogBrandId == brandId)

4. データ取得
   ├─ LongCountAsyncで総件数取得
   └─ OrderBy/Skip/Take/ToListAsyncでページデータ取得

5. レスポンス生成
   └─ PaginatedItems<CatalogItem>を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[brandId取得]
    B --> C[GetAllItems呼び出し]
    C --> D{brandId指定?}
    D -->|Yes| E[CatalogBrandId条件構築]
    D -->|No| F[フィルタなし]
    E --> G[総件数カウント]
    F --> G
    G --> H[名前順ソート]
    H --> I[Skip/Take適用]
    I --> J[データ取得]
    J --> K[PaginatedItems生成]
    K --> L[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | タイプフィルタなし | パス `/type/all/` によりタイプフィルタは適用しない | 常時 |
| BR-002 | ブランドフィルタオプション | brandIdはnull許容、指定時のみフィルタ適用 | brandId指定時 |
| BR-003 | ソート順 | 商品名の昇順でソートする | 常時 |

### 計算ロジック

- 内部的にGetAllItemsメソッドを呼び出し、brandIdパラメータのみを指定（name=null, type=null）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ブランド別検索 | Catalog | SELECT | ブランドIDでフィルタして取得 |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE CatalogBrandId = {brandId} (brandId指定時) | Embeddingは除外（JsonIgnore） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | パラメータ形式が不正 | ProblemDetails形式でエラー詳細を返却 |
| 500 | InternalServerError | データベース接続エラー等 | サーバーエラーとして処理 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。

## パフォーマンス要件

- CatalogBrandIdによるフィルタリングでインデックスを活用
- ページネーションにより大量検索結果でも効率的に処理

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- SQLインジェクション対策: Entity Framework CoreのLINQクエリによりパラメータ化

## 備考

- V1 APIのみで提供
- パスに `/type/all/` を含むことで、タイプフィルタなしであることを明示
- 存在しないbrandIdを指定した場合は空の結果が返却される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | CatalogBrandIdプロパティ |
| 1-2 | CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ブランドエンティティの構造 |

**読解のコツ**: CatalogItemにはCatalogBrandId（int）の外部キーがあり、ナビゲーションプロパティCatalogBrandも定義されている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemsByBrandIdメソッド |

**主要処理フロー**:
1. **72-76行目**: `/items/type/all/brand/{brandId:int?}` エンドポイント定義（V1のみ）
2. **301-308行目**: `GetItemsByBrandId` メソッド実装
3. **307行目**: GetAllItemsへの委譲（name=null, type=null, brand=brandId）

#### Step 3: 内部実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetAllItemsメソッドのbrandフィルタ |

**主要処理フロー**:
- **144-147行目**: brand指定時のCatalogBrandId条件追加

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items/type/all/brand/{brandId})
    │
    └─ CatalogApi.GetItemsByBrandId()
           │
           └─ CatalogApi.GetAllItems(name=null, type=null, brand=brandId)
                  │
                  └─ CatalogServices.Context (CatalogContext)
                         └─ DbSet<CatalogItem>.CatalogItems
                                ├─ Where(c => c.CatalogBrandId == brandId) [brandId指定時]
                                ├─ LongCountAsync()
                                ├─ OrderBy(c => c.Name)
                                ├─ Skip(pageSize * pageIndex)
                                ├─ Take(pageSize)
                                └─ ToListAsync()
                                       │
                                       └─ PaginatedItems<CatalogItem>
```

### データフロー図

```
[入力]                       [処理]                              [出力]

HTTPリクエスト               GetItemsByBrandId()                 HTTPレスポンス
  │                              │                                  │
  ├─ brandId (optional) ───────▶│   GetAllItems()                  │
  ├─ PageSize ─────────────────▶│      │                           │
  └─ PageIndex ────────────────▶│      ▼                           │
                                 │   PostgreSQL                     │
                                 │   WHERE CatalogBrandId = x       │
                                 │      │                           │
                                 │      ▼                           │
                                 └─▶ PaginatedItems ───────────────▶ JSON
                                      ├─ PageIndex
                                      ├─ PageSize
                                      ├─ Count
                                      └─ Data[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティ（外部キー含む） |
| CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ソース | ブランドエンティティ |
| PaginatedItems.cs | `src/Catalog.API/Model/PaginatedItems.cs` | ソース | ページネーション結果モデル |
