# 画面設計書 26-商品詳細画面

## 概要

本ドキュメントは、HybridApp（MAUI Blazor Hybrid）の商品詳細画面の設計仕様を記載する。

### 本画面の処理概要

本画面は、eShopシステムのMAUI Hybrid版アプリケーションにおける個別商品の詳細情報を表示する画面である。ユーザーはこの画面を通じて、商品名、価格、説明文、ブランド情報、商品画像を確認することができる。WebAppの商品詳細画面と同等の表示機能を提供するが、カート追加機能は未実装である。

**業務上の目的・背景**：商品カタログから特定の商品を選択した際に、その商品の詳細情報を閲覧するための画面が必要である。この画面は、ユーザーが購入検討のために商品の詳細な情報（説明、価格、ブランド）を確認する業務課題を解決する。MAUI Hybrid版では読み取り専用の閲覧機能を提供し、将来的なカート機能の実装基盤となる。

**画面へのアクセス方法**：カタログ画面の商品カード（CatalogListItem）をクリックすることでアクセスする。URLは/item/{itemId}で、itemIdパラメータで表示する商品を指定する。

**主要な操作・処理内容**：
1. 商品詳細情報の取得と表示
2. 商品画像の表示
3. ブランド情報の表示
4. 価格の表示

**画面遷移**：
- 遷移元：カタログ画面（/）のCatalogListItem
- 遷移先：なし（閲覧専用画面）

**権限による表示制御**：HybridAppは認証機能が未実装のため、権限による表示制御はない。すべてのユーザーが商品詳細を閲覧できる。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品詳細取得 | 主機能 | CatalogService.GetCatalogItemで商品詳細情報を取得 |
| 4 | 商品画像取得 | 補助機能 | IProductImageUrlProviderで商品画像URLを取得 |
| 61 | HybridApp商品詳細 | 主機能 | MAUI Blazor Hybrid用商品詳細表示機能 |

## 画面種別

詳細

## URL/ルーティング

| 項目 | 値 |
|------|-----|
| URL | `/item/{itemId:int}` |
| ルート定義 | `@page "/item/{itemId:int}"` |
| 認証要件 | 不要 |

## 入出力項目

### ルートパラメータ（入力）

| 項目名 | 論理名 | データ型 | 必須 | 説明 |
|--------|--------|----------|------|------|
| itemId | 商品ID | int | はい | 表示する商品のID |

## 表示項目

### 商品情報（正常時）

| 項目名 | 論理名 | データ型 | 説明 |
|--------|--------|----------|------|
| item.Name | 商品名 | string | ヘッダータイトルおよびPageTitleに表示 |
| item.CatalogBrand?.Brand | ブランド名 | string | ヘッダーサブタイトルおよび詳細欄に表示 |
| item.Description | 商品説明 | string | 商品の詳細説明文 |
| item.Price | 価格 | decimal | 商品価格（$表記、小数点以下2桁） |
| ProductImages.GetProductImageUrl(item.Id) | 商品画像 | string | 商品画像のURL |

### エラー時（notFound）

| 項目名 | 論理名 | 値 | 説明 |
|--------|--------|-----|------|
| page-header-title | ヘッダータイトル | "Not found" | 商品が見つからない場合 |
| メッセージ | エラーメッセージ | "Sorry, we couldn't find any such product." | 商品が見つからない場合 |

## イベント仕様

### 1-ページ初期化（OnInitializedAsync）

**トリガー**：ページアクセス時

**処理フロー**：
1. ItemIdパラメータから商品IDを取得
2. CatalogService.GetCatalogItemを呼び出し
3. 正常取得：itemプロパティに商品データを設定
4. 404エラー：notFoundフラグをtrueに設定

**データの流れ**：
```
ItemId → CatalogService.GetCatalogItem(ItemId)
              │
              ▼
       GET /api/catalog/items/{id} (Mobile BFF)
              │
              ├─ 成功 → item = response
              │
              └─ 404 → notFound = true
```

### 2-商品画像表示

**トリガー**：画面描画時（itemがnullでない場合）

**処理フロー**：
1. IProductImageUrlProvider.GetProductImageUrlで画像URLを生成
2. img要素のsrc属性に設定

**データの流れ**：
```
item.Id → ProductImages.GetProductImageUrl(item.Id)
               │
               ▼
        {MobileBffHost}/api/catalog/items/{id}/pic?api-version=2.0
```

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| ページ表示 | Catalog | SELECT | 商品詳細取得（API経由） |

### 備考

HybridAppアプリケーションはデータベースに直接アクセスせず、Mobile BFF（localhost:11632）のAPIを通じてデータを取得する。本画面は読み取り専用であり、データベースへの書き込み操作は発生しない。

## メッセージ仕様

| 種別 | メッセージ | 表示条件 |
|------|----------|----------|
| エラー | Sorry, we couldn't find any such product. | 404エラー（商品未存在）の場合 |

## 例外処理

| 例外種別 | 発生条件 | 対処 |
|---------|---------|------|
| HttpRequestException (404) | 指定IDの商品が存在しない | notFoundフラグをtrueにしてエラーメッセージを表示 |
| HttpRequestException (その他) | Mobile BFFへの接続失敗 | 例外がスローされる |
| JsonException | APIレスポンスのパースエラー | 例外がスローされる |

## 備考

- カート追加機能は未実装（コメントアウトされている）
- BasketState関連のコードはコメントアウトされており、将来的な実装を想定
- WebAppの商品詳細画面と異なり、ログイン状態の確認やカート内数量表示は行わない
- 商品が見つからない場合の404ステータスコード設定もコメントアウトされている（MAUI環境ではHttpContextが使用できないため）

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、画面で扱うデータモデルの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | 商品データ構造（Id, Name, Price, Description, CatalogBrand等） |

**読解のコツ**: CatalogItemレコードには関連するCatalogBrandが含まれていることを確認する。

#### Step 2: エントリーポイントを理解する

処理の起点となるRazorページを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ItemPage.razor | `src/HybridApp/Components/Pages/Item/ItemPage.razor` | ページ構成とデータ取得ロジック |

**主要処理フロー**:
1. **1行目**: @page "/item/{itemId:int}"でルート定義
2. **3行目**: CatalogServiceの依存性注入
3. **6行目**: IProductImageUrlProviderの依存性注入
4. **8-31行目**: 正常表示時のレイアウト
5. **32-38行目**: 404エラー時のレイアウト
6. **52-65行目**: OnInitializedAsyncでデータ取得と例外処理

#### Step 3: APIクライアントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogService.cs | `src/HybridApp/Services/CatalogService.cs` | GetCatalogItemメソッドの実装 |

**主要処理フロー**:
- **13-17行目**: GetCatalogItemでGET /api/catalog/items/{id}を呼び出し

#### Step 4: 画像URL生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ProductImageUrlProvider.cs | `src/HybridApp/Services/ProductImageUrlProvider.cs` | GetProductImageUrlの実装 |

**主要処理フロー**:
- **7-8行目**: MobileBffHostをベースに画像URLを構築

### プログラム呼び出し階層図

```
ItemPage.razor (/item/{itemId})
    │
    ├─ OnInitializedAsync()
    │      │
    │      └─ CatalogService.GetCatalogItem(ItemId)
    │             │
    │             └─ GET /api/catalog/items/{id}?api-version=2.0 (Mobile BFF)
    │                    │
    │                    ├─ 成功 → item = response
    │                    │
    │                    └─ 404 → HttpRequestException → notFound = true
    │
    └─ 画面描画
           │
           ├─ item != null
           │      │
           │      ├─ PageTitle: {item.Name} | AdventureWorks
           │      ├─ page-header-title: {item.Name}
           │      ├─ page-header-subtitle: {item.CatalogBrand?.Brand}
           │      │
           │      └─ item-details
           │             ├─ img: ProductImages.GetProductImageUrl(item.Id)
           │             ├─ Description: {item.Description}
           │             ├─ Brand: {item.CatalogBrand?.Brand}
           │             └─ Price: ${item.Price}
           │
           └─ notFound
                  │
                  ├─ page-header-title: "Not found"
                  │
                  └─ item-details: "Sorry, we couldn't find any such product."
```

### データフロー図

```
[入力]                     [処理]                           [出力]

ItemId ─────────────────▶ OnInitializedAsync
(ルートパラメータ)               │
                                ▼
                    CatalogService.GetCatalogItem(ItemId)
                                │
                                ▼
                    GET /api/catalog/items/{id} (Mobile BFF)
                                │
                    ┌───────────┴───────────┐
                    ▼                       ▼
               成功: CatalogItem        失敗: 404
                    │                       │
                    ▼                       ▼
               item = data             notFound = true
                    │                       │
                    ▼                       ▼
               商品詳細表示            エラーメッセージ表示
                    │
                    ▼
          ProductImages.GetProductImageUrl(item.Id)
                    │
                    ▼
          {MobileBffHost}/api/catalog/items/{id}/pic
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ItemPage.razor | `src/HybridApp/Components/Pages/Item/ItemPage.razor` | ソース | 商品詳細画面メインページ |
| CatalogService.cs | `src/HybridApp/Services/CatalogService.cs` | ソース | APIクライアント |
| CatalogItem.cs | `src/WebAppComponents/Catalog/CatalogItem.cs` | ソース | データモデル定義 |
| ProductImageUrlProvider.cs | `src/HybridApp/Services/ProductImageUrlProvider.cs` | ソース | 画像URL生成 |
| MauiProgram.cs | `src/HybridApp/MauiProgram.cs` | ソース | アプリケーション起動・DI設定 |
| MainLayout.razor | `src/HybridApp/Components/Layout/MainLayout.razor` | ソース | レイアウト定義 |
