# 画面設計書 8-チャットボット

## 概要

本ドキュメントは、eShop WebAppのAIチャットボット機能（.NET Concierge）の設計仕様を定義する。フローティングパネルとして表示され、商品検索やカート操作の支援を行うAIアシスタント機能を提供するコンポーネントである。

### 本画面の処理概要

チャットボットは、Azure OpenAI（または互換サービス）を使用したAIアシスタント機能を提供するフローティングコンポーネントである。

**業務上の目的・背景**：ECサイトにおいて、顧客が商品を探したりカート操作を行う際のサポートを提供する。AIによる自然言語での対話を通じて、商品検索、カートへの追加、カート内容の確認などの操作を支援し、ユーザー体験を向上させる。セマンティック検索機能により、あいまいな商品説明からも関連商品を見つけることができる。

**画面へのアクセス方法**：URLのクエリパラメータ（?chat=true等）によりチャットボットが表示される。フローティングパネルとして他の画面コンテンツの上に重畳表示される。閉じるボタン（X）で非表示にできる。

**主要な操作・処理内容**：
1. チャット対話：テキストボックスにメッセージを入力し、AIからの応答を受け取る
2. 商品検索支援：自然言語での商品説明からセマンティック検索で商品を見つける
3. カートへの追加：AIを通じて商品をカートに追加する（ログイン必須）
4. カート内容確認：現在のカート内容をAIに問い合わせる
5. ユーザー情報参照：AIがユーザーのプロフィール情報を参照して応答する

**画面遷移**：
- 遷移元：任意の画面（クエリパラメータ?chatにより表示）
- 遷移先：なし（オーバーレイコンポーネントとして動作）

**権限による表示制御**：チャットボット自体は認証不要で表示されるが、カートへの追加操作は認証が必要。未認証時にカート追加を試みると、AIが認証が必要である旨を応答する。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | 商品一覧取得 | API連携 | CatalogServiceで商品検索・一覧取得 |
| 6 | AI商品検索 | API連携 | ChatStateでセマンティック検索による商品推薦 |
| 15 | かご更新 | API連携 | BasketStateでカート操作支援 |
| 46 | AIチャットボット | 主機能 | Azure OpenAI使用のチャット機能 |

## 画面種別

フローティングパネル（コンポーネント）

## URL/ルーティング

| 項目 | 値 |
|------|-----|
| トリガー | クエリパラメータ `?chat` |
| コンポーネント | フローティングオーバーレイ |
| 認証 | 不要（ただしカート操作は認証必須） |

## 入出力項目

### 入力項目

| 項目名 | 要素 | 型 | 必須 | 説明 |
|--------|-----|-----|------|------|
| メッセージ | textarea | string | 必須 | ユーザーからの質問・指示 |

### 出力項目

| 項目名 | 型 | 説明 |
|--------|-----|------|
| チャット履歴 | IList\<ChatMessage\> | ユーザーとAIのメッセージ履歴 |
| AI応答 | string | AIからの応答テキスト（HTML含む可能性あり） |

## 表示項目

### チャットパネル

| 項目名 | 条件 | 表示形式 | 備考 |
|--------|------|---------|------|
| 閉じるボタン | 常時 | X印リンク | クエリパラメータを削除してチャットを閉じる |
| メッセージ一覧 | chatState存在 | p.message要素 | ロール別にCSSクラス適用 |
| 設定エラー | missingConfiguration | 警告メッセージ | OpenAI設定が不足している旨を表示 |
| 思考中表示 | thinking | "Thinking..." | AI応答待機中 |

### 入力フォーム

| 項目名 | 要素 | 備考 |
|--------|------|------|
| メッセージ入力 | textarea | placeholder="Start chatting..." |
| 送信ボタン | button | chatStateがnullの場合は無効化 |

## イベント仕様

### 1-メッセージ送信

テキストボックスにメッセージを入力し送信すると、AIが応答する。

**トリガー**: フォームsubmit、またはEnterキー押下
**処理内容**:
1. SendMessageAsyncメソッドが呼び出される
2. メッセージをトリム処理、入力欄をクリア
3. thinking=trueに設定（「Thinking...」表示）
4. chatState.AddUserMessageAsync(message, onMessageAdded)を呼び出し
5. AIからの応答を受信後、thinking=falseに設定
6. StateHasChanged()で画面更新

**データの流れ**:
```
ユーザー → メッセージ入力
                ↓
        SendMessageAsync
                ↓
        ChatState.AddUserMessageAsync
                ├─ Messages.Add(user message)
                ├─ _chatClient.GetResponseAsync
                │      └─ (Tool呼び出しの可能性)
                │             ├─ GetUserInfo
                │             ├─ SearchCatalog
                │             ├─ AddToCart
                │             └─ GetCartContents
                └─ Messages.AddMessages(response)
                ↓
        画面更新（StateHasChanged）
```

### 2-商品検索（AI Tool）

AIが商品検索が必要と判断した場合、SearchCatalogツールが呼び出される。

**トリガー**: AIによるTool呼び出し
**処理内容**:
1. SearchCatalog(productDescription)が呼び出される
2. CatalogService.GetCatalogItemsWithSemanticRelevanceでセマンティック検索
3. 商品画像URLを付与
4. JSON形式で結果を返却
5. AIが結果をもとに自然言語で応答を生成

### 3-カート追加（AI Tool）

AIがカート追加を実行する場合、AddToCartツールが呼び出される。

**トリガー**: AIによるTool呼び出し
**処理内容**:
1. AddToCart(itemId)が呼び出される
2. CatalogService.GetCatalogItem(itemId)で商品取得
3. BasketState.AddAsync(item)でカートに追加
4. 成功メッセージまたはエラーメッセージを返却
5. 未認証時は「You must be logged in.」を返却

### 4-カート内容確認（AI Tool）

AIがカート内容を確認する場合、GetCartContentsツールが呼び出される。

**トリガー**: AIによるTool呼び出し
**処理内容**:
1. GetCartContents()が呼び出される
2. BasketState.GetBasketItemsAsync()でカート内容取得
3. JSON形式で結果を返却
4. AIが結果をもとに自然言語で応答を生成

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| 商品検索 | Catalog | SELECT | セマンティック検索で商品取得 |
| カート追加 | Basket | UPDATE | カートに商品を追加（Redis経由） |
| カート確認 | Basket | SELECT | カート内容を取得（Redis経由） |

### テーブル別更新項目詳細

#### Catalog（参照のみ）

| 操作 | 項目（カラム名） | 取得条件 | 備考 |
|-----|-----------------|---------|------|
| SELECT | Id, Name, Description, Price, PictureUrl等 | セマンティック検索条件 | 最大8件取得 |

## メッセージ仕様

| メッセージ種別 | 条件 | メッセージ内容 |
|--------------|------|--------------|
| 初期メッセージ | 常時 | "Hi! I'm the AdventureWorks Concierge. How can I help?" |
| 設定エラー | missingConfiguration | "The chatbot is missing required configuration. Please set 'useOpenAI = true'..." |
| 思考中 | thinking | "Thinking..." |
| AI応答エラー | 例外発生 | "My apologies, but I encountered an unexpected error." |
| カート追加成功 | AddToCart成功 | "Item added to shopping cart." |
| 認証エラー | 未認証時のカート操作 | "Unable to add an item to the cart. You must be logged in." |

## 例外処理

| 例外条件 | 挙動 | 備考 |
|---------|------|------|
| IChatClient未設定 | missingConfiguration=true、設定エラーメッセージ表示 | OpenAI未設定時 |
| AI応答エラー | エラーメッセージをチャットに追加 | ログ出力あり |
| 商品検索エラー | "Error accessing catalog."をAIに返却 | HTTP例外 |
| カート追加認証エラー | "You must be logged in."をAIに返却 | gRPC Unauthenticatedエラー |
| その他カート操作エラー | エラーメッセージをAIに返却 | ログ出力あり |

## 備考

- InteractiveServerモード（prerender: false）で動作し、サーバーサイドで状態を保持
- Microsoft.Extensions.AIライブラリを使用したIChatClientインターフェースで抽象化
- AIFunctionFactoryでツール関数を定義し、AIが必要に応じて呼び出す
- システムプロンプトでAIの振る舞いを定義（AdventureWorks専用、簡潔な応答等）
- Enterキーでメッセージ送信可能（submitOnEnter JavaScriptモジュール）
- メッセージ送信後に自動スクロール（scrollToEnd JavaScriptモジュール）
- MessageProcessorでAI応答内の画像を許可（AllowImages）

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、チャットで扱うデータの構造を把握することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Microsoft.Extensions.AI | (ライブラリ) | ChatMessage、ChatRole、IChatClientの概念理解 |

**読解のコツ**: Microsoft.Extensions.AIはAI統合のための標準インターフェースを提供。ChatMessageはRole（User/Assistant/System）とTextを持つ。

#### Step 2: エントリーポイントを理解する

処理の起点となるRazorコンポーネントを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Chatbot.razor | `src/WebApp/Components/Chatbot/Chatbot.razor` | InteractiveServerモード、ChatState初期化、メッセージ送信処理 |

**主要処理フロー**:
1. **行1**: `@rendermode @(new InteractiveServerRenderMode(prerender: false))` - プリレンダリング無効のサーバーモード
2. **行54-66**: `OnInitializedAsync` - IChatClient取得、ChatState初期化
3. **行68-79**: `SendMessageAsync` - ユーザーメッセージ送信、AI応答取得
4. **行81-91**: `OnAfterRenderAsync` - JSモジュール読み込み、自動スクロール、Enterキー設定

#### Step 3: チャット状態管理を理解する

ChatStateによる会話管理とAIツールの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ChatState.cs | `src/WebApp/Components/Chatbot/ChatState.cs` | ツール定義、システムプロンプト、メッセージ処理 |

**主要処理フロー**:
- **行19-65**: コンストラクタ - サービス注入、ツール登録、システムプロンプト設定
- **行41-47**: `_chatOptions.Tools` - GetUserInfo、SearchCatalog、AddToCart、GetCartContentsを登録
- **行52-60**: システムプロンプト - AIの役割と制約を定義
- **行69-93**: `AddUserMessageAsync` - ユーザーメッセージ追加、AI応答取得
- **行117-134**: `SearchCatalog` - セマンティック検索ツール
- **行136-153**: `AddToCart` - カート追加ツール
- **行155-167**: `GetCartContents` - カート内容取得ツール

### プログラム呼び出し階層図

```
Chatbot.razor (コンポーネント)
    │
    ├─ OnInitializedAsync
    │      ├─ ServiceProvider.GetService<IChatClient>()
    │      └─ new ChatState(...)
    │
    ├─ SendMessageAsync
    │      └─ ChatState.AddUserMessageAsync
    │             ├─ Messages.Add(user message)
    │             └─ _chatClient.GetResponseAsync
    │                    │
    │                    └─ (Tool呼び出し)
    │                           ├─ GetUserInfo → User Claims
    │                           ├─ SearchCatalog → CatalogService
    │                           ├─ AddToCart → BasketState
    │                           └─ GetCartContents → BasketState
    │
    └─ OnAfterRenderAsync
           └─ JS.InvokeAsync (スクロール、Enterキー)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

                          Chatbot.razor
ユーザー入力 ─────────────┬─ SendMessageAsync
                          │      │
                          │      └─ ChatState.AddUserMessageAsync
                          │             │
                          │             ├─ ユーザーメッセージ追加
                          │             │
                          │             └─ IChatClient.GetResponseAsync
                          │                    │
                          │                    ├─ (ツール呼び出し)
                          │                    │      ├─ SearchCatalog ──▶ 商品リスト
                          │                    │      ├─ AddToCart ──────▶ カート更新
                          │                    │      └─ GetCartContents ▶ カート内容
                          │                    │
                          │                    └─ AI応答生成 ────────────▶ メッセージ表示
                          │
                          └─ 画面更新（StateHasChanged）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Chatbot.razor | `src/WebApp/Components/Chatbot/Chatbot.razor` | ソース | チャットボットUIコンポーネント |
| ChatState.cs | `src/WebApp/Components/Chatbot/ChatState.cs` | ソース | チャット状態管理、AIツール実装 |
| Chatbot.razor.js | `src/WebApp/Components/Chatbot/Chatbot.razor.js` | JavaScript | スクロール、Enterキー処理 |
| CatalogService.cs | `src/WebAppComponents/Services/CatalogService.cs` | ソース | 商品検索サービス |
| BasketState.cs | `src/WebApp/Services/BasketState.cs` | ソース | カート状態管理 |
