# セキュリティ設計書

## 概要

本ドキュメントは、Etherpad-liteのセキュリティアーキテクチャと実装について記載する。Etherpad-liteはリアルタイム共同編集エディタであり、認証・認可、通信暗号化、入力検証、セッション管理、アクセス制御、監査ログなど多層的なセキュリティ機構を実装している。

## 認証設計

### 認証方式

Etherpad-liteは以下の認証方式をサポートする：

| 認証方式 | 説明 | 設定 |
| --- | --- | --- |
| OAuth2/OIDC (SSO) | OpenID Connect準拠のSSOプロバイダー。デフォルト認証方式。RS256署名を使用 | `authenticationMethod: "sso"` |
| HTTPベーシック認証 | RFC 7617準拠のBase64エンコード認証。プラグインが認証を処理しない場合のフォールバック | `authenticationMethod: "apikey"` |
| APIキー認証 | 32文字のランダム文字列によるAPI認証。`APIKEY.txt`に保存 | API経由のアクセス時 |

**認証フロー（4ステップ）：**
1. **preAuthorize**: プラグインによる早期許可/拒否判定
2. **認可判定（認証前）**: 認証なし状態での認可チェック
3. **認証実行**: HTTPベーシック認証またはプラグイン認証
4. **認可判定（認証後）**: 認証後の権限チェック

**セキュリティ対策：**
- **ブルートフォース対策**: 認証失敗時に1秒の遅延を実装 (`authnFailureDelayMs: 1000`)
- **プロトタイプ汚染対策**: ユーザー名として`__proto__`、`constructor`、`prototype`を拒否
- **パスワード保護**: セッション保存時にパスワードを削除しログやDBに出現させない

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 10日間 (864,000,000ms) | `cookie.sessionLifetime`で設定可能 |
| セッションリフレッシュ間隔 | 1日間 (86,400,000ms) | `cookie.sessionRefreshInterval`で設定 |
| セッションID形式 | `s.` + 16文字ランダム文字列 | SessionManager.tsで生成 |
| クッキー名 | `express_sid` | Express-session標準 |
| SameSite属性 | `Lax` | CSRF対策。埋め込み時は`None`に変更可能 |
| Secure属性 | `auto` | HTTPS自動検出 |
| Rolling | `true` | アクティビティでセッション更新 |
| セッション固定化対策 | 秘密鍵ローテーション | 24時間ごとに鍵を自動ローテーション |

**秘密鍵ローテーション：**
- `SecretRotator`クラスにより定期的な鍵交換を実施
- HKDF-SHA256を使用した鍵導出
- 複数インスタンス間での秘密の同期をサポート
- レガシー互換のための`LegacyStaticSecret`もサポート

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| admin | create, modify, readOnly, is_admin | 管理者権限。/adminへのアクセス可能 |
| user | create, modify, readOnly | 一般ユーザー。パッドの作成・編集・閲覧 |
| guest | readOnly | 未認証ユーザー。閲覧のみ |

**ユーザープロパティ：**
- `password`: ユーザーパスワード
- `is_admin`: 管理者フラグ（`/admin`へのアクセス権）
- `readOnly`: 読み取り専用フラグ（`true`の場合、編集・作成不可）
- `canCreate`: パッド作成権限（デフォルト`true`）
- `padAuthorizations`: パッド単位の個別権限設定

### アクセス制御

**アクセス制御レベル：**

| レベル | 説明 | 条件 |
| --- | --- | --- |
| create | パッド新規作成可能 | `canCreate: true` かつ `readOnly: false` |
| modify | パッド編集可能 | `readOnly: false` |
| readOnly | 閲覧のみ | `readOnly: true` または読み取り専用パッドID |

**パッド種別によるアクセス制御：**

| パッド種別 | 識別方法 | アクセス条件 |
| --- | --- | --- |
| 通常パッド | IDに`$`を含まない | 設定に応じた認証/認可 |
| グループパッド（公開） | IDに`$`を含む、`publicStatus: true` | セッション不要 |
| グループパッド（非公開） | IDに`$`を含む、`publicStatus: false` | 有効なAPIセッション必須 |
| 読み取り専用パッド | `r.`で始まるID | 閲覧のみ、作成不可 |

**プラグインフック：**
- `preAuthorize`: 早期許可/拒否
- `authorize`: 認可判定
- `authenticate`: 認証処理
- `onAccessCheck`: アクセスチェック

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | Node.jsネイティブSSL/TLSサポート。鍵・証明書・CA証明書の設定可能 |
| HSTS | SSL有効時に`Strict-Transport-Security: max-age=31536000; includeSubDomains`を送信 |
| Referrer-Policy | `same-origin`でリファラー情報を制限 |
| X-UA-Compatible | `IE=Edge,chrome=1`でIE互換性モード防止 |

**SSL/TLS設定例：**
```json
{
  "ssl": {
    "key": "/path-to-your/epl-server.key",
    "cert": "/path-to-your/epl-server.crt",
    "ca": ["/path-to-your/epl-intermediate-cert1.crt"]
  }
}
```

**Trust Proxy設定：**
- リバースプロキシ経由の場合は`trustProxy: true`を設定
- SSL終端をプロキシで行う場合に必須（Secureクッキーの適切な設定のため）
- 実際のクライアントIPアドレスのログ記録に必要

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS 1.2以上（Node.js設定依存） |
| セッション署名鍵 | HKDF-SHA256で32バイト鍵を導出 |
| OAuth2トークン署名 | RS256 (RSA + SHA-256) |
| APIキー | 32文字のランダム文字列（`randomString(32)`） |
| セッションID | 16文字のランダム文字列（`randomString(16)`） |

### 機密情報管理

| 情報種別 | 保存場所 | 保護方式 |
| --- | --- | --- |
| ユーザーパスワード | settings.json / credentials.json | 平文（ep_hash_authプラグインでハッシュ化推奨） |
| APIキー | APIKEY.txt | ファイル権限で保護 |
| セッション署名鍵 | データベース | 自動ローテーション |
| OAuth2秘密鍵 | メモリ内生成 | 起動時に自動生成 |
| DBクレデンシャル | credentials.json | settings.jsonから分離可能 |

**環境変数による機密情報管理：**
- すべての設定値は環境変数で上書き可能
- 構文: `${ENV_VAR}` または `${ENV_VAR:default_value}`
- Docker等のコンテナ環境に適した設定方式

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Referrer-Policy、X-UA-Compatibleヘッダー設定。テンプレートエンジンによる自動エスケープ |
| SQLインジェクション | ueberdb2抽象化レイヤーによるパラメータ化クエリ |
| CSRF | SameSiteクッキー属性（`Lax`デフォルト）、Express-sessionによるセッション管理 |
| ディレクトリトラバーサル | パス名サニタイズ（絶対パス拒否、`..`検出） |
| プロトタイプ汚染 | ユーザー名検証（`__proto__`、`constructor`、`prototype`を拒否） |
| DoS (レート制限) | インポート/エクスポート：90秒間に10リクエスト、コミット：1秒間に10変更 |
| 大容量ファイル攻撃 | インポートファイルサイズ上限：50MB、Socket.IOバッファ上限：50KB |

**入力検証詳細：**

| 入力種別 | 検証内容 | 実装箇所 |
| --- | --- | --- |
| 著者トークン | `t.` + Base64url形式（RFC 4648準拠） | pad_utils.ts:378-390 |
| パッドID | isValidPadId()による検証 | PadManager.ts:111-125 |
| パッドテキスト | 文字列型、100,000文字以下 | PadManager.ts:115-120 |
| セッション有効期限 | 正の整数値、将来の日時 | SessionManager.ts:119-141 |
| パスワード | 空白・null値の明確な区別 | settings.json設定 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 認証ログ | 認証成功/失敗、IPアドレス、ユーザー名 | Log4js設定依存 |
| アクセス制御ログ | アクセス拒否理由、パッドID | Log4js設定依存 |
| Socket.IOログ | 接続/切断イベント | Log4js設定依存 |
| APIログ | API呼び出し | Log4js設定依存 |
| 秘密ローテーションログ | 鍵ローテーション操作 | Log4js設定依存 |

**ログカテゴリ：**
- `http`: HTTP認証ログ（成功/失敗）
- `security/auth`: セキュリティ関連ログ
- `socket.io`: WebSocket接続ログ
- `openapi`: API操作ログ
- `secret-rotation`: 秘密鍵ローテーションログ

**ログ設定：**
```json
{
  "loglevel": "INFO",
  "logLayoutType": "colored",
  "disableIPlogging": false
}
```

**プライバシー考慮：**
- `disableIPlogging: true`でIPアドレスのログ記録を無効化可能

## レート制限

| 対象 | 制限値 | 設定 |
| --- | --- | --- |
| インポート/エクスポート | 90秒間に10リクエスト/IP | `importExportRateLimiting` |
| パッドコミット | 1秒間に10変更/IP | `commitRateLimiting` |
| Socket.IOバッファ | 50,000バイト | `socketIo.maxHttpBufferSize` |

## OAuth2/OIDC設定

| 項目 | 設定値 |
| --- | --- |
| 発行者 | `sso.issuer`（デフォルト: http://localhost:9001） |
| 署名アルゴリズム | RS256 |
| スコープ | openid, profile, email |
| レスポンス形式 | JWT |
| 認可コード有効期限 | 600秒（10分） |
| アクセストークン有効期限 | 3600秒（1時間） |
| リフレッシュトークン有効期限 | 86400秒（24時間） |

**エンドポイント：**
- 認証: `/interaction/:uid`
- ユーザー情報: `/userinfo`
- トークン: `/token`
- OIDC設定: `/.well-known/openid-configuration`

## 備考

### 未実装/制限事項

| 項目 | 現状 | 推奨対応 |
| --- | --- | --- |
| Content-Security-Policy (CSP) | 未実装 | 明示的なCSPヘッダー設定を推奨 |
| X-Frame-Options | 埋め込みパッドのため未設定 | 埋め込み不要な場合は`DENY`設定を検討 |
| CORS制限 | OpenAPI定義で広範な許可 | 特定オリジンへの制限を推奨 |
| パスワードハッシュ化 | 平文保存（デフォルト） | ep_hash_authプラグインの導入を推奨 |

### セキュリティ設定チェックリスト

- [ ] 本番環境で`NODE_ENV=production`を設定
- [ ] SSL/TLSを有効化またはリバースプロキシでSSL終端
- [ ] `trustProxy`を適切に設定
- [ ] 管理者パスワードをデフォルトから変更
- [ ] ep_hash_authプラグインでパスワードハッシュ化
- [ ] `exposeVersion: false`でバージョン情報を隠蔽
- [ ] 適切なログレベルを設定
- [ ] レート制限を環境に応じて調整
- [ ] 不要なプラグインを削除
