# バッチ設計書 19-importSqlFile.ts

## 概要

本ドキュメントは、SQL ファイルからデータをインポートするための TypeScript スクリプト `importSqlFile.ts` のバッチ設計書です。

### 本バッチの処理概要

本バッチは、Etherpad のデータベースダンプ（SQL 形式）からデータを読み込み、settings.json で設定されたデータベースにインポートします。バックアップからのリストアやデータ移行時に使用されます。

**業務上の目的・背景**：Etherpad のデータバックアップやデータベース間の移行において、SQL 形式のダンプファイルを使用することがあります。本バッチはこのような SQL ファイルからデータを読み込み、ueberdb2 を通じてデータベースにインポートすることで、バックアップからの復旧やデータ移行を支援します。

**バッチの実行タイミング**：バックアップからのリストア時、または SQL 形式のデータ移行時に手動で実行します。

**主要な処理内容**：
1. コマンドライン引数から SQL ファイルパスを取得
2. settings.json からデータベース接続情報を読み込み
3. データベースを初期化（キャッシュ無効、100ms 書き込みインターバル）
4. SQL ファイルを行単位で読み込み
5. `REPLACE INTO store VALUES` 形式の行からキー・値を抽出
6. エスケープシーケンスをアンエスケープ
7. データベースに書き込み
8. 1000件ごとに進捗を出力

**前後の処理との関連**：本バッチ実行前に settings.json を適切なデータベース設定に更新しておく必要があります。インポート完了後、Etherpad は設定されたデータベースを使用して起動します。

**影響範囲**：settings.json で設定されたデータベースにデータを書き込みます。既存データがある場合は REPLACE により上書きされます。

## バッチ種別

データ連携処理（データインポート）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（リストア時） |
| 実行時刻 | 計画メンテナンス時 |
| 実行曜日 | 指定なし |
| 実行日 | 指定なし |
| トリガー | 手動 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js | TypeScript 実行環境 |
| settings.json | データベース設定が完了 |
| SQL ファイル | インポート対象ファイルが存在 |
| Etherpad停止 | データ整合性のため推奨 |

### 実行可否判定

- SQL ファイルパスがコマンドライン引数で指定されていること
- SQL ファイルが存在し、読み取り可能であること
- settings.json に正しいデータベース設定があること

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| SQLFILE | 文字列 | Yes | - | インポートする SQL ファイルのパス |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| SQL ファイル | SQL | `REPLACE INTO store VALUES` 形式 |
| settings.json | JSON | データベースの接続設定 |

### SQL ファイル形式

```sql
REPLACE INTO store VALUES ('key', 'value');
```

- キーは最初の `('` と `', '` の間
- 値は `', '` と `');` の間
- エスケープシーケンス: `\0`, `\n`, `\r`, `\b`, `\t`, `\Z`, `\\`, `\'`, `\"`

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| 設定されたDB | 各種 | インポートされたデータ |
| 標準出力 | テキスト | 進捗・完了メッセージ |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| 出力先 | settings.json で設定されたデータベース |

## 処理フロー

### 処理シーケンス

```
1. 引数検証
   └─ SQL ファイルパスが指定されていることを確認
2. 設定読み込み
   ├─ settings モジュールから dbType, dbSettings を取得
   └─ dbWrapperSettings: cache=0, writeInterval=100, json=false
3. データベース初期化
   └─ ueberdb2 Database インスタンス作成・初期化
4. SQL ファイル読み込み
   ├─ "Opening {sqlFile}..." メッセージ出力
   └─ ReadStream でファイルを開く
5. 行単位処理
   ├─ "Reading {sqlFile}..." メッセージ出力
   ├─ readline.createInterface でストリーム処理
   ├─ "REPLACE INTO store VALUES (" で始まる行を抽出
   ├─ キーと値を解析
   ├─ unescape 関数で値をデコード
   └─ db.set でデータベースに書き込み
6. 進捗表示
   └─ 1000件ごとに行番号を出力
7. 完了処理
   ├─ "done. waiting for db to finish transaction..." メッセージ出力
   ├─ データベースをクローズ
   └─ "finished, imported {n} keys." メッセージ出力
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{SQLファイル指定あり?}
    B -->|No| C[エラー終了]
    B -->|Yes| D[settings.json読み込み]
    D --> E[DB初期化]
    E --> F[SQLファイルオープン]
    F --> G[行ループ開始]
    G --> H{REPLACE INTO store VALUES?}
    H -->|No| I[次の行へ]
    H -->|Yes| J[キー・値解析]
    J --> K[unescape処理]
    K --> L[DBに書き込み]
    L --> M{1000件ごと?}
    M -->|Yes| N[進捗表示]
    M -->|No| O{次の行あり?}
    N --> O
    O -->|Yes| G
    O -->|No| P[DBクローズ待機]
    P --> Q[完了メッセージ出力]
    Q --> R[バッチ終了]
    I --> O
```

## データベース操作仕様

### 操作別データベース影響一覧

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| データ書き込み | store | INSERT/UPDATE | キー・値ペアを書き込み |

### テーブル別操作詳細

#### store テーブル（ueberdb2 標準）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SET | key, value | SQL ファイルから抽出 | 既存キーは上書き |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| (例外) | 引数不足 | SQL ファイル未指定 | 正しい引数で再実行 |
| (例外) | ファイル読み込み失敗 | SQL ファイルが存在しない | ファイルパスを確認 |
| (例外) | DB接続失敗 | 接続設定が不正 | 設定内容を確認 |
| (例外) | 書き込み失敗 | 対象DB のエラー | 対象DB の状態を確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

### 障害時対応

- unhandledRejection を例外に変換するため、エラー時はプロセス終了
- 部分的にインポートされたデータは対象DBに残る可能性あり
- 再実行で上書き可能

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | レコード単位 |
| コミットタイミング | 100ms インターバル |
| ロールバック条件 | 個別エラー時は継続 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | SQL ファイルのレコード数依存 |
| 目標処理時間 | DBMS のパフォーマンス依存 |
| メモリ使用量上限 | ストリーム処理のため低メモリ |

## 排他制御

- Etherpad を停止してから実行することを推奨
- 同一キーへの同時書き込みを防ぐため

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 情報ログ | DB初期化時 | "initializing db" / "done" |
| 情報ログ | ファイルオープン時 | "Opening {sqlFile}..." |
| 情報ログ | 読み込み開始時 | "Reading {sqlFile}..." |
| デバッグログ | 各レコード | "key: {key} val: {value}" / "unval: {unescapedValue}" |
| 進捗ログ | 1000件ごと | "{keyNo}" |
| 完了ログ | 終了時 | "finished, imported {n} keys." |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| なし | - | - |

## 備考

- 実行コマンド例: `node importSqlFile.js $SQLFILE`
- json: false 設定により、データは既に JSON エンコード済みとして扱われる
- unescape 関数で MySQL 形式のエスケープシーケンスをデコード
- NULL、true、false は特別な値として処理
- writeInterval: 100 により、100ms ごとにデータベースへ書き込み
- 処理経過時間をミリ秒単位でログ出力
