# 機能設計書 29-保存済みリビジョン一覧取得

## 概要

本ドキュメントは、Etherpadにおける保存済みリビジョン一覧取得機能の設計を定義するものである。

### 本機能の処理概要

この機能は、指定されたパッドに対して明示的に保存された「保存済みリビジョン」の一覧を取得する機能である。保存済みリビジョンは、ユーザーがマイルストーンとしてマークした特定のリビジョン番号の配列として返却される。

**業務上の目的・背景**：通常のリビジョンは編集操作ごとに自動的に作成されるが、保存済みリビジョンはユーザーが意図的にマークした重要なバージョンである。これにより、文書の特定時点への復元や、バージョン管理が容易になる。外部システムとの連携では、保存済みリビジョンのみを対象とした処理が可能になる。

**機能の利用シーン**：
- タイムスライダー画面での保存済みリビジョンの表示
- バージョン管理システムとの連携
- 重要なマイルストーンの一覧表示
- 復元対象のリビジョン選択
- ドキュメント履歴の管理

**主要な処理内容**：
1. リクエストパラメータのバリデーション（padID）
2. パッドの存在確認と取得
3. パッドオブジェクトから保存済みリビジョン一覧を取得
4. リビジョン番号の配列をソートして返却

**関連システム・外部連携**：
- ueberDB2データベース層を経由してパッドデータを取得
- REST API（`GET /api/2/savedRevisions`）として外部システムから呼び出し可能

**権限による制御**：API認証（APIキーまたはOAuth2トークン）が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面からの直接利用なし（API経由のみ） |

## 機能種別

データ取得（SELECT）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| padID | string | Yes | パッドの識別子 | 文字列型であること、有効なパッドIDであること |

### 入力データソース

- HTTPリクエストのクエリパラメータ（GETリクエスト）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | integer | レスポンスコード（0: 成功） |
| message | string | レスポンスメッセージ（"ok"） |
| data.savedRevisions | number[] | 保存済みリビジョン番号の配列（昇順ソート） |

### 出力先

- HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. APIリクエストの受信
   └─ GET /api/2/savedRevisions?padID=xxx

2. 入力パラメータのバリデーション
   └─ padIDの文字列型チェック

3. パッドの取得
   └─ getPadSafe()でパッドIDの妥当性確認とパッド取得

4. 保存済みリビジョン一覧の取得
   └─ pad.getSavedRevisionsList()でリビジョン番号配列を取得

5. レスポンスの返却
   └─ {savedRevisions: number[]}形式でJSON返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[リクエスト受信]
    B --> C[padIDバリデーション]
    C --> D{padIDは有効?}
    D -->|No| E[エラー: padID is not a string]
    D -->|Yes| F[パッド取得 getPadSafe]
    F --> G{パッド存在?}
    G -->|No| H[エラー: padID does not exist]
    G -->|Yes| I[getSavedRevisionsList実行]
    I --> J[リビジョン配列を返却]
    J --> K[終了]
    E --> K
    H --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-29-01 | 昇順ソート | 返却されるリビジョン番号は昇順でソートされる | 常に適用 |
| BR-29-02 | 空配列 | 保存済みリビジョンがない場合は空配列を返却 | 常に適用 |
| BR-29-03 | 重複なし | 同一リビジョン番号は1回のみ含まれる | 常に適用 |

### 計算ロジック

`getSavedRevisionsList()`メソッドは、`savedRevisions`配列からリビジョン番号（`revNum`プロパティ）を抽出し、昇順でソートして返却する。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| パッド取得 | pad:{padID} | SELECT | パッドオブジェクト（savedRevisionsプロパティ含む）を取得 |

### テーブル別操作詳細

#### pad:{padID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | savedRevisions | キー: pad:{padID} | Padオブジェクト内のsavedRevisions配列 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| apierror | パラメータ型エラー | padIDが文字列でない | 文字列型のpadIDを指定する |
| apierror | パッド不存在 | 指定されたpadIDが存在しない | 正しいpadIDを指定する |

### リトライ仕様

特になし（読み取り専用操作のため）

## トランザクション仕様

読み取り専用操作のためトランザクション管理は不要。

## パフォーマンス要件

- レスポンス時間: 通常50ms以内
- パッドオブジェクトはメモリキャッシュされている

## セキュリティ考慮事項

- API認証（APIキーまたはOAuth2トークン）が必要
- パッドIDはサニタイズされる

## 備考

- 保存済みリビジョンはsaveRevision APIまたはSocket.IO経由のSAVE_REVISIONメッセージで作成される
- 各保存済みリビジョンにはラベルやタイムスタンプなどのメタ情報も含まれるが、本APIではリビジョン番号のみを返却

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Pad.ts | `src/node/db/Pad.ts` | savedRevisionsプロパティ（49行目、66行目） |

**読解のコツ**: `savedRevisions`は`{revNum, savedById, label, timestamp, id}`形式のオブジェクト配列。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | `/savedRevisions` GETエンドポイント定義（1116-1130行目）|
| 2-2 | APIHandler.ts | `src/node/handler/APIHandler.ts` | listSavedRevisions関数のパラメータ定義（109行目）|

**主要処理フロー**:
1. **1116-1130行目** (RestAPI.ts): GETエンドポイント定義
2. **109行目** (APIHandler.ts): `listSavedRevisions: ['padID']`

#### Step 3: API関数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | listSavedRevisions関数の実装（439-443行目）|

**主要処理フロー**:
- **440-441行目**: getPadSafeでパッド取得
- **442行目**: pad.getSavedRevisionsList()で一覧取得

#### Step 4: Padクラスのメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Pad.ts | `src/node/db/Pad.ts` | getSavedRevisionsList メソッド（81-85行目）|

**主要処理フロー**:
- **82-84行目**: savedRevisionsからrevNumを抽出し、ソートして返却

### プログラム呼び出し階層図

```
RestAPI.expressCreateServer (RestAPI.ts)
    │
    ├─ Express Router: GET /api/2/savedRevisions
    │      │
    │      └─ APIHandler.handle (APIHandler.ts)
    │             │
    │             └─ API.listSavedRevisions (API.ts:439-443)
    │                    │
    │                    ├─ getPadSafe (API.ts:894-920)
    │                    │      │
    │                    │      └─ padManager.getPad (PadManager.ts)
    │                    │
    │                    └─ Pad.getSavedRevisionsList (Pad.ts:81-85)
    │                           │
    │                           └─ savedRevisions.map(rev => rev.revNum).sort()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

padID ─────────────────> getPadSafe ─────────────────> Pad Object
                              │
                              v
                    getSavedRevisionsList
                              │
                              ├─ savedRevisions.map(rev => rev.revNum)
                              │
                              └─ sort((a, b) => a - b)
                              │
                              v
                    ───────────────────────────────> {savedRevisions: number[]}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | listSavedRevisions関数の実装 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | APIバージョン管理とパラメータ定義 |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | RESTエンドポイント定義 |
| Pad.ts | `src/node/db/Pad.ts` | ソース | Padクラスと保存済みリビジョン取得メソッド |
| PadManager.ts | `src/node/db/PadManager.ts` | ソース | パッド管理とキャッシュ |
| pad.ts | `src/tests/backend/specs/api/pad.ts` | テスト | APIテストケース（147-154行目、257-264行目） |
