# 機能設計書 30-リビジョン保存

## 概要

本ドキュメントは、Etherpadにおけるリビジョン保存機能の設計を定義するものである。

### 本機能の処理概要

この機能は、指定されたパッドの現在のリビジョン（または指定されたリビジョン）を「保存済みリビジョン」としてマークする機能である。保存済みリビジョンはマイルストーンとして機能し、後で参照や復元に使用できる。

**業務上の目的・背景**：文書作成において、重要なバージョンをマークしておくことで、後から特定時点の内容に戻したり、変更履歴を追跡したりすることが容易になる。これはバージョン管理の基本機能であり、チームでのドキュメント作成において重要な役割を果たす。

**機能の利用シーン**：
- ドキュメントの重要なマイルストーンの作成
- レビュー前のスナップショット保存
- 公開前のバージョン固定
- 復元ポイントの作成
- 定期的なバックアップポイントの設定

**主要な処理内容**：
1. リクエストパラメータのバリデーション（padID、rev）
2. パッドの存在確認と取得
3. リビジョン番号の決定（rev未指定時はヘッドリビジョン）
4. 著者IDの生成（API経由の場合は"API"著者）
5. 保存済みリビジョンの追加とデータベースへの保存

**関連システム・外部連携**：
- ueberDB2データベース層を経由してパッドデータを更新
- REST API（`POST /api/2/savedRevisions`）として外部システムから呼び出し可能
- Socket.IO経由のSAVE_REVISIONメッセージでも呼び出し可能

**権限による制御**：API認証（APIキーまたはOAuth2トークン）が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面からの直接利用なし（API経由のみ） |

## 機能種別

データ更新（INSERT）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| padID | string | Yes | パッドの識別子 | 文字列型であること、有効なパッドIDであること |
| rev | number | No | 保存対象のリビジョン番号 | 整数であること、0以上かつヘッドリビジョン以下であること |

### 入力データソース

- HTTPリクエストボディ（POSTリクエスト、JSON形式）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | integer | レスポンスコード（0: 成功） |
| message | string | レスポンスメッセージ（"ok"） |
| data | null | データなし |

### 出力先

- HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. APIリクエストの受信
   └─ POST /api/2/savedRevisions

2. 入力パラメータのバリデーション
   └─ revパラメータが指定されている場合は数値変換・検証

3. パッドの取得
   └─ getPadSafe()でパッドIDの妥当性確認とパッド取得

4. リビジョン番号の決定
   ├─ revが指定されている場合: 指定値を使用
   └─ revが未指定の場合: ヘッドリビジョン番号を使用

5. リビジョン妥当性チェック
   └─ revがヘッドリビジョンを超える場合はエラー

6. 著者の作成
   └─ authorManager.createAuthor('API')でAPI著者を作成

7. 保存済みリビジョンの追加
   └─ pad.addSavedRevision(rev, authorID, label)

8. データベースへの保存
   └─ pad.saveToDatabase()で永続化

9. レスポンスの返却
   └─ 成功レスポンスを返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[リクエスト受信]
    B --> C{revパラメータあり?}
    C -->|Yes| D[revを検証]
    C -->|No| E[rev=undefined]
    D --> F[パッド取得 getPadSafe]
    E --> F
    F --> G{パッド存在?}
    G -->|No| H[エラー: padID does not exist]
    G -->|Yes| I[ヘッドリビジョン取得]
    I --> J{revが指定されている?}
    J -->|Yes| K{rev > head?}
    J -->|No| L[rev = head]
    K -->|Yes| M[エラー: rev is higher than head]
    K -->|No| N[著者作成 API]
    L --> N
    N --> O[addSavedRevision実行]
    O --> P[saveToDatabase実行]
    P --> Q[成功レスポンス返却]
    Q --> R[終了]
    H --> R
    M --> R
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-30-01 | リビジョン範囲制限 | 保存可能なリビジョンは0からヘッドリビジョンまで | 常に適用 |
| BR-30-02 | デフォルトリビジョン | rev未指定時は最新リビジョンを保存 | revパラメータがundefined時 |
| BR-30-03 | 重複保存防止 | 同じリビジョン番号が既に保存されている場合は無視（サイレント成功） | 常に適用 |
| BR-30-04 | ラベル自動生成 | API経由の場合はラベルが"Saved through API call"となる | 常に適用 |

### 計算ロジック

`addSavedRevision()`メソッドは以下の処理を行う：
1. 指定されたリビジョン番号が既に保存されているか確認
2. 新しい保存済みリビジョンオブジェクトを作成（revNum, savedById, label, timestamp, id）
3. savedRevisions配列に追加
4. saveToDatabase()でデータベースに永続化

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| パッド取得 | pad:{padID} | SELECT | パッドオブジェクトを取得 |
| パッド更新 | pad:{padID} | UPDATE | savedRevisions配列を更新 |
| 著者作成 | globalAuthor:{authorID} | INSERT | API著者を作成 |

### テーブル別操作詳細

#### pad:{padID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| UPDATE | savedRevisions | 新しい保存済みリビジョンオブジェクトを配列に追加 | Padオブジェクト全体を保存 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| apierror | パッド不存在 | 指定されたpadIDが存在しない | 正しいpadIDを指定する |
| apierror | リビジョン範囲超過 | revがヘッドリビジョンより大きい | 有効なリビジョン番号を指定する |
| apierror | 無効なリビジョン | revが数値でない、または負数 | 有効な整数を指定する |

### リトライ仕様

特になし

## トランザクション仕様

単一パッドの更新のため、トランザクション管理は不要。ueberDB2が自動的にキャッシュとデータベースの一貫性を保証。

## パフォーマンス要件

- レスポンス時間: 通常100ms以内
- データベース書き込みを伴うため、読み取り専用操作より若干遅い

## セキュリティ考慮事項

- API認証（APIキーまたはOAuth2トークン）が必要
- パッドIDはサニタイズされる

## 備考

- 同じリビジョンを複数回保存しても、1つのエントリしか作成されない
- 保存済みリビジョンは削除APIがないため、一度保存すると削除できない
- Socket.IO経由でもSAVE_REVISIONメッセージで同様の機能が利用可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Pad.ts | `src/node/db/Pad.ts` | savedRevisionsプロパティ構造（49行目、623-641行目） |

**読解のコツ**: 保存済みリビジョンは`{revNum, savedById, label, timestamp, id}`形式のオブジェクト。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | `/savedRevisions` POSTエンドポイント定義（1133-1156行目）|
| 2-2 | APIHandler.ts | `src/node/handler/APIHandler.ts` | saveRevision関数のパラメータ定義（110行目）|

**主要処理フロー**:
1. **1133-1156行目** (RestAPI.ts): POSTエンドポイント定義
2. **110行目** (APIHandler.ts): `saveRevision: ['padID', 'rev']`

#### Step 3: API関数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | saveRevision関数の実装（455-476行目）|

**主要処理フロー**:
- **457-459行目**: revパラメータのバリデーション
- **462行目**: getPadSafeでパッド取得
- **463行目**: ヘッドリビジョン取得
- **466-472行目**: revの決定とバリデーション
- **474行目**: 著者作成
- **475行目**: addSavedRevision実行

#### Step 4: Padクラスのメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Pad.ts | `src/node/db/Pad.ts` | addSavedRevision メソッド（623-642行目）|

**主要処理フロー**:
- **625-628行目**: 重複チェック
- **631-637行目**: 保存済みリビジョンオブジェクト作成
- **640行目**: savedRevisions配列に追加
- **641行目**: saveToDatabase()で永続化

#### Step 5: Socket.IO経由の呼び出しを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | PadMessageHandler.ts | `src/node/handler/PadMessageHandler.ts` | handleSaveRevisionMessage関数（444-448行目）|

**主要処理フロー**:
- **444-448行目**: Socket.IO経由でのリビジョン保存処理

### プログラム呼び出し階層図

```
RestAPI.expressCreateServer (RestAPI.ts)
    │
    ├─ Express Router: POST /api/2/savedRevisions
    │      │
    │      └─ APIHandler.handle (APIHandler.ts)
    │             │
    │             └─ API.saveRevision (API.ts:455-476)
    │                    │
    │                    ├─ checkValidRev (checkValidRev.ts)
    │                    │
    │                    ├─ getPadSafe (API.ts:894-920)
    │                    │      │
    │                    │      └─ padManager.getPad (PadManager.ts)
    │                    │
    │                    ├─ authorManager.createAuthor('API')
    │                    │
    │                    └─ Pad.addSavedRevision (Pad.ts:623-642)
    │                           │
    │                           ├─ 重複チェック
    │                           │
    │                           ├─ savedRevisions.push()
    │                           │
    │                           └─ this.saveToDatabase()
    │                                  │
    │                                  └─ db.set('pad:{id}', this)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

padID ─────────────────> getPadSafe ─────────────────> Pad Object
                              │
rev (optional) ──────────> checkValidRev ──────────> Validated Rev Number
                              │
                              v
                    rev決定（未指定時はhead）
                              │
                              v
                    authorManager.createAuthor('API') → authorID
                              │
                              v
                    addSavedRevision(rev, authorID, label)
                              │
                              ├─ 重複チェック
                              │
                              ├─ savedRevisions配列更新
                              │
                              └─ saveToDatabase → DB更新
                              │
                              v
                    ───────────────────────────────> {code: 0, message: "ok"}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | saveRevision関数の実装 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | APIバージョン管理とパラメータ定義 |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | RESTエンドポイント定義 |
| Pad.ts | `src/node/db/Pad.ts` | ソース | Padクラスとリビジョン保存メソッド |
| AuthorManager.ts | `src/node/db/AuthorManager.ts` | ソース | 著者作成 |
| PadMessageHandler.ts | `src/node/handler/PadMessageHandler.ts` | ソース | Socket.IO経由のリビジョン保存 |
| checkValidRev.ts | `src/node/utils/checkValidRev.ts` | ソース | リビジョン番号バリデーション |
| pad.ts | `src/tests/backend/specs/api/pad.ts` | テスト | APIテストケース（240-246行目） |
