# 機能設計書 38-著者作成

## 概要

本ドキュメントは、Etherpadの著者作成機能（createAuthor）の設計仕様を記載する。

### 本機能の処理概要

本機能は、新しい著者（Author）を作成するAPI機能である。著者はパッドへの貢献者を識別するためのエンティティであり、一意のIDと名前、色情報を持つ。

**業務上の目的・背景**：Etherpadの共同編集において、各ユーザーの編集内容を区別するために著者という概念が使用される。この機能は、API経由で著者を作成することで、外部システムとの連携やバッチ処理での著者事前登録を可能にする。著者IDは編集履歴に記録され、誰がどの部分を編集したかを追跡できる。

**機能の利用シーン**：外部認証システムとの連携時、バッチ処理での著者事前作成、テストデータの作成など。

**主要な処理内容**：
1. ランダムな著者ID（a.xxxxxxxxxxxxxx形式）を生成
2. ランダムな色IDを割り当て
3. 著者オブジェクトをデータベースに保存
4. 生成された著者IDを返却

**関連システム・外部連携**：REST API経由でHTTPリクエストを受け付け、JSON形式でレスポンスを返す。データベース（ueberDB2）に著者データを保存する。

**権限による制御**：APIキー認証またはSSO認証により、API呼び出しの認可を行う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | API経由 | 主機能 | REST API呼び出しで著者を作成 |

## 機能種別

データ追加（INSERT操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | string | No | 著者の表示名 | - |

### 入力データソース

REST API POSTリクエストのボディとしてJSON形式でパラメータを受け取る。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | integer | 結果コード（0: 成功） |
| message | string | 結果メッセージ（"ok"） |
| data.authorID | string | 生成された著者ID（a.xxxxxxxxxxxxxxxx形式） |

### 出力先

HTTPレスポンスとしてJSON形式で返却

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ POST /api/2/authors
2. 認証・認可チェック
   └─ APIキーまたはSSOトークンの検証
3. 著者ID生成
   └─ a. + 16文字のランダム文字列
4. 色ID割り当て
   └─ カラーパレットからランダムに選択
5. 著者オブジェクト構築
   └─ {colorId, name, timestamp}
6. データベース保存
   └─ db.set(`globalAuthor:${authorID}`, authorObj)
7. レスポンス生成・返却
   └─ {authorID: string}形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証チェック}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[著者ID生成]
    D --> E[色ID割り当て]
    E --> F[著者オブジェクト構築]
    F --> G[データベース保存]
    G --> H[JSONレスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-38-01 | ID形式 | 著者IDは「a.」+ 16文字のランダム英数字 | 常時 |
| BR-38-02 | 色ランダム割り当て | カラーパレット（52色）からランダムに選択 | 常時 |
| BR-38-03 | タイムスタンプ記録 | 作成時刻がtimestampとして記録される | 常時 |
| BR-38-04 | 名前任意 | nameパラメータは省略可能 | 常時 |

### 計算ロジック

著者ID生成:
```javascript
const author = `a.${randomString(16)}`;
```

色ID割り当て:
```javascript
const colorId = Math.floor(Math.random() * (colorPalette.length)); // 0-51
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 著者保存 | globalAuthor:{authorID} | INSERT | 著者オブジェクトを保存 |

### テーブル別操作詳細

#### globalAuthor:{authorID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | colorId | 0-51のランダム整数 | カラーパレットのインデックス |
| INSERT | name | 入力値（またはnull） | 著者の表示名 |
| INSERT | timestamp | Date.now() | 作成時刻 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 4 | 認証エラー | APIキーが無効 | 正しいAPIキーを使用 |

### リトライ仕様

リトライは安全（冪等ではないが、新しいIDが生成されるため影響なし）

## トランザクション仕様

単一のデータベース書き込みのため、トランザクション管理は不要。

## パフォーマンス要件

- レスポンス時間: 100ms以内（通常条件下）

## セキュリティ考慮事項

- APIキー認証またはSSO認証が必須
- 著者IDはランダム生成のため推測困難

## 備考

API Version 1.0で追加された機能。著者IDは「a.」で始まり、16文字のランダム英数字が続く形式。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、著者オブジェクトの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | 著者オブジェクトの構造（203-207行目） |
| 1-2 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | カラーパレットの定義（27-92行目） |

**読解のコツ**: 著者オブジェクトは{colorId, name, timestamp}の3プロパティを持つシンプルな構造。

#### Step 2: エントリーポイントを理解する

REST APIのエンドポイント定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | POST /authorsのマッピング定義（381-397行目） |

**主要処理フロー**:
1. **381-397行目**: エンドポイント定義とcreateAuthor関数へのマッピング

#### Step 3: API関数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | createAuthorへのエクスポート（61行目） |
| 3-2 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | createAuthor関数の実装（198-213行目） |

**主要処理フロー**:
- **200行目**: 著者IDの生成（a. + randomString(16)）
- **203-207行目**: 著者オブジェクトの構築
- **210行目**: データベースへの保存
- **212行目**: 著者IDを返却

### プログラム呼び出し階層図

```
RestAPI.expressCreateServer
    │
    ├─ app.use('/api/2', ...)
    │      └─ apiHandler.handle('1', 'createAuthor', fields)
    │              └─ API.createAuthor(name)
    │                      └─ authorManager.createAuthor(name)
    │                              ├─ randomString(16)
    │                              ├─ Math.random() * colorPalette.length
    │                              ├─ db.set(`globalAuthor:${author}`, authorObj)
    │                              └─ return {authorID: author}
    │
    └─ res.json({authorID})
```

### データフロー図

```
[入力]               [処理]                    [出力]

name ────────────▶ createAuthor()
                           │
                           ▼
                   randomString(16) ─────────▶ authorID (a.xxx...)
                           │
                           ▼
                   random colorId (0-51)
                           │
                           ▼
                   authorObj = {colorId, name, timestamp}
                           │
                           ▼
                   db.set(`globalAuthor:${authorID}`)
                           │
                           ▼
                   {authorID} ────────────────▶ JSONレスポンス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | createAuthorのエクスポート |
| AuthorManager.ts | `src/node/db/AuthorManager.ts` | ソース | 著者管理の実装 |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | REST APIエンドポイント定義 |
| randomstring.ts | `src/node/utils/randomstring.ts` | ソース | ランダム文字列生成 |
