# 機能設計書 40-著者名取得

## 概要

本ドキュメントは、Etherpadの著者名取得機能（getAuthorName）の設計仕様を記載する。

### 本機能の処理概要

本機能は、指定された著者IDに対応する著者の表示名を取得するAPI機能である。著者の識別情報から人間が理解しやすい名前を取得することで、UI表示や履歴表示に利用される。

**業務上の目的・背景**：Etherpadの共同編集において、各編集者は著者IDで管理されるが、ユーザーには著者IDではなく表示名（名前）で表示する必要がある。この機能は、著者IDから表示名を取得することで、編集履歴やユーザー一覧において人間が識別しやすい形で著者情報を表示することを可能にする。

**機能の利用シーン**：編集履歴の表示、パッド内のユーザー一覧表示、外部システムでの著者情報表示など。

**主要な処理内容**：
1. 著者IDの検証
2. データベースから著者オブジェクトの名前フィールドを取得
3. 著者名を返却

**関連システム・外部連携**：REST API経由でHTTPリクエストを受け付け、JSON形式でレスポンスを返す。データベース（ueberDB2）から著者データを読み取る。

**権限による制御**：APIキー認証またはSSO認証により、API呼び出しの認可を行う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | API経由 | 主機能 | REST API呼び出しで著者名を取得 |

## 機能種別

データ参照（SELECT操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| authorID | string | Yes | 著者ID（a.xxxxxxxxxxxxxxxx形式） | 文字列型であること |

### 入力データソース

REST API GETリクエストのクエリパラメータとして受け取る。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | integer | 結果コード（0: 成功） |
| message | string | 結果メッセージ（"ok"） |
| data.authorName | string | 著者の表示名（未設定の場合はnull） |

### 出力先

HTTPレスポンスとしてJSON形式で返却

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ GET /api/2/authors/name?authorID=xxx
2. 認証・認可チェック
   └─ APIキーまたはSSOトークンの検証
3. 著者名取得
   └─ db.getSub(`globalAuthor:${author}`, ['name'])
4. レスポンス生成・返却
   └─ {authorName: string | null}形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証チェック}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[著者名取得]
    D --> E{名前存在?}
    E -->|存在する| F[authorName返却]
    E -->|存在しない| G[null返却]
    F --> H[JSONレスポンス返却]
    G --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-01 | 著者ID必須 | authorIDパラメータは必須 | 常時 |
| BR-40-02 | 未設定時null | 名前が設定されていない場合はnullを返す | 名前未設定時 |
| BR-40-03 | 読み取り専用 | この操作はデータを変更しない | 常時 |

### 計算ロジック

著者名取得:
```javascript
await db.getSub(`globalAuthor:${author}`, ['name']);
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 著者名取得 | globalAuthor:{authorID} | SELECT | 著者オブジェクトのnameフィールドを取得 |

### テーブル別操作詳細

#### globalAuthor:{authorID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | name | キー: globalAuthor:{authorID} | 著者の表示名 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 4 | 認証エラー | APIキーが無効 | 正しいAPIキーを使用 |

### リトライ仕様

リトライは安全（読み取り専用操作）

## トランザクション仕様

読み取り専用操作のため、トランザクション管理は不要。

## パフォーマンス要件

- レスポンス時間: 50ms以内（通常条件下）
- 単一のDBアクセスで完結

## セキュリティ考慮事項

- APIキー認証またはSSO認証が必須
- 著者IDが既知であれば任意の著者名を取得可能

## 備考

API Version 1.1で追加された機能。著者名が設定されていない場合はnullが返される。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、著者オブジェクトの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | globalAuthor:{authorID}のデータ構造（203-207行目のcreateAuthor参照） |

**読解のコツ**: 著者オブジェクトは{colorId, name, timestamp}の3プロパティを持つ。getAuthorNameはこのうちnameフィールドのみを取得する。

#### Step 2: エントリーポイントを理解する

REST APIのエンドポイント定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | GET /authors/nameのマッピング定義（770-784行目） |

**主要処理フロー**:
1. **770-784行目**: エンドポイント定義とgetAuthorName関数へのマッピング
2. **771行目**: APIバージョン1.1で追加
3. **772行目**: レスポンス形式は{authorName: string}

#### Step 3: API関数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | getAuthorNameのエクスポート（63行目） |
| 3-2 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | getAuthorName関数の実装（239行目） |

**主要処理フロー**:
- **239行目**: `db.getSub(`globalAuthor:${author}`, ['name'])`でnameフィールドのみを取得

### プログラム呼び出し階層図

```
RestAPI.expressCreateServer
    │
    ├─ app.use('/api/2', ...)
    │      └─ apiHandler.handle('1.1', 'getAuthorName', fields)
    │              └─ API.getAuthorName(authorID)
    │                      └─ authorManager.getAuthorName(author)
    │                              └─ db.getSub(`globalAuthor:${author}`, ['name'])
    │
    └─ res.json({authorName})
```

### データフロー図

```
[入力]               [処理]                    [出力]

authorID ────────▶ db.getSub(`globalAuthor:${authorID}`, ['name'])
                           │
                           ▼
                   nameフィールド取得
                           │
                           ▼
                   {authorName} ────────────▶ JSONレスポンス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | getAuthorNameのエクスポート |
| AuthorManager.ts | `src/node/db/AuthorManager.ts` | ソース | 著者管理の実装 |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | REST APIエンドポイント定義 |
