# 機能設計書 41-著者のパッド一覧取得

## 概要

本ドキュメントは、Etherpadにおける著者のパッド一覧取得機能の詳細設計を記載したものである。この機能は、特定の著者が貢献したすべてのパッドIDを取得するAPI機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：
協調編集環境において、各著者がどのドキュメント（パッド）に貢献したかを追跡することは、ユーザーの活動履歴管理、監査、およびコンテンツ管理において重要である。この機能により、管理者やシステムは特定の著者の編集活動範囲を把握でき、ユーザー自身も自分が参加したパッドを確認できる。

**機能の利用シーン**：
- ユーザーが自分の編集したパッド一覧を確認したい場合
- 管理者が特定ユーザーの活動範囲を監査する場合
- システムが著者の貢献度分析を行う場合
- ユーザー削除時に関連パッドを特定する場合

**主要な処理内容**：
1. 入力された著者IDの妥当性確認
2. データベースから著者オブジェクト（globalAuthor）を取得
3. 著者オブジェクト内のpadIDsプロパティからパッドID一覧を抽出
4. パッドID配列をJSON形式で返却

**関連システム・外部連携**：
- ueberDB2データベースによるデータ永続化
- REST API経由でのクライアントからのアクセス

**権限による制御**：
- APIキーまたはOAuth2トークンによる認証が必要
- 管理者権限（admin）またはクライアント認証情報での認可

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は直接的な画面連携はなく、APIとして提供される |

## 機能種別

データ取得（SELECT操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| authorID | string | Yes | 著者ID（形式: a.XXXXXXXXXXXXXXXX） | 16文字のランダム文字列を含む形式 |

### 入力データソース

- REST API: `GET /api/2/authors/pads?authorID={authorID}`
- クエリパラメータとして著者IDを受け取る

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | number | 処理結果コード（0: 成功、1: パラメータエラー） |
| message | string | 処理結果メッセージ |
| data.padIDs | string[] | 著者が貢献したパッドIDの配列 |

### 出力先

- HTTP Response（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ RestAPI.ts で /api/2/authors/pads エンドポイントを処理

2. 認証・認可チェック
   └─ APIHandler.ts で APIキーまたはOAuth2トークンを検証

3. パラメータ抽出
   └─ クエリパラメータからauthorIDを取得

4. API関数呼び出し
   └─ API.ts の listPadsOfAuthor を呼び出し

5. 著者データ取得
   └─ AuthorManager.ts の listPadsOfAuthor で globalAuthor:{authorID} をDBから取得

6. 存在チェック
   └─ 著者が存在しない場合はapierrorをスロー

7. パッドID一覧抽出
   └─ author.padIDs オブジェクトのキーを配列化

8. レスポンス返却
   └─ {padIDs: [...]} 形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証チェック}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[authorIDパラメータ取得]
    D --> E[DB: globalAuthor取得]
    E --> F{著者存在?}
    F -->|No| G[エラー: authorID does not exist]
    F -->|Yes| H[padIDsからキー配列生成]
    H --> I[レスポンス返却]
    G --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-41-01 | 著者存在チェック | 指定された著者IDがデータベースに存在しない場合はエラーとする | 常時 |
| BR-41-02 | 空配列許容 | 著者が存在するがパッドに貢献していない場合は空配列を返す | 著者存在時 |
| BR-41-03 | パッド削除時の整合性 | パッド削除時には著者のpadIDsから該当パッドが削除される | パッド削除時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 著者情報取得 | globalAuthor:{authorID} | SELECT | 著者オブジェクト全体を取得 |

### テーブル別操作詳細

#### globalAuthor:{authorID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | colorId | 取得 | 著者の色ID |
| SELECT | name | 取得 | 著者名 |
| SELECT | timestamp | 取得 | 最終更新タイムスタンプ |
| SELECT | padIDs | 取得 | パッドIDをキーとするオブジェクト |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | apierror | authorIDが存在しない | 正しい著者IDを指定する |
| 4 | 認証エラー | APIキーまたはトークンが無効 | 正しい認証情報を設定する |

### リトライ仕様

リトライは不要（同じ入力に対して結果は変わらない）

## トランザクション仕様

読み取り専用操作のため、トランザクション管理は不要

## パフォーマンス要件

- レスポンス時間: 100ms以内（標準的なデータ量の場合）
- 大量のパッドに貢献している著者の場合、padIDsオブジェクトのサイズに応じて処理時間が増加する可能性あり

## セキュリティ考慮事項

- APIキー認証またはOAuth2認証による保護
- 著者IDの形式検証によるインジェクション対策
- 他ユーザーの貢献パッド一覧も取得可能（プライバシーへの配慮が必要）

## 備考

- padIDsプロパティは、著者がパッドに初めて貢献した際にAuthorManager.addPadで追加される
- パッド削除時にはAuthorManager.removePadで対応するエントリが削除される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

著者オブジェクト（globalAuthor）の構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | 著者データ構造とpadIDsの管理方法を理解 |

**読解のコツ**:
- globalAuthor:{authorID}というキーでDBに保存されるオブジェクトの構造を把握する
- padIDsはオブジェクト形式で、キーがパッドID、値は1（使用されていない）

#### Step 2: エントリーポイントを理解する

REST APIエンドポイントの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | /api/2/authors/padsエンドポイントの定義 |

**主要処理フロー**:
1. **425-438行目**: GET /authors/pads エンドポイントの定義
2. **426行目**: functionName: 'listPadsOfAuthor' の指定
3. **427-434行目**: authorIDパラメータの定義

#### Step 3: API関数層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | API関数のエクスポート定義 |
| 3-2 | APIHandler.ts | `src/node/handler/APIHandler.ts` | 認証と関数呼び出しの処理 |

**主要処理フロー**:
- **64行目（API.ts）**: `exports.listPadsOfAuthor = authorManager.listPadsOfAuthor;` でAuthorManagerの関数を直接エクスポート
- **44行目（APIHandler.ts）**: バージョン1のAPI定義で`listPadsOfAuthor: ['authorID']`

#### Step 4: ビジネスロジック層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | AuthorManager.ts | `src/node/db/AuthorManager.ts` | 著者のパッド一覧取得の実装 |

**主要処理フロー**:
- **253-271行目**: listPadsOfAuthor関数の実装
- **260行目**: `db.get('globalAuthor:${authorID}')`で著者データ取得
- **262-265行目**: 著者が存在しない場合のエラーハンドリング
- **268行目**: `Object.keys(author.padIDs || {})`でパッドID配列を生成

### プログラム呼び出し階層図

```
HTTP Request: GET /api/2/authors/pads?authorID=a.xxx
    │
    ├─ RestAPI.ts: expressCreateServer (エンドポイント定義)
    │      └─ mapping.get(GET)!["/authors/pads"]
    │
    ├─ APIHandler.ts: handle (認証・パラメータ処理)
    │      ├─ 認証チェック（APIキー or OAuth2）
    │      └─ api[functionName].apply() 呼び出し
    │
    └─ AuthorManager.ts: listPadsOfAuthor
           ├─ db.get(`globalAuthor:${authorID}`)
           └─ Object.keys(author.padIDs || {})
```

### データフロー図

```
[入力]                    [処理]                         [出力]

authorID (string)  ───▶  AuthorManager.listPadsOfAuthor ───▶ {padIDs: string[]}
                              │
                              ▼
                         DB: globalAuthor:{authorID}
                              │
                              ▼
                         {colorId, name, timestamp, padIDs: {padID1: 1, padID2: 1}}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | API関数の集約・エクスポート |
| AuthorManager.ts | `src/node/db/AuthorManager.ts` | ソース | 著者管理のビジネスロジック |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | RESTエンドポイント定義 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | API認証・パラメータ処理 |
| DB.ts | `src/node/db/DB.ts` | ソース | データベース接続・操作 |
| customError.ts | `src/node/utils/customError.ts` | ソース | カスタムエラークラス |
| sessionsAndGroups.ts | `src/tests/backend/specs/api/sessionsAndGroups.ts` | テスト | APIのテストケース |
