# 機能設計書 44-セッション情報取得

## 概要

本ドキュメントは、Etherpadにおけるセッション情報取得機能の詳細設計を記載したものである。この機能は、指定されたセッションIDの詳細情報（グループID、著者ID、有効期限）を取得するAPI機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：
セッション管理において、特定のセッションの状態を確認することは重要である。この機能により、管理者やシステムはセッションの有効性、関連するグループや著者、有効期限を確認でき、アクセス制御の監査やトラブルシューティングに活用できる。

**機能の利用シーン**：
- セッションの有効性を確認する場合
- セッションに紐付くグループや著者を確認する場合
- セッションの有効期限を確認する場合
- アクセス制御のデバッグ・トラブルシューティング時

**主要な処理内容**：
1. セッションIDの存在確認
2. セッション情報の取得
3. セッション情報の返却

**関連システム・外部連携**：
- ueberDB2データベースによるデータ永続化
- SecurityManagerでのセッション検証時に内部利用

**権限による制御**：
- APIキーまたはOAuth2トークンによる認証が必要

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は直接的な画面連携はなく、APIとして提供される |

## 機能種別

データ取得（SELECT操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| sessionID | string | Yes | セッションID（形式: s.XXXXXXXXXXXXXXXX） | セッションが存在すること |

### 入力データソース

- REST API: `GET /api/2/sessions/info?sessionID={sessionID}`
- クエリパラメータとしてセッションIDを受け取る

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | number | 処理結果コード（0: 成功、1: パラメータエラー） |
| message | string | 処理結果メッセージ |
| data.groupID | string | セッションに紐付くグループID |
| data.authorID | string | セッションに紐付く著者ID |
| data.validUntil | number | セッションの有効期限（UNIXタイムスタンプ） |

### 出力先

- HTTP Response（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ RestAPI.ts で /api/2/sessions/info エンドポイントを処理

2. 認証・認可チェック
   └─ APIHandler.ts で APIキーまたはOAuth2トークンを検証

3. パラメータ抽出
   └─ クエリパラメータから sessionID を取得

4. API関数呼び出し
   └─ API.ts 経由で SessionManager.getSessionInfo を呼び出し

5. セッション情報取得
   └─ session:{sessionID} からセッション情報を取得

6. 存在チェック
   └─ セッションが存在しない場合はapierrorをスロー

7. レスポンス返却
   └─ {groupID, authorID, validUntil} 形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証チェック}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[sessionIDパラメータ取得]
    D --> E[DB: session取得]
    E --> F{セッション存在?}
    F -->|No| G[エラー: sessionID does not exist]
    F -->|Yes| H[セッション情報返却]
    G --> I[レスポンス返却]
    H --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-44-01 | セッション存在チェック | 指定されたセッションIDが存在しない場合はエラー | 常時 |
| BR-44-02 | 期限切れ情報も返却 | 有効期限が過ぎたセッションも情報を返却する | 常時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| セッション取得 | session:{sessionID} | SELECT | セッション情報の取得 |

### テーブル別操作詳細

#### session:{sessionID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | groupID | 取得 | グループID |
| SELECT | authorID | 取得 | 著者ID |
| SELECT | validUntil | 取得 | 有効期限 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | apierror | sessionIDが存在しない | 正しいセッションIDを指定 |
| 4 | 認証エラー | APIキーまたはトークンが無効 | 正しい認証情報を設定 |

### リトライ仕様

リトライは不要（読み取り専用操作）

## トランザクション仕様

読み取り専用操作のため、トランザクション管理は不要

## パフォーマンス要件

- レスポンス時間: 50ms以内（標準的な環境）
- 単一キーの取得のため高速

## セキュリティ考慮事項

- APIキー認証またはOAuth2認証による保護
- セッションIDを知っていれば誰でも情報取得可能（認証後）
- 有効期限のチェックは呼び出し側の責任

## 備考

- 有効期限が過ぎたセッションの情報も取得可能
- 有効期限のチェックはfindAuthorID関数で行われる
- 内部的にはlistSessionsWithDBKeyからも呼び出される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

セッションデータの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SessionManager.ts | `src/node/db/SessionManager.ts` | セッションデータ構造を理解 |

**読解のコツ**:
- session:{sessionID} に保存される {groupID, authorID, validUntil} の構造を把握

#### Step 2: エントリーポイントを理解する

REST APIエンドポイントの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | GET /sessions/info エンドポイントの定義 |

**主要処理フロー**:
1. **488-517行目**: GET /sessions/info エンドポイントの定義
2. **489行目**: functionName: 'getSessionInfo' の指定

#### Step 3: ビジネスロジック層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SessionManager.ts | `src/node/db/SessionManager.ts` | getSessionInfo関数の実装 |

**主要処理フロー**:
- **166-177行目**: getSessionInfo関数の実装
- **168行目**: db.getでセッション情報取得
- **170-173行目**: 存在チェックとエラースロー
- **176行目**: セッション情報を返却

### プログラム呼び出し階層図

```
HTTP Request: GET /api/2/sessions/info?sessionID=s.xxx
    │
    ├─ RestAPI.ts: expressCreateServer (エンドポイント定義)
    │      └─ mapping.get(GET)!["/sessions/info"]
    │
    ├─ APIHandler.ts: handle (認証・パラメータ処理)
    │      ├─ 認証チェック（APIキー or OAuth2）
    │      └─ api[functionName].apply() 呼び出し
    │
    └─ SessionManager.ts: getSessionInfo
           └─ db.get(`session:${sessionID}`)
```

### データフロー図

```
[入力]                    [処理]                              [出力]

sessionID (string)  ───▶  SessionManager.getSessionInfo  ───▶  {groupID, authorID, validUntil}
                               │
                               └─▶ DB: session:{sessionID}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | API関数の集約・エクスポート |
| SessionManager.ts | `src/node/db/SessionManager.ts` | ソース | セッション管理のビジネスロジック |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | RESTエンドポイント定義 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | API認証・パラメータ処理 |
| DB.ts | `src/node/db/DB.ts` | ソース | データベース接続・操作 |
| sessionsAndGroups.ts | `src/tests/backend/specs/api/sessionsAndGroups.ts` | テスト | APIのテストケース |
