# 機能設計書 45-グループセッション一覧取得

## 概要

本ドキュメントは、Etherpadにおけるグループセッション一覧取得機能の詳細設計を記載したものである。この機能は、指定されたグループに関連するすべてのセッション情報を取得するAPI機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：
グループパッドへのアクセス管理において、特定グループに対してどのようなセッションが発行されているかを把握することは重要である。この機能により、管理者はグループへのアクセス権の現状を一覧で確認でき、不要なセッションの特定や監査に活用できる。

**機能の利用シーン**：
- グループに対するアクセス権の監査を行う場合
- グループのセッションを一括管理する場合
- 特定グループへのアクセス状況を確認する場合
- グループ削除前にセッションを確認する場合

**主要な処理内容**：
1. グループIDの存在確認
2. group2sessionsからセッションID一覧を取得
3. 各セッションIDの詳細情報を取得
4. セッション情報の一覧を返却

**関連システム・外部連携**：
- ueberDB2データベースによるデータ永続化
- セッション管理機能との連携

**権限による制御**：
- APIキーまたはOAuth2トークンによる認証が必要

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は直接的な画面連携はなく、APIとして提供される |

## 機能種別

データ取得（SELECT操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| groupID | string | Yes | グループID（形式: g.XXXXXXXXXXXXXXXX） | グループが存在すること |

### 入力データソース

- REST API: `GET /api/2/sessions/group?groupID={groupID}`
- クエリパラメータとしてグループIDを受け取る

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | number | 処理結果コード（0: 成功、1: パラメータエラー） |
| message | string | 処理結果メッセージ |
| data | object | セッションIDをキー、セッション情報を値とするオブジェクト（null: セッションなし） |
| data.{sessionID}.groupID | string | グループID |
| data.{sessionID}.authorID | string | 著者ID |
| data.{sessionID}.validUntil | number | 有効期限（UNIXタイムスタンプ） |

### 出力先

- HTTP Response（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ RestAPI.ts で /api/2/sessions/group エンドポイントを処理

2. 認証・認可チェック
   └─ APIHandler.ts で APIキーまたはOAuth2トークンを検証

3. パラメータ抽出
   └─ クエリパラメータから groupID を取得

4. API関数呼び出し
   └─ API.ts 経由で SessionManager.listSessionsOfGroup を呼び出し

5. グループ存在チェック
   └─ GroupManager.doesGroupExist でグループの存在確認

6. セッション一覧取得
   └─ group2sessions:{groupID} からセッションID一覧を取得

7. 各セッション情報取得
   └─ 各セッションIDについてgetSessionInfoを呼び出し

8. レスポンス返却
   └─ {sessionID: {groupID, authorID, validUntil}, ...} 形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証チェック}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[groupIDパラメータ取得]
    D --> E{グループ存在?}
    E -->|No| F[エラー: groupID does not exist]
    E -->|Yes| G[group2sessions取得]
    G --> H{セッションあり?}
    H -->|No| I[null返却]
    H -->|Yes| J[各セッション情報取得]
    J --> K[セッション一覧返却]
    F --> L[レスポンス返却]
    I --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-45-01 | グループ存在必須 | 指定されたグループIDが存在しない場合はエラー | 常時 |
| BR-45-02 | 空の場合はnull | セッションが1つもない場合はnullを返す | セッションなし時 |
| BR-45-03 | 不正セッションはnull | 存在しないセッションIDはnullとして返す | セッション取得エラー時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| グループ確認 | group:{groupID} | SELECT | グループの存在確認 |
| セッション一覧取得 | group2sessions:{groupID} | SELECT | セッションID一覧の取得 |
| セッション情報取得 | session:{sessionID} | SELECT | 各セッションの詳細取得（複数回） |

### テーブル別操作詳細

#### group2sessions:{groupID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | sessionIDs | 取得 | セッションIDをキーとするオブジェクト |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | apierror | groupIDが存在しない | 正しいグループIDを指定 |
| 4 | 認証エラー | APIキーまたはトークンが無効 | 正しい認証情報を設定 |

### リトライ仕様

リトライは不要（読み取り専用操作）

## トランザクション仕様

読み取り専用操作のため、トランザクション管理は不要

## パフォーマンス要件

- レスポンス時間: セッション数に応じて増加（100セッション以下で500ms以内目安）
- 各セッション情報の取得は逐次処理

## セキュリティ考慮事項

- APIキー認証またはOAuth2認証による保護
- グループIDを知っていれば関連する全セッション情報を取得可能
- 有効期限のチェックは含まない（期限切れセッションも表示）

## 備考

- セッションが1つもない場合はnull（空オブジェクトではない）を返す
- 不正なセッションレコードがある場合はwarningログを出力しnullとして返す
- 内部ではlistSessionsWithDBKeyを共通関数として使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

group2sessionsの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SessionManager.ts | `src/node/db/SessionManager.ts` | group2sessionsデータ構造を理解 |

**読解のコツ**:
- group2sessions:{groupID} = { sessionIDs: { sessionID1: 1, sessionID2: 1, ... } } の構造
- 値（1）は使われず、キーのみが意味を持つ

#### Step 2: エントリーポイントを理解する

REST APIエンドポイントの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | GET /sessions/group エンドポイントの定義 |

**主要処理フロー**:
1. **520-556行目**: GET /sessions/group エンドポイントの定義
2. **521行目**: functionName: 'listSessionsOfGroup' の指定

#### Step 3: ビジネスロジック層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SessionManager.ts | `src/node/db/SessionManager.ts` | listSessionsOfGroup関数と共通関数 |

**主要処理フロー**:
- **213-222行目**: listSessionsOfGroup関数
- **215-218行目**: グループ存在チェック
- **220行目**: listSessionsWithDBKeyの呼び出し
- **246-266行目**: listSessionsWithDBKey共通関数
- **252-263行目**: 各セッションの情報取得とエラーハンドリング

### プログラム呼び出し階層図

```
HTTP Request: GET /api/2/sessions/group?groupID=g.xxx
    │
    ├─ RestAPI.ts: expressCreateServer (エンドポイント定義)
    │      └─ mapping.get(GET)!["/sessions/group"]
    │
    ├─ APIHandler.ts: handle (認証・パラメータ処理)
    │      ├─ 認証チェック（APIキー or OAuth2）
    │      └─ api[functionName].apply() 呼び出し
    │
    └─ SessionManager.ts: listSessionsOfGroup
           ├─ GroupManager.doesGroupExist
           └─ listSessionsWithDBKey(`group2sessions:${groupID}`)
                  ├─ db.get(`group2sessions:${groupID}`)
                  └─ for each sessionID:
                         getSessionInfo(sessionID)
```

### データフロー図

```
[入力]                    [処理]                                    [出力]

groupID (string)  ───▶  SessionManager.listSessionsOfGroup  ───▶  {sessionID: {...}, ...}
                             │                                        または null
                             ├─▶ DB: group2sessions:{groupID}
                             └─▶ DB: session:{sessionID} (複数)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | API関数の集約・エクスポート |
| SessionManager.ts | `src/node/db/SessionManager.ts` | ソース | セッション管理のビジネスロジック |
| GroupManager.ts | `src/node/db/GroupManager.ts` | ソース | グループ存在確認 |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | RESTエンドポイント定義 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | API認証・パラメータ処理 |
| DB.ts | `src/node/db/DB.ts` | ソース | データベース接続・操作 |
| sessionsAndGroups.ts | `src/tests/backend/specs/api/sessionsAndGroups.ts` | テスト | APIのテストケース |
