# 機能設計書 47-トークン検証

## 概要

本ドキュメントは、Etherpadにおけるトークン検証機能の詳細設計を記載したものである。この機能は、APIトークン（APIキーまたはOAuth2トークン）の有効性を検証するシンプルなAPI機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：
API連携において、クライアントは実際のAPI呼び出しを行う前に、保持しているAPIトークンが有効かどうかを確認したい場合がある。この機能により、クライアントは軽量な方法でトークンの有効性を確認でき、無効なトークンでの不必要なAPI呼び出しを避けることができる。

**機能の利用シーン**：
- API連携開始時にトークンの有効性を確認する場合
- 定期的なトークン有効性チェック
- デバッグ時のトークン検証
- クライアントアプリケーションの初期化時

**主要な処理内容**：
1. リクエストヘッダーまたはパラメータからトークンを取得
2. APIHandler側でトークンの認証処理（APIキーまたはOAuth2検証）
3. 認証成功時はok応答を返却
4. 認証失敗時はエラー応答を返却

**関連システム・外部連携**：
- APIキー認証システム
- OAuth2認証システム（SSO有効時）

**権限による制御**：
- 有効なAPIキーまたはOAuth2トークンが必要

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は直接的な画面連携はなく、APIとして提供される |

## 機能種別

認証確認（ヘルスチェック的機能）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| apikey / api_key / Authorization | string | Yes | APIキーまたはOAuth2トークン | 有効なトークンであること |

### 入力データソース

- REST API: `GET /api/2/checkToken`
- リクエストヘッダー（Authorization）またはクエリパラメータ（apikey）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | number | 処理結果コード（0: 成功、4: 認証エラー） |
| message | string | 処理結果メッセージ（"ok" または エラーメッセージ） |
| data | null | 常にnull |

### 出力先

- HTTP Response（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ RestAPI.ts で /api/2/checkToken エンドポイントを処理

2. 認証チェック
   └─ APIHandler.ts でトークンを検証
      ├─ APIキー認証モードの場合: apikey === APIKEY.txt内容
      └─ SSO認証モードの場合: JWTトークンを検証

3. 検証結果に応じた応答
   ├─ 成功: API.checkToken（空の関数）を呼び出し
   └─ 失敗: 401 Unauthorizedエラー

4. レスポンス返却
   └─ {code: 0, message: "ok", data: null} または エラー応答
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証モード?}
    B -->|APIキー| C{APIキー有効?}
    B -->|SSO| D{JWTトークン有効?}
    C -->|No| E[401 Unauthorized]
    C -->|Yes| F[checkToken呼び出し]
    D -->|No| E
    D -->|Yes| F
    F --> G[成功レスポンス返却]
    E --> H[エラーレスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-47-01 | APIキー認証 | APIKEY.txtの内容と一致すること | APIキー認証モード時 |
| BR-47-02 | OAuth2認証 | 有効なJWTトークンであること | SSO認証モード時 |
| BR-47-03 | admin権限チェック | authorization_codeフローの場合admin claim必須 | SSO認証モード時 |
| BR-47-04 | client_credentialsチェック | client_idがsso.clientsに含まれること | SSO認証モード時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし（本機能はデータベースを使用しない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 4 | 認証エラー | APIキーが無効または欠落 | 正しいAPIキーを指定 |
| 4 | 認証エラー | OAuth2トークンが無効 | 正しいトークンを指定 |
| 4 | 認証エラー | admin権限がない | 管理者権限を持つユーザーでログイン |

### リトライ仕様

トークンの問題を修正してからリトライすること

## トランザクション仕様

データベース操作なしのため該当なし

## パフォーマンス要件

- レスポンス時間: 50ms以内（ネットワーク除く）
- 処理自体は極めて軽量

## セキュリティ考慮事項

- トークンの有効性のみを確認し、トークン自体は返却しない
- 認証失敗時は具体的な理由を詳細に返却しない（セキュリティ上の理由）
- ブルートフォース攻撃対策としてレート制限との併用を推奨

## 備考

- checkToken関数自体は空の関数（何もしない）
- 実際の認証はAPIHandler.handleの共通認証処理で行われる
- APIバージョン1.2から利用可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: API関数を理解する

checkToken関数は空の関数である点を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | API.ts | `src/node/db/API.ts` | checkToken関数の定義 |

**読解のコツ**:
- **791-792行目**: checkToken関数は空の非同期関数
- 実際の認証処理はここでは行われない

#### Step 2: エントリーポイントを理解する

REST APIエンドポイントの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | GET /checkToken エンドポイントの定義 |

**主要処理フロー**:
1. **859-874行目**: GET /checkToken エンドポイントの定義
2. **860行目**: functionName: 'checkToken' の指定

#### Step 3: 認証処理を理解する

実際の認証処理はAPIHandlerで行われる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | APIHandler.ts | `src/node/handler/APIHandler.ts` | 認証処理の実装 |

**主要処理フロー**:
- **175-200行目**: 認証処理
- **175-180行目**: APIキー認証
- **182-199行目**: OAuth2認証（JWTトークン検証）

### プログラム呼び出し階層図

```
HTTP Request: GET /api/2/checkToken
    │
    ├─ RestAPI.ts: expressCreateServer (エンドポイント定義)
    │      └─ mapping.get(GET)!["/checkToken"]
    │
    ├─ APIHandler.ts: handle (認証処理)
    │      ├─ APIキー認証モード
    │      │     └─ fields.apikey === apikey
    │      │
    │      └─ SSO認証モード
    │            ├─ jwtDecode(token)
    │            └─ jwtVerify(token, publicKey)
    │
    └─ API.ts: checkToken (空の関数)
```

### データフロー図

```
[入力]                        [処理]                    [出力]

apikey / Authorization ───▶  APIHandler.handle  ───▶  {code: 0, message: "ok"}
                                   │                       または
                                   ▼                  {code: 4, message: "error"}
                              認証チェック
                                   │
                                   ▼
                              API.checkToken (何もしない)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| API.ts | `src/node/db/API.ts` | ソース | checkToken関数（空） |
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | RESTエンドポイント定義 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | 実際の認証処理 |
| APIKeyHandler.ts | `src/node/handler/APIKeyHandler.ts` | ソース | APIキー読み込み |
| OAuth2Provider.ts | `src/node/security/OAuth2Provider.ts` | ソース | OAuth2プロバイダー（公開鍵提供） |
