# 機能設計書 48-OAuth2認証

## 概要

本ドキュメントは、EtherpadにおけるOAuth2認証機能の詳細設計を記載したものである。この機能は、OpenID Connect準拠のOAuth2認証プロバイダーを提供し、ユーザー認証とアクセストークン発行を行う。

### 本機能の処理概要

**業務上の目的・背景**：
Etherpadを企業環境やセキュアな環境で運用する際、標準的な認証プロトコルであるOAuth2/OpenID Connectをサポートすることで、既存の認証基盤との統合やシングルサインオン（SSO）を実現できる。これにより、ユーザーは統一された認証体験を得られ、管理者は中央集権的なアクセス管理が可能となる。

**機能の利用シーン**：
- シングルサインオン（SSO）環境でのユーザー認証
- サードパーティアプリケーションからのAPI認証
- クライアント認証情報（client_credentials）フローでのマシン間通信
- 認可コード（authorization_code）フローでのユーザー認証

**主要な処理内容**：
1. OIDCプロバイダーの初期化とJWK鍵ペア生成
2. 認可エンドポイント（/oidc/auth）の提供
3. トークンエンドポイント（/oidc/token）の提供
4. ユーザー情報エンドポイント（/oidc/userinfo）の提供
5. アクセストークン・IDトークンの発行と検証

**関連システム・外部連携**：
- oidc-providerライブラリによるOIDC実装
- joseライブラリによるJWT署名・検証
- settings.jsonでのクライアント・ユーザー設定

**権限による制御**：
- is_admin: true のユーザーは管理者権限を持つ
- adminスコープによるAPI操作権限の制御

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | SSO ログイン画面 | 主機能 | OpenID Connect準拠のOAuth2認証フローを処理 |
| 5 | 同意画面 | 主機能 | OAuth2認証フローの一部として同意を取得 |

## 機能種別

認証・認可処理

## 入力仕様

### 認可エンドポイント（/oidc/auth）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| client_id | string | Yes | クライアントID | settings.sso.clientsに存在すること |
| redirect_uri | string | Yes | リダイレクトURI | クライアント設定と一致すること |
| response_type | string | Yes | レスポンスタイプ | code等 |
| scope | string | Yes | 要求スコープ | openid等 |
| state | string | No | 状態パラメータ | - |

### トークンエンドポイント（/oidc/token）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| grant_type | string | Yes | グラントタイプ | authorization_code / client_credentials |
| code | string | 条件付き | 認可コード | authorization_codeフロー時 |
| client_id | string | Yes | クライアントID | - |
| client_secret | string | Yes | クライアントシークレット | - |

### 入力データソース

- 認可エンドポイント: `GET /oidc/auth`
- トークンエンドポイント: `POST /oidc/token`
- ユーザー情報: `GET /oidc/userinfo`

## 出力仕様

### トークンレスポンス

| 項目名 | 型 | 説明 |
|--------|-----|------|
| access_token | string | アクセストークン（JWT形式） |
| token_type | string | トークンタイプ（Bearer） |
| expires_in | number | 有効期限（秒） |
| id_token | string | IDトークン（JWT形式） |
| refresh_token | string | リフレッシュトークン（オプション） |

### 出力先

- HTTP Response（JSON形式）
- リダイレクト（認可フロー時）

## 処理フロー

### 処理シーケンス（Authorization Codeフロー）

```
1. クライアントが認可エンドポイントにリダイレクト
   └─ /oidc/auth にパラメータ付きでリクエスト

2. ログイン画面表示
   └─ /interaction/:uid にリダイレクト
   └─ /views/login.html を表示

3. ユーザー認証
   └─ POST /interaction/:uid でログイン処理
   └─ settings.usersでユーザー名・パスワード検証

4. 同意画面表示（必要な場合）
   └─ /views/consent.html を表示

5. 同意処理
   └─ POST /interaction/:uid で同意処理
   └─ Grant作成・スコープ設定

6. 認可コード発行
   └─ redirect_uriに認可コードを付与してリダイレクト

7. トークン取得
   └─ クライアントがトークンエンドポイントに認可コードを送信
   └─ アクセストークン・IDトークンを発行

8. APIアクセス
   └─ アクセストークンを使用してAPI呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[認可リクエスト] --> B[ログイン画面]
    B --> C{認証成功?}
    C -->|No| D[エラー]
    C -->|Yes| E{同意必要?}
    E -->|Yes| F[同意画面]
    E -->|No| G[認可コード発行]
    F --> H{同意?}
    H -->|No| D
    H -->|Yes| G
    G --> I[リダイレクト]
    I --> J[トークン取得]
    J --> K[アクセストークン発行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-48-01 | ユーザー認証 | settings.usersに定義されたユーザーのみ認証可能 | ログイン時 |
| BR-48-02 | 管理者フラグ | is_admin: trueのユーザーはadminクレームを持つ | トークン発行時 |
| BR-48-03 | クライアント認証 | settings.sso.clientsに定義されたクライアントのみ許可 | 全フロー |
| BR-48-04 | スコープ制限 | openid, profile, email, adminスコープをサポート | トークン発行時 |
| BR-48-05 | JWT署名 | RS256アルゴリズムで署名 | トークン発行時 |

### 計算ロジック

- TTL設定: settings.ttlで各トークンの有効期限を設定可能

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| セッション保存 | LRUCache（メモリ） | INSERT/UPDATE | OIDCセッション情報 |
| Grant保存 | LRUCache（メモリ） | INSERT/UPDATE | 認可情報 |

### テーブル別操作詳細

OIDCAdapterによるLRUCacheベースのメモリストレージを使用

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| invalid_client | OAuth2エラー | クライアントIDが無効 | 正しいクライアントIDを使用 |
| invalid_grant | OAuth2エラー | 認可コードが無効または期限切れ | 再度認可フローを実行 |
| access_denied | OAuth2エラー | ユーザーが同意を拒否 | ユーザーに同意を求める |
| invalid_credentials | ログインエラー | ユーザー名またはパスワードが無効 | 正しい認証情報を入力 |

### リトライ仕様

認可コードは一度使用すると無効化されるため、再度認可フローを実行すること

## トランザクション仕様

LRUCacheベースのメモリストレージのため、サーバー再起動でセッションは失われる

## パフォーマンス要件

- 認可リクエスト: 100ms以内（リダイレクト除く）
- トークン発行: 200ms以内（署名処理含む）
- LRUCacheサイズ: 最大500エントリ、5000サイズ制限

## セキュリティ考慮事項

- RS256署名によるトークンの改ざん防止
- HTTPSでの通信を推奨
- client_secretの安全な保管が必要
- PKCE（Proof Key for Code Exchange）のサポート検討
- CORSの適切な設定

## 備考

- oidc-providerライブラリを使用したOpenID Connect準拠の実装
- メモリベースのアダプターを使用（本番環境では永続化検討）
- devInteractionsは無効化（カスタムログイン画面使用）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 設定構造を理解する

OAuth2プロバイダーの設定を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OAuth2Provider.ts | `src/node/security/OAuth2Provider.ts` | 設定オブジェクトの構造 |

**読解のコツ**:
- **13-63行目**: configurationオブジェクトの定義
- **15-48行目**: findAccount関数でのユーザー検索ロジック
- **49行目**: TTL設定
- **50-55行目**: クレーム定義

#### Step 2: プロバイダー初期化を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OAuth2Provider.ts | `src/node/security/OAuth2Provider.ts` | expressCreateServer関数 |

**主要処理フロー**:
- **72-78行目**: RSA鍵ペアの生成
- **80-154行目**: oidc-providerの初期化とオプション設定
- **119-151行目**: extraTokenClaimsでカスタムクレーム追加

#### Step 3: インタラクションエンドポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | OAuth2Provider.ts | `src/node/security/OAuth2Provider.ts` | ログイン・同意処理 |

**主要処理フロー**:
- **157-227行目**: POST /interaction/:uid（ログイン・同意処理）
- **167-189行目**: ログイン処理（prompt.name === 'login'）
- **191-221行目**: 同意処理（prompt.name === 'consent'）
- **230-259行目**: GET /interaction/:uid（画面表示）

#### Step 4: アダプターを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | OIDCAdapter.ts | `src/node/security/OIDCAdapter.ts` | メモリアダプター |

**主要処理フロー**:
- **5-21行目**: LRUCacheオプション設定
- **35-113行目**: MemoryAdapterクラスの実装

### プログラム呼び出し階層図

```
認可フロー:
GET /oidc/auth
    │
    ├─ oidc-provider: authorization endpoint
    │      └─ interactionDetails で状態確認
    │
    ├─ GET /interaction/:uid
    │      └─ リダイレクト: /views/login.html または /views/consent.html
    │
    └─ POST /interaction/:uid
           ├─ prompt.name === 'login'
           │      ├─ settings.users でユーザー検証
           │      └─ interactionFinished
           │
           └─ prompt.name === 'consent'
                  ├─ Grant 作成/更新
                  └─ interactionFinished

トークン発行:
POST /oidc/token
    │
    └─ oidc-provider: token endpoint
           ├─ 認可コード検証
           ├─ extraTokenClaims でカスタムクレーム追加
           └─ JWT署名・発行
```

### データフロー図

```
[入力]                              [処理]                           [出力]

認可リクエスト ───▶  oidc-provider  ───▶  認可コード
                          │
                          ▼
ユーザー認証情報 ───▶  /interaction/:uid  ───▶  セッション確立
                          │
                          ▼
認可コード ───▶  /oidc/token  ───▶  アクセストークン（JWT）
                          │
                          └─▶  IDトークン（JWT）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OAuth2Provider.ts | `src/node/security/OAuth2Provider.ts` | ソース | OIDCプロバイダー設定・初期化 |
| OIDCAdapter.ts | `src/node/security/OIDCAdapter.ts` | ソース | メモリベースのストレージアダプター |
| OAuth2User.ts | `src/node/security/OAuth2User.ts` | ソース | ユーザー情報管理 |
| login.html | `src/static/oidc/login.html` | 静的ファイル | ログイン画面 |
| consent.html | `src/static/oidc/consent.html` | 静的ファイル | 同意画面 |
| Settings.ts | `src/node/utils/Settings.ts` | ソース | 設定読み込み（sso設定含む） |
