# 機能設計書 5-グループ内パッド一覧取得

## 概要

本ドキュメントは、Etherpad APIにおける「グループ内パッド一覧取得」機能の設計仕様を記載する。

### 本機能の処理概要

指定されたグループに属するすべてのパッドIDを一覧で取得する機能である。グループ単位でのパッド管理やダッシュボード構築に利用される。

**業務上の目的・背景**：
グループ機能を利用してパッドを管理する場合、グループ内にどのパッドが存在するかを把握する必要がある。本機能により、特定グループのパッド一覧を取得でき、グループ管理画面の構築やグループ単位での操作（一括エクスポート等）が可能になる。

**機能の利用シーン**：
- グループ管理ダッシュボードでパッド一覧を表示する場合
- グループ内の全パッドに対して一括操作を行う前の確認
- グループの使用状況（パッド数）を把握する場合
- データ移行時にグループ内パッドを特定する場合

**主要な処理内容**：
1. グループの存在確認
2. グループレコードからパッド情報を取得
3. パッドIDの配列を生成
4. パッドID配列をクライアントに返却

**関連システム・外部連携**：
- ueberDB2データベースからのデータ取得

**権限による制御**：
APIキーまたはOAuth2トークンによる認証が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面からの直接操作はなく、API経由でのみ利用される |

## 機能種別

CRUD操作（Read）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| groupID | string | Yes | 対象のグループID | 存在するグループIDであること |
| apikey | string | Yes（APIキー認証時） | API認証キー | 設定ファイルのAPIキーと一致すること |
| authorization | string | Yes（OAuth2認証時） | Bearerトークン | 有効なJWTトークンであること |

### 入力データソース

HTTP GETリクエスト（`/api/2/groups/pads`）のクエリパラメータから取得。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | number | 処理結果コード（0: 成功） |
| message | string | 処理結果メッセージ（"ok"） |
| data.padIDs | string[] | パッドIDの配列 |

### 出力先

HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. API認証
   └─ APIキーまたはOAuth2トークンを検証
2. グループ存在確認
   └─ doesGroupExist(groupID)を呼び出し
   └─ 存在しなければエラー返却
3. パッド情報取得
   └─ db.getSub(`group:${groupID}`, ['pads'])
4. キー抽出
   └─ Object.keys(result) でパッドID配列を生成
5. レスポンス返却
   └─ {padIDs: [...]} を返却
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/2/groups/pads] --> B{認証検証}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[doesGroupExist groupID]
    D --> E{グループ存在?}
    E -->|No| F[400 groupID does not exist]
    E -->|Yes| G["db.getSub group:groupID, pads"]
    G --> H["padIDs = Object.keys(result)"]
    H --> I["返却 {padIDs}"]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 空配列返却 | グループ内にパッドがない場合は空配列を返す | パッドが未登録の場合 |
| BR-002 | 順序不定 | 返却されるパッドIDの順序は保証されない | 常に |

### 計算ロジック

- パッドID抽出: `Object.keys(result)`
- グループパッドIDの形式: `{groupID}${padName}`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| グループ存在確認 | group:{groupID} | SELECT | グループ実体確認 |
| パッド情報取得 | group:{groupID} | SELECT (getSub) | padsプロパティ取得 |

### テーブル別操作詳細

#### group:{groupID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | - | 存在確認 | doesGroupExist経由 |
| SELECT | pads | サブプロパティ取得 | getSub経由 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | パラメータエラー | groupIDが存在しない | 正しいgroupIDを指定 |
| 4 | 認証エラー | APIキーが無効または未指定 | 正しいAPIキーを指定 |
| 2 | 内部エラー | データベース操作失敗 | サーバーログを確認 |

### リトライ仕様

データベース操作失敗時のリトライはueberDB2層で処理される。

## トランザクション仕様

読み取り操作のみのため、トランザクション管理は不要。

## パフォーマンス要件

- レスポンス時間: 通常100ms以下
- パッド数に応じてレスポンスサイズが増加

## セキュリティ考慮事項

- APIキーまたはOAuth2トークンによる認証が必須
- グループ内パッドIDは公開されるが、パッドへのアクセスにはセッションが必要

## 備考

- パッドIDの形式: `{groupID}${padName}`
- 例: `g.s8oes9dhwrvt0zif$test`

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

グループ内パッドデータの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | GroupManager.ts | `src/node/db/GroupManager.ts` | group.padsの構造（{padID: 1, ...}形式） |

**読解のコツ**: グループの`pads`プロパティはオブジェクトで、各パッドIDがキー、値は1（存在フラグ）として保存される。パッドIDは`{groupID}${padName}`の形式。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | GET /groups/pads のルーティング |
| 2-2 | APIHandler.ts | `src/node/handler/APIHandler.ts` | listPadsパラメータ定義 |

**主要処理フロー**:
1. **349-362行目（RestAPI.ts）**: GET /groups/pads のルーティング定義
2. **39行目（APIHandler.ts）**: `listPads: ['groupID']` パラメータ定義

#### Step 3: コアロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | listPads関数のエクスポート |
| 3-2 | GroupManager.ts | `src/node/db/GroupManager.ts` | listPads関数の実装 |

**主要処理フロー**:
- **48行目（API.ts）**: `exports.listPads = groupManager.listPads;`
- **170-183行目（GroupManager.ts）**: listPads関数の実装
  - **171行目**: `doesGroupExist(groupID)` でグループ存在確認
  - **174-176行目**: グループ不在時のエラー処理
  - **179行目**: `db.getSub(\`group:${groupID}\`, ['pads'])` でパッド情報取得
  - **180行目**: `Object.keys(result)` でID配列生成

### プログラム呼び出し階層図

```
GET /api/2/groups/pads (RestAPI.ts)
    │
    ├─ APIHandler.handle() (APIHandler.ts:161)
    │      │
    │      ├─ 認証検証 (APIHandler.ts:175-200)
    │      │
    │      └─ api.listPads(groupID) (API.ts:48)
    │              │
    │              └─ groupManager.listPads() (GroupManager.ts:170)
    │                      │
    │                      ├─ doesGroupExist(groupID) (GroupManager.ts:85)
    │                      │      └─ db.get(`group:${groupID}`)
    │                      │
    │                      └─ db.getSub(`group:${groupID}`, ['pads'])
    │
    └─ Response: {code: 0, message: "ok", data: {padIDs: [...]}}
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP GET        ─────▶ RestAPI.ts
/api/2/groups/pads     ルーティング
                           │
groupID         ─────▶ APIHandler.ts
apikey/token           認証・パラメータ抽出
                           │
                       GroupManager.ts
                       │
                       ├─ doesGroupExist()
                       │
                       └─ db.getSub(pads)
                              │
                              ▼
                          Object.keys()
                              │
                              ▼
                       {padIDs: [...]} ─────▶ JSON Response
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | APIルーティング定義 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | API認証・パラメータ処理 |
| API.ts | `src/node/db/API.ts` | ソース | API関数エクスポート |
| GroupManager.ts | `src/node/db/GroupManager.ts` | ソース | グループ管理コアロジック |
| DB.ts | `src/node/db/DB.ts` | ソース | データベースアクセス層 |
