# 機能設計書 70-認証情報読み込み

## 概要

本ドキュメントは、Etherpadの認証情報読み込み機能の設計を記載する。この機能は、credentials.jsonから機密情報（データベース接続情報、ユーザー認証情報等）を読み込み、メイン設定とマージする。

### 本機能の処理概要

認証情報読み込み機能は、Etherpadの起動時にcredentials.jsonから機密性の高い設定を読み込む機能である。settings.jsonと同様にJSONコメントの除去、環境変数の展開、設定値の型変換を行い、メイン設定とマージする。

**業務上の目的・背景**：機密情報（データベースパスワード、APIキー、管理者パスワード等）をメイン設定ファイルから分離することで、セキュリティを向上させる。credentials.jsonは本番環境のみに配置し、settings.jsonはテンプレートとしてバージョン管理に含めることが可能となる。

**機能の利用シーン**：
- Etherpadサーバー起動時
- 管理画面からサーバー再起動した時
- データベース接続情報を分離管理する時
- ユーザー認証情報を分離管理する時

**主要な処理内容**：
1. credentials.jsonファイルの読み込み
2. JSONコメント（// や /* */）の除去
3. 環境変数プレースホルダー（${ENV_VAR}）の展開
4. 設定値の型変換（文字列→数値/ブール値）
5. メイン設定へのマージ（settings.jsonの値を上書き）

**関連システム・外部連携**：
- ファイルシステム（credentials.json読み込み）
- 環境変数（認証情報上書き）
- settings.json読み込み機能（連携動作）

**権限による制御**：認証情報ファイルの読み込みはサーバープロセスの権限で行われる。ファイルには適切なパーミッションを設定することを推奨。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | 設定管理画面 | 参照 | credentials.jsonから読み込んだ設定も含めて表示 |

## 機能種別

設定読み込み / ファイル操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| credentialsFilename | string | No | 認証情報ファイルパス | デフォルト: credentials.json |
| --credentials | string | No | コマンドライン引数 | ファイルパス |

### 入力データソース

- credentials.json（認証情報ファイル）
- 環境変数（認証情報上書き）
- コマンドライン引数（--credentials）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| credentials | object | 認証情報オブジェクト |

### 出力先

- グローバル設定オブジェクト（settings）にマージ

## 処理フロー

### 処理シーケンス

```
1. 認証情報ファイルパスの決定
   └─ コマンドライン引数 > デフォルト(credentials.json)
2. ファイル読み込み
   └─ fs.readFileSync()
3. JSONコメント除去
   └─ jsonminify()
4. JSON解析
   └─ JSON.parse()
5. 環境変数展開
   └─ lookupEnvironmentVariables()
6. 設定値のマージ
   └─ storeSettings()（メイン設定を上書き）
```

### フローチャート

```mermaid
flowchart TD
    A[reloadSettings呼び出し] --> B[parseSettings - settings.json]
    B --> C[parseSettings - credentials.json]
    C --> D[ファイル読み込み]
    D --> E{ファイル存在?}
    E -->|No| F[info出力・スキップ]
    E -->|Yes| G[jsonminify]
    G --> H[JSON.parse]
    H --> I[lookupEnvironmentVariables]
    I --> J[storeSettings - settings]
    J --> K[storeSettings - credentials]
    K --> L[設定マージ完了]
    F --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-70-01 | ファイル不存在時の動作 | credentials.jsonが存在しない場合はinfo出力のみで処理継続 | ファイル未検出時 |
| BR-70-02 | 設定上書き | credentials.jsonの設定がsettings.jsonの同名設定を上書き | 同一キー存在時 |
| BR-70-03 | 環境変数展開 | ${ENV_VAR}形式でプレースホルダーを環境変数値に置換 | 設定値が文字列の場合 |
| BR-70-04 | デフォルト値指定 | ${ENV_VAR:default}形式でデフォルト値を指定可能 | 環境変数未設定の場合 |
| BR-70-05 | 型変換 | "true"/"false"は真偽値、数値文字列は数値に変換 | 環境変数展開時 |

### 計算ロジック

認証情報ファイルパス決定:
```typescript
credentialsFilename: absolutePaths.makeAbsolute(argv.credentials || 'credentials.json')
```

設定マージ順序:
```typescript
// settings.jsonを先に読み込み
const settingsParsed = parseSettings(settings.settingsFilename, true);
// credentials.jsonを後に読み込み（上書き）
const credentials = parseSettings(settings.credentialsFilename, false);
storeSettings(settingsParsed);
storeSettings(credentials);
```

## データベース操作仕様

本機能はデータベースを操作しない。ただし、credentials.jsonにはデータベース接続情報（dbSettings）を記述可能。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ファイル未検出 | credentials.jsonが存在しない | info出力、スキップ（正常動作） |
| - | JSON解析エラー | 無効なJSON形式 | エラー出力、プロセス終了 |
| - | 未知の設定 | 定義されていない設定名 | 警告出力、無視 |

### リトライ仕様

エラー時は自動リトライなし。

## トランザクション仕様

ファイル読み込みのため、トランザクション制御は不要。

## パフォーマンス要件

- 認証情報読み込み: 起動時に1回のみ、または再起動時

## セキュリティ考慮事項

- credentials.jsonには機密情報が含まれるため、ファイルパーミッションを600（所有者のみ読み書き可能）に設定することを推奨
- 環境変数での上書きにより機密情報をファイルに残さない運用が可能
- バージョン管理（Git等）にcredentials.jsonを含めないことを推奨（.gitignoreに追加）
- ログ出力時に機密情報が出力されないよう注意が必要

## 備考

- Etherpad 1.6.0以降でcredentials.json機能が追加された
- settings.jsonにも同じ設定を記述可能（後方互換性）
- Docker環境では環境変数での設定上書きを推奨

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 認証情報ファイルパスの決定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Settings.ts | `src/node/utils/Settings.ts` | credentialsFilenameの定義（302行目） |

**読解のコツ**: コマンドライン引数`--credentials`または デフォルトの`credentials.json`からパスが決定される。

#### Step 2: 認証情報読み込み処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Settings.ts | `src/node/utils/Settings.ts` | reloadSettings関数（940-944行目） |
| 2-2 | Settings.ts | `src/node/utils/Settings.ts` | parseSettings関数（92-131行目） |

**主要処理フロー**:
- **940-942行目**: reloadSettings関数 - settings.jsonとcredentials.jsonの読み込み
- **942行目**: parseSettings(settings.credentialsFilename, false) - credentials.json読み込み
- **944行目**: storeSettings(credentials) - メイン設定へのマージ
- **101-105行目**: isSettings=falseの場合の処理（credentials用）

#### Step 3: 認証情報ファイル未検出時の動作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Settings.ts | `src/node/utils/Settings.ts` | parseSettings内のエラーハンドリング（101-105, 110-115行目） |

**主要処理フロー**:
- **101-105行目**: credentials.jsonの場合はnotFoundFunction=logger.info、notFoundMessage='Ignoring.'
- **110-115行目**: ファイル未検出時はinfo出力してnull返却（エラーにならない）

#### Step 4: 設定マージ処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Settings.ts | `src/node/utils/Settings.ts` | storeSettings関数（712-740行目） |

**主要処理フロー**:
- **712-740行目**: storeSettings関数 - 設定のマージ
- **727-734行目**: 既存設定の上書きまたはオブジェクトのマージ
- **728-730行目**: オブジェクト型の場合は_.defaultsでマージ
- **732-733行目**: その他の型は直接上書き

#### Step 5: 格納可能な認証情報の型を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Settings.ts | `src/node/utils/Settings.ts` | SettingsType型定義（159-296行目） |

**主要処理フロー**:
- **190-191行目**: dbType, dbSettings（データベース接続情報）
- **261行目**: users（ユーザー認証情報）
- **262-265行目**: sso（SSO設定）

### プログラム呼び出し階層図

```
Etherpad起動 / 管理画面からの再起動
    │
    └─ reloadSettings()
           │
           ├─ parseSettings(settingsFilename, true)   [settings.json]
           │
           ├─ parseSettings(credentialsFilename, false) [credentials.json]
           │      │
           │      ├─ fs.readFileSync()
           │      │      └─ ファイル未検出時: logger.info → return null
           │      │
           │      ├─ jsonminify()
           │      ├─ JSON.parse()
           │      └─ lookupEnvironmentVariables()
           │             │
           │             ├─ replaceEnvs() [再帰処理]
           │             └─ coerceValue() [型変換]
           │
           ├─ storeSettings(settingsParsed) [settings.json適用]
           │
           └─ storeSettings(credentials)    [credentials.json上書き]
                  │
                  └─ 各認証情報のマージ
                        ├─ dbSettings（データベース）
                        ├─ users（ユーザー情報）
                        └─ その他の機密設定
```

### データフロー図

```
[入力]               [処理]                    [出力]

credentials.json ───▶ parseSettings(false)
                           │
                           ▼
                    jsonminify()
                           │
                           ▼
                    JSON.parse()
                           │
                           ▼
           lookupEnvironmentVariables()
                           │
    ┌──────────────────────┼──────────────────────┐
    │                      │                      │
    ▼                      ▼                      ▼
環境変数           デフォルト値            型変換
    │                      │                      │
    └──────────────────────┼──────────────────────┘
                           │
                           ▼
                  credentials object
                           │
                           ▼
settings.json ───▶ storeSettings(settingsParsed)
                           │
                           ▼
credentials ───────▶ storeSettings(credentials)
                           │
                           ▼
                    settings ───▶ グローバル設定オブジェクト
                                    │
                                    ├─ dbSettings（DB接続情報）
                                    ├─ users（ユーザー情報）
                                    └─ その他設定
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Settings.ts | `src/node/utils/Settings.ts` | ソース | 設定管理本体（認証情報含む） |
| Cli.ts | `src/node/utils/Cli.ts` | ソース | コマンドライン引数解析（--credentials） |
| AbsolutePaths.ts | `src/node/utils/AbsolutePaths.ts` | ソース | パス解決 |
| credentials.json | ルートディレクトリ | 設定 | 認証情報ファイル（要作成） |
| settings.json.template | ルートディレクトリ | テンプレート | 設定テンプレート（credentials.jsonの説明含む） |
