# 機能設計書 9-パッドコピー

## 概要

本ドキュメントは、Etherpad APIにおける「パッドコピー」機能の設計仕様を記載する。

### 本機能の処理概要

既存のパッドを別のIDにコピーする機能である。全リビジョン、チャット履歴、著者情報を含む完全なコピーが作成される。オプションで既存のパッドを強制的に上書きすることも可能。

**業務上の目的・背景**：
ドキュメントのバックアップ、テンプレートからの新規作成、パッドのリネーム（コピー＋削除）などに利用される。完全な履歴を保持したまま複製できるため、監査やバージョン管理の目的にも適している。

**機能の利用シーン**：
- パッドのバックアップを作成する場合
- テンプレートパッドから新しいパッドを作成する場合
- パッドを別のグループに移動する前準備としてコピーする場合
- 特定時点のパッド状態を保存する場合

**主要な処理内容**：
1. コピー元パッドの存在確認
2. コピー先がグループパッドの場合、グループ存在確認
3. コピー先に既存パッドがある場合の処理（forceオプション）
4. パッド本体レコードのコピー
5. 全リビジョンのコピー
6. 全チャットメッセージのコピー
7. 著者情報のコピー（新パッドを著者のパッドリストに追加）
8. グループパッドの場合、グループへの登録
9. コピー先パッドのキャッシュ初期化

**関連システム・外部連携**：
- ueberDB2データベースによるデータコピー
- フック機構（padCopy）

**権限による制御**：
APIキーまたはOAuth2トークンによる認証が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面からの直接操作はなく、API経由でのみ利用される |

## 機能種別

CRUD操作（Create） - 複製処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| sourceID | string | Yes | コピー元パッドID | 存在するパッドIDであること |
| destinationID | string | Yes | コピー先パッドID | 有効なパッドIDであること |
| force | boolean | No | 既存パッドの上書きを許可 | true/false（デフォルト: false） |
| apikey | string | Yes（APIキー認証時） | API認証キー | 設定ファイルのAPIキーと一致すること |

### 入力データソース

HTTP POSTリクエスト（`/api/2/pads/copypad`）のボディから取得。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | number | 処理結果コード（0: 成功） |
| message | string | 処理結果メッセージ（"ok"） |
| data.padID | string | コピー先パッドID |

### 出力先

HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. API認証
   └─ APIキーまたはOAuth2トークンを検証
2. コピー元パッド取得
   └─ getPadSafe(sourceID, true)で存在確認と取得
3. コピー元パッドをDBにフラッシュ
   └─ pad.saveToDatabase()
4. コピー先グループ確認（グループパッドの場合）
   └─ $を含むdestinationIDの場合、グループ存在確認
5. 既存パッド処理
   └─ force=trueで既存パッドがある場合は削除
   └─ force=falseで既存パッドがある場合はエラー
6. レコードコピー（並列・バッチ処理）
   └─ pad:{padID} → pad:{destinationID}
   └─ pad:{padID}:revs:{0..head} → 全てコピー
   └─ pad:{padID}:chat:{0..chatHead} → 全てコピー
7. 著者情報コピー
   └─ 全著者のパッドリストにdestinationIDを追加
8. グループ登録（グループパッドの場合）
   └─ db.setSub(`group:${destGroupID}`, ['pads', destinationID], 1)
9. コピー先パッド初期化
   └─ padManager.getPad(destinationID)でキャッシュ登録
10. フック実行
    └─ padCopy フックを実行
11. レスポンス返却
    └─ {padID: destinationID}
```

### フローチャート

```mermaid
flowchart TD
    A[POST /api/2/pads/copypad] --> B{認証検証}
    B -->|失敗| C[401 Unauthorized]
    B -->|成功| D[getPadSafe sourceID, true]
    D --> E{パッド存在?}
    E -->|No| F[400 padID does not exist]
    E -->|Yes| G[saveToDatabase]
    G --> H{destinationIDに$?}
    H -->|Yes| I[グループ存在確認]
    H -->|No| J{destinationID存在?}
    I --> K{グループ存在?}
    K -->|No| L[400 groupID does not exist]
    K -->|Yes| J
    J -->|Yes| M{force=true?}
    J -->|No| N[レコードコピー]
    M -->|No| O[400 destinationID already exists]
    M -->|Yes| P[既存パッド削除]
    P --> N
    N --> Q[著者情報コピー]
    Q --> R{グループパッド?}
    R -->|Yes| S[グループに登録]
    R -->|No| T[padManager.getPad]
    S --> T
    T --> U[padCopy hook]
    U --> V["返却 {padID}"]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 完全コピー | 全リビジョン・チャットをコピー | 常に |
| BR-002 | 上書き制御 | force=trueの場合のみ既存パッドを上書き | 常に |
| BR-003 | グループ検証 | コピー先グループが存在しないとエラー | グループパッドへコピー時 |
| BR-004 | 著者継承 | コピー元の著者全員にコピー先パッドを関連付け | 常に |
| BR-005 | バッチ処理 | 100件ずつのバッチで並列コピー | 常に |

### 計算ロジック

- グループID抽出: `destinationID.split('$')[0]`
- バッチサイズ: 100（.batch(100).buffer(99)）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| コピー元取得 | pad:{sourceID} | SELECT | パッド本体取得 |
| グループ確認 | group:{groupID} | SELECT | グループ存在確認 |
| 既存削除 | pad:{destinationID}等 | DELETE | force時の削除 |
| パッドコピー | pad:{destinationID} | INSERT | パッド本体コピー |
| リビジョンコピー | pad:{destinationID}:revs:{n} | INSERT | リビジョンコピー |
| チャットコピー | pad:{destinationID}:chat:{n} | INSERT | チャットコピー |
| 著者更新 | author:{authorID} | UPDATE | パッドリスト追加 |
| グループ更新 | group:{groupID} | UPDATE | padsに追加 |

### テーブル別操作詳細

#### pad:{destinationID}

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | 全項目 | コピー元と同一 | パッドIDのみ異なる |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | パラメータエラー | sourceIDが存在しない | 正しいsourceIDを指定 |
| 1 | パラメータエラー | destinationIDのグループが存在しない | 正しいgroupIDを指定 |
| 1 | パラメータエラー | destinationIDが既に存在（force=false） | force=trueまたは別のIDを指定 |
| 4 | 認証エラー | APIキーが無効 | 正しいAPIキーを指定 |
| 2 | 内部エラー | データベース操作失敗 | サーバーログを確認 |

### リトライ仕様

データベース操作失敗時のリトライはueberDB2層で処理される。

## トランザクション仕様

完全なトランザクション管理は行われない。コピー処理はStream APIによるバッチ処理で実行される。途中で失敗した場合、部分的なコピー状態になる可能性がある。

## パフォーマンス要件

- レスポンス時間: リビジョン数・チャット数に依存（数秒〜数十秒）
- バッチ処理により大量データでもメモリ効率的に処理

## セキュリティ考慮事項

- APIキーまたはOAuth2トークンによる認証が必須
- コピー先グループへのアクセス権限はセッションで制御
- force=trueの場合、既存データが完全に上書きされる

## 備考

- APIバージョン1.2.9から利用可能
- 履歴なしコピーが必要な場合はcopyPadWithoutHistory APIを使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コピー対象となるデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Pad.ts | `src/node/db/Pad.ts` | Padクラスの構造とcopy()メソッド |

**読解のコツ**: コピー対象は3種類：パッド本体（`pad:{padID}`）、リビジョン（`pad:{padID}:revs:{n}`）、チャット（`pad:{padID}:chat:{n}`）。Stream APIでバッチ処理される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RestAPI.ts | `src/node/handler/RestAPI.ts` | POST /pads/copypad のルーティング |
| 2-2 | APIHandler.ts | `src/node/handler/APIHandler.ts` | copyPadパラメータ定義 |

**主要処理フロー**:
1. **1035-1061行目（RestAPI.ts）**: POST /pads/copypad のルーティング定義
2. **97行目（APIHandler.ts）**: `copyPad: ['sourceID', 'destinationID', 'force']` パラメータ定義

#### Step 3: コアロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | API.ts | `src/node/db/API.ts` | copyPad関数の実装 |
| 3-2 | Pad.ts | `src/node/db/Pad.ts` | copy関数の実装 |

**主要処理フロー**:
- **615-618行目（API.ts）**: copyPad関数の実装
  - **616行目**: `getPadSafe(sourceID, true)` でパッド取得
  - **617行目**: `pad.copy(destinationID, force)` でコピー実行
- **403-454行目（Pad.ts）**: copy関数の実装
  - **410行目**: `saveToDatabase()` でフラッシュ
  - **413行目**: `checkIfGroupExistAndReturnIt()` でグループ確認
  - **416行目**: `removePadIfForceIsTrueAndAlreadyExist()` で既存削除
  - **418-421行目**: copyRecord関数定義
  - **423-432行目**: ジェネレータでコピー対象を生成
  - **433行目**: `.batch(100).buffer(99)` でバッチ処理
  - **436行目**: `padManager.getPad()` でキャッシュ初期化
  - **439-451行目**: padCopyフック実行

### プログラム呼び出し階層図

```
POST /api/2/pads/copypad (RestAPI.ts)
    │
    ├─ APIHandler.handle() (APIHandler.ts:161)
    │      │
    │      ├─ 認証検証 (APIHandler.ts:175-200)
    │      │
    │      └─ api.copyPad(sourceID, destinationID, force) (API.ts:615)
    │              │
    │              ├─ getPadSafe(sourceID, true)
    │              │
    │              └─ pad.copy(destinationID, force) (Pad.ts:403)
    │                      │
    │                      ├─ saveToDatabase()
    │                      │
    │                      ├─ checkIfGroupExistAndReturnIt()
    │                      │      └─ groupManager.doesGroupExist()
    │                      │
    │                      ├─ removePadIfForceIsTrueAndAlreadyExist()
    │                      │      └─ pad.remove()
    │                      │
    │                      ├─ Stream.batch(100).buffer(99)
    │                      │      ├─ copyRecord('') [pad本体]
    │                      │      ├─ copyRecord(':revs:0') × head
    │                      │      └─ copyRecord(':chat:0') × chatHead
    │                      │
    │                      ├─ copyAuthorInfoToDestinationPad()
    │                      │
    │                      ├─ [グループ] db.setSub(group, pads)
    │                      │
    │                      ├─ padManager.getPad(destinationID)
    │                      │
    │                      └─ hooks.aCallAll('padCopy')
    │
    └─ Response: {code: 0, message: "ok", data: {padID}}
```

### データフロー図

```
[入力]                    [処理]                              [出力]

HTTP POST            ─────▶ RestAPI.ts
/api/2/pads/copypad         ルーティング
                                │
sourceID             ─────▶ API.ts
destinationID               getPadSafe()
force                           │
                            Pad.ts copy()
                            │
                            ├─ saveToDatabase()
                            │
                            ├─ checkGroupExist()
                            │
                            ├─ [force] removePad()
                            │
                            ├─ Stream.batch() [並列コピー]
                            │      ├─ pad本体
                            │      ├─ revs × N
                            │      └─ chat × M
                            │
                            ├─ copyAuthorInfo()
                            │
                            ├─ [グループ] グループ登録
                            │
                            └─ padManager.getPad()
                                    │
                                    ▼
                            {padID} ─────▶ JSON Response
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| RestAPI.ts | `src/node/handler/RestAPI.ts` | ソース | APIルーティング定義 |
| APIHandler.ts | `src/node/handler/APIHandler.ts` | ソース | API認証・パラメータ処理 |
| API.ts | `src/node/db/API.ts` | ソース | copyPad実装 |
| Pad.ts | `src/node/db/Pad.ts` | ソース | copy実装 |
| PadManager.ts | `src/node/db/PadManager.ts` | ソース | キャッシュ管理 |
| GroupManager.ts | `src/node/db/GroupManager.ts` | ソース | グループ存在確認 |
| AuthorManager.ts | `src/node/db/AuthorManager.ts` | ソース | 著者パッドリスト更新 |
| Stream.ts | `src/node/utils/Stream.ts` | ソース | バッチ処理ユーティリティ |
| DB.ts | `src/node/db/DB.ts` | ソース | データベースアクセス層 |
