# API設計書

## 概要

本ドキュメントは、FastAPIフレームワーク（バージョン0.128.0）が提供するAPI機能の設計仕様を記載したものです。FastAPIは、Python 3.8+で高性能なWeb APIを構築するためのモダンなWebフレームワークであり、Starlette（ASGIフレームワーク）とPydantic（データバリデーション）を基盤としています。

本設計書では、FastAPIが提供するコア機能、パラメータ処理、セキュリティ機構、レスポンス形式、およびOpenAPI自動生成機能について詳細に記載します。

## 共通仕様

### ベースURL

```
http://localhost:8000
```

FastAPIアプリケーションのデフォルトURL。実際の運用時は環境に応じて変更されます。

### 認証方式

FastAPIは以下の認証方式をサポートしています:

| 認証方式 | クラス名 | 説明 |
| --- | --- | --- |
| HTTP Basic認証 | `HTTPBasic` | ユーザー名とパスワードによる基本認証（RFC 7617） |
| HTTP Bearer Token | `HTTPBearer` | Bearerトークンによる認証 |
| HTTP Digest認証 | `HTTPDigest` | Digest認証方式 |
| API Key (Header) | `APIKeyHeader` | HTTPヘッダーでAPIキーを送信 |
| API Key (Query) | `APIKeyQuery` | クエリパラメータでAPIキーを送信 |
| API Key (Cookie) | `APIKeyCookie` | CookieでAPIキーを送信 |
| OAuth2 Password Bearer | `OAuth2PasswordBearer` | OAuth2パスワードフローによる認証 |
| OAuth2 Authorization Code | `OAuth2AuthorizationCodeBearer` | OAuth2認可コードフローによる認証 |
| OpenID Connect | `OpenIdConnect` | OpenID Connect認証 |

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Authorization | - | 認証トークン（認証が必要なエンドポイントのみ） |
| Content-Type | ○ | application/json（リクエストボディがある場合） |
| Accept | - | application/json |

### 共通エラーレスポンス

| ステータスコード | 説明 | 例外クラス |
| --- | --- | --- |
| 400 | Bad Request - リクエストパラメータ不正 | `HTTPException` |
| 401 | Unauthorized - 認証エラー | `HTTPException` |
| 403 | Forbidden - 権限エラー | `HTTPException` |
| 404 | Not Found - リソース未検出 | `HTTPException` |
| 422 | Unprocessable Entity - バリデーションエラー | `RequestValidationError` |
| 500 | Internal Server Error - サーバーエラー | `ResponseValidationError` |

### エラーレスポンス形式

```json
{
  "detail": [
    {
      "type": "missing",
      "loc": ["query", "item_id"],
      "msg": "Field required",
      "input": null
    }
  ]
}
```

## パラメータタイプ一覧

FastAPIは以下のパラメータタイプをサポートしています:

| パラメータタイプ | 関数名 | 説明 |
| --- | --- | --- |
| Path Parameter | `Path()` | URLパスに含まれるパラメータ（必須） |
| Query Parameter | `Query()` | URLクエリ文字列のパラメータ |
| Header Parameter | `Header()` | HTTPヘッダーのパラメータ |
| Cookie Parameter | `Cookie()` | Cookieのパラメータ |
| Request Body | `Body()` | リクエストボディのJSONデータ |
| Form Data | `Form()` | フォームデータ（application/x-www-form-urlencoded） |
| File Upload | `File()` | ファイルアップロード（multipart/form-data） |

## HTTPメソッド一覧

FastAPIは以下のHTTPメソッドをサポートしています:

| メソッド | デコレータ | 説明 |
| --- | --- | --- |
| GET | `@app.get()` | リソースの取得 |
| POST | `@app.post()` | リソースの作成 |
| PUT | `@app.put()` | リソースの更新（全体） |
| PATCH | `@app.patch()` | リソースの部分更新 |
| DELETE | `@app.delete()` | リソースの削除 |
| OPTIONS | `@app.options()` | 通信オプションの確認 |
| HEAD | `@app.head()` | ヘッダー情報のみ取得 |
| TRACE | `@app.trace()` | リクエストのループバック |

## 自動生成エンドポイント

FastAPIは以下のエンドポイントを自動生成します:

| エンドポイント | メソッド | 説明 |
| --- | --- | --- |
| `/openapi.json` | GET | OpenAPI 3.1.0スキーマ（JSON） |
| `/docs` | GET | Swagger UI（対話的APIドキュメント） |
| `/redoc` | GET | ReDoc（APIドキュメント） |

## API一覧

### パラメータ関数

| カテゴリ | 関数 | メソッド | 説明 |
| --- | --- | --- | --- |
| パラメータ | Path() | - | パスパラメータの定義 |
| パラメータ | Query() | - | クエリパラメータの定義 |
| パラメータ | Header() | - | ヘッダーパラメータの定義 |
| パラメータ | Cookie() | - | Cookieパラメータの定義 |
| パラメータ | Body() | - | リクエストボディの定義 |
| パラメータ | Form() | - | フォームデータの定義 |
| パラメータ | File() | - | ファイルアップロードの定義 |
| 依存性 | Depends() | - | 依存性注入 |
| セキュリティ | Security() | - | セキュリティスコープ付き依存性注入 |

### セキュリティ関数

| カテゴリ | クラス | 説明 |
| --- | --- | --- |
| HTTP認証 | HTTPBasic | HTTP Basic認証 |
| HTTP認証 | HTTPBearer | Bearer Token認証 |
| HTTP認証 | HTTPDigest | Digest認証 |
| APIキー | APIKeyHeader | ヘッダーによるAPIキー認証 |
| APIキー | APIKeyQuery | クエリパラメータによるAPIキー認証 |
| APIキー | APIKeyCookie | CookieによるAPIキー認証 |
| OAuth2 | OAuth2PasswordBearer | OAuth2パスワードフロー |
| OAuth2 | OAuth2PasswordRequestForm | OAuth2パスワードリクエストフォーム |
| OAuth2 | OAuth2AuthorizationCodeBearer | OAuth2認可コードフロー |
| OAuth2 | SecurityScopes | セキュリティスコープ |
| OpenID | OpenIdConnect | OpenID Connect |

## 各APIエンドポイント定義

### パラメータ定義

#### 1. Path() - パスパラメータ

URLパス内の変数部分を定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `Path()` |
| インポート | `from fastapi import Path` |
| パラメータタイプ | path |
| デフォルト値 | 必須（`...`） |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| default | Any | - | `...` | デフォルト値（パスパラメータは常に必須） |
| alias | str | - | None | 別名 |
| title | str | - | None | 人間可読なタイトル |
| description | str | - | None | 説明文 |
| gt | float | - | None | より大きい（数値のみ） |
| ge | float | - | None | 以上（数値のみ） |
| lt | float | - | None | より小さい（数値のみ） |
| le | float | - | None | 以下（数値のみ） |
| min_length | int | - | None | 最小文字数（文字列のみ） |
| max_length | int | - | None | 最大文字数（文字列のみ） |
| pattern | str | - | None | 正規表現パターン（文字列のみ） |
| examples | list[Any] | - | None | サンプル値リスト |
| deprecated | bool | - | None | 非推奨フラグ |
| include_in_schema | bool | - | True | OpenAPIスキーマに含めるか |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, Path

app = FastAPI()

@app.get("/items/{item_id}")
async def read_item(
    item_id: Annotated[int, Path(title="アイテムID", ge=1)]
):
    return {"item_id": item_id}
```

---

#### 2. Query() - クエリパラメータ

URLクエリ文字列のパラメータを定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `Query()` |
| インポート | `from fastapi import Query` |
| パラメータタイプ | query |
| デフォルト値 | 任意 |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| default | Any | - | Undefined | デフォルト値 |
| alias | str | - | None | 別名 |
| title | str | - | None | 人間可読なタイトル |
| description | str | - | None | 説明文 |
| gt | float | - | None | より大きい（数値のみ） |
| ge | float | - | None | 以上（数値のみ） |
| lt | float | - | None | より小さい（数値のみ） |
| le | float | - | None | 以下（数値のみ） |
| min_length | int | - | None | 最小文字数 |
| max_length | int | - | None | 最大文字数 |
| pattern | str | - | None | 正規表現パターン |
| examples | list[Any] | - | None | サンプル値リスト |
| deprecated | bool | - | None | 非推奨フラグ |
| include_in_schema | bool | - | True | OpenAPIスキーマに含めるか |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, Query

app = FastAPI()

@app.get("/items/")
async def read_items(
    q: Annotated[str | None, Query(max_length=50, description="検索クエリ")] = None
):
    return {"q": q}
```

---

#### 3. Header() - ヘッダーパラメータ

HTTPヘッダーのパラメータを定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `Header()` |
| インポート | `from fastapi import Header` |
| パラメータタイプ | header |
| デフォルト値 | 任意 |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, Header

app = FastAPI()

@app.get("/items/")
async def read_items(
    user_agent: Annotated[str | None, Header()] = None
):
    return {"User-Agent": user_agent}
```

---

#### 4. Cookie() - Cookieパラメータ

Cookieのパラメータを定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `Cookie()` |
| インポート | `from fastapi import Cookie` |
| パラメータタイプ | cookie |
| デフォルト値 | 任意 |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, Cookie

app = FastAPI()

@app.get("/items/")
async def read_items(
    session_id: Annotated[str | None, Cookie()] = None
):
    return {"session_id": session_id}
```

---

#### 5. Body() - リクエストボディ

リクエストボディのJSONデータを定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `Body()` |
| インポート | `from fastapi import Body` |
| Content-Type | application/json |
| デフォルト値 | 任意 |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, Body

app = FastAPI()

@app.post("/items/")
async def create_item(
    importance: Annotated[int, Body()]
):
    return {"importance": importance}
```

---

#### 6. Form() - フォームデータ

フォームデータを定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `Form()` |
| インポート | `from fastapi import Form` |
| Content-Type | application/x-www-form-urlencoded |
| デフォルト値 | 任意 |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, Form

app = FastAPI()

@app.post("/login/")
async def login(
    username: Annotated[str, Form()],
    password: Annotated[str, Form()]
):
    return {"username": username}
```

---

#### 7. File() - ファイルアップロード

ファイルアップロードを定義します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数名 | `File()` |
| インポート | `from fastapi import File` |
| Content-Type | multipart/form-data |
| 戻り値型 | `UploadFile` または `bytes` |

**使用例**

```python
from typing import Annotated
from fastapi import FastAPI, File, UploadFile

app = FastAPI()

@app.post("/uploadfile/")
async def create_upload_file(
    file: Annotated[UploadFile, File(description="アップロードするファイル")]
):
    return {"filename": file.filename}
```

---

### セキュリティ機能

#### 8. HTTPBasic - HTTP Basic認証

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `HTTPBasic` |
| インポート | `from fastapi.security import HTTPBasic, HTTPBasicCredentials` |
| 認証ヘッダー | `Authorization: Basic <credentials>` |
| 戻り値型 | `HTTPBasicCredentials` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| scheme_name | str | - | None | セキュリティスキーム名 |
| realm | str | - | None | HTTP Basic認証のrealm |
| description | str | - | None | 説明文 |
| auto_error | bool | - | True | 認証失敗時に自動でエラーを返すか |

**使用例**

```python
from typing import Annotated
from fastapi import Depends, FastAPI
from fastapi.security import HTTPBasic, HTTPBasicCredentials

app = FastAPI()
security = HTTPBasic()

@app.get("/users/me")
def read_current_user(
    credentials: Annotated[HTTPBasicCredentials, Depends(security)]
):
    return {"username": credentials.username}
```

---

#### 9. HTTPBearer - Bearer Token認証

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `HTTPBearer` |
| インポート | `from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials` |
| 認証ヘッダー | `Authorization: Bearer <token>` |
| 戻り値型 | `HTTPAuthorizationCredentials` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| bearerFormat | str | - | None | Bearerトークンのフォーマット |
| scheme_name | str | - | None | セキュリティスキーム名 |
| description | str | - | None | 説明文 |
| auto_error | bool | - | True | 認証失敗時に自動でエラーを返すか |

**使用例**

```python
from typing import Annotated
from fastapi import Depends, FastAPI
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials

app = FastAPI()
security = HTTPBearer()

@app.get("/items/")
def read_items(
    credentials: Annotated[HTTPAuthorizationCredentials, Depends(security)]
):
    return {"token": credentials.credentials}
```

---

#### 10. OAuth2PasswordBearer - OAuth2パスワードフロー

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `OAuth2PasswordBearer` |
| インポート | `from fastapi.security import OAuth2PasswordBearer` |
| 認証ヘッダー | `Authorization: Bearer <token>` |
| 戻り値型 | `str`（トークン文字列） |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| tokenUrl | str | ○ | - | トークン取得エンドポイントURL |
| scheme_name | str | - | None | セキュリティスキーム名 |
| scopes | dict | - | None | 利用可能なスコープ |
| description | str | - | None | 説明文 |
| auto_error | bool | - | True | 認証失敗時に自動でエラーを返すか |

**使用例**

```python
from typing import Annotated
from fastapi import Depends, FastAPI
from fastapi.security import OAuth2PasswordBearer

app = FastAPI()
oauth2_scheme = OAuth2PasswordBearer(tokenUrl="token")

@app.get("/items/")
async def read_items(
    token: Annotated[str, Depends(oauth2_scheme)]
):
    return {"token": token}
```

---

#### 11. OAuth2PasswordRequestForm - OAuth2パスワードリクエストフォーム

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `OAuth2PasswordRequestForm` |
| インポート | `from fastapi.security import OAuth2PasswordRequestForm` |
| Content-Type | application/x-www-form-urlencoded |

**フォームフィールド**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| grant_type | str | - | 認可タイプ（"password"） |
| username | str | ○ | ユーザー名 |
| password | str | ○ | パスワード |
| scope | str | - | スコープ（スペース区切り） |
| client_id | str | - | クライアントID |
| client_secret | str | - | クライアントシークレット |

**使用例**

```python
from typing import Annotated
from fastapi import Depends, FastAPI
from fastapi.security import OAuth2PasswordRequestForm

app = FastAPI()

@app.post("/token")
async def login(
    form_data: Annotated[OAuth2PasswordRequestForm, Depends()]
):
    return {
        "access_token": form_data.username,
        "token_type": "bearer"
    }
```

---

### レスポンスクラス

FastAPIは以下のレスポンスクラスを提供しています:

| クラス名 | Content-Type | 説明 |
| --- | --- | --- |
| `JSONResponse` | application/json | JSONレスポンス（デフォルト） |
| `HTMLResponse` | text/html | HTMLレスポンス |
| `PlainTextResponse` | text/plain | プレーンテキストレスポンス |
| `RedirectResponse` | - | リダイレクトレスポンス |
| `StreamingResponse` | - | ストリーミングレスポンス |
| `FileResponse` | - | ファイルダウンロードレスポンス |
| `UJSONResponse` | application/json | ujson使用の高速JSONレスポンス |
| `ORJSONResponse` | application/json | orjson使用の高速JSONレスポンス |

---

### 例外クラス

#### HTTPException - HTTP例外

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `HTTPException` |
| インポート | `from fastapi import HTTPException` |
| 親クラス | `starlette.exceptions.HTTPException` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| status_code | int | ○ | HTTPステータスコード |
| detail | Any | - | エラー詳細（JSONレスポンスの`detail`キー） |
| headers | dict[str, str] | - | レスポンスヘッダー |

**使用例**

```python
from fastapi import FastAPI, HTTPException

app = FastAPI()

@app.get("/items/{item_id}")
async def read_item(item_id: str):
    if item_id not in items:
        raise HTTPException(status_code=404, detail="Item not found")
    return {"item": items[item_id]}
```

---

#### WebSocketException - WebSocket例外

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `WebSocketException` |
| インポート | `from fastapi import WebSocketException` |
| 親クラス | `starlette.exceptions.WebSocketException` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| code | int | ○ | WebSocket終了コード（RFC 6455） |
| reason | str | - | 終了理由 |

---

### APIRouterクラス

複数のパス操作をグループ化するためのルータークラスです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `APIRouter` |
| インポート | `from fastapi import APIRouter` |
| 親クラス | `starlette.routing.Router` |

**初期化パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| prefix | str | - | "" | パスプレフィックス |
| tags | list[str] | - | None | タグリスト |
| dependencies | Sequence[Depends] | - | None | 依存性リスト |
| default_response_class | type[Response] | - | JSONResponse | デフォルトレスポンスクラス |
| responses | dict | - | None | 追加レスポンス定義 |
| callbacks | list[BaseRoute] | - | None | OpenAPIコールバック |
| redirect_slashes | bool | - | True | スラッシュリダイレクト |
| deprecated | bool | - | None | 非推奨フラグ |
| include_in_schema | bool | - | True | OpenAPIスキーマに含めるか |

**使用例**

```python
from fastapi import APIRouter, FastAPI

app = FastAPI()
router = APIRouter(prefix="/api/v1", tags=["users"])

@router.get("/users/")
async def read_users():
    return [{"username": "Rick"}, {"username": "Morty"}]

app.include_router(router)
```

---

### FastAPIアプリケーション設定

FastAPIアプリケーションの主要な初期化パラメータ:

| パラメータ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| debug | bool | False | デバッグモード |
| title | str | "FastAPI" | APIタイトル |
| summary | str | None | API概要 |
| description | str | "" | API説明（Markdown対応） |
| version | str | "0.1.0" | APIバージョン |
| openapi_url | str | "/openapi.json" | OpenAPIスキーマURL |
| openapi_tags | list[dict] | None | OpenAPIタグ定義 |
| servers | list[dict] | None | サーバー接続情報 |
| docs_url | str | "/docs" | Swagger UI URL |
| redoc_url | str | "/redoc" | ReDoc URL |

## バリデーションルール

FastAPIはPydanticを使用して、以下のバリデーションをサポートしています:

### 数値バリデーション

| パラメータ | 説明 |
| --- | --- |
| gt | 指定値より大きい |
| ge | 指定値以上 |
| lt | 指定値より小さい |
| le | 指定値以下 |
| multiple_of | 指定値の倍数 |

### 文字列バリデーション

| パラメータ | 説明 |
| --- | --- |
| min_length | 最小文字数 |
| max_length | 最大文字数 |
| pattern | 正規表現パターン |

### その他のバリデーション

| パラメータ | 説明 |
| --- | --- |
| strict | 厳密な型チェック |
| allow_inf_nan | 無限大・NaNを許可 |
| max_digits | 最大桁数（数値） |
| decimal_places | 小数点以下桁数 |

## 備考

### OpenAPI仕様

- FastAPIはOpenAPI 3.1.0仕様に準拠したスキーマを自動生成します
- JSON Schema 2020-12に対応しています
- Swagger UIとReDocによる対話的ドキュメントを自動提供します

### 依存性注入

- `Depends()`を使用した依存性注入パターンをサポート
- 依存性のスコープ（関数、リクエスト、ミドルウェア）を制御可能
- `yield`を使用したリソースのクリーンアップをサポート

### 非同期サポート

- `async def`による非同期エンドポイントをネイティブサポート
- 同期関数もスレッドプールで自動実行
- WebSocketのフルサポート

### パフォーマンス

- Starlette（ASGI）ベースの高性能アーキテクチャ
- Pydanticによる高速なデータバリデーション
- UJSONやORJSONによるJSON処理の高速化オプション
