# アーキテクチャ設計書

## 概要

本ドキュメントは、FastAPIフレームワーク（バージョン0.128.0）のアーキテクチャ設計について記述する。FastAPIは、Python 3.9以上で動作する高性能な非同期WebフレームワークであΔあり、Starletteをベースとして構築され、Pydanticによる型検証機能を統合している。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| Starlette | Python Import（継承） | ASGIアプリケーション基盤、ルーティング、ミドルウェア |
| Pydantic | Python Import | データ検証、シリアライゼーション、OpenAPIスキーマ生成 |
| Uvicorn（推奨） | ASGI Protocol | ASGIサーバー（本番環境） |
| OpenAPI/Swagger UI | HTTP | API仕様書の自動生成・表示（/docs） |
| ReDoc | HTTP | API仕様書の表示（/redoc） |

## レイヤー構成

### アーキテクチャスタイル

**ミドルウェア＋デコレータベースのルーティングアーキテクチャ**

FastAPIは、ASGI（Asynchronous Server Gateway Interface）に準拠したミドルウェアパターンを採用している。Starletteの継承により、リクエスト/レスポンスの処理をミドルウェアスタックで制御し、デコレータベースのルーティングでエンドポイントを定義する。依存性注入（DI）パターンを活用し、横断的関心事を分離している。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Presentation | HTTPリクエスト/レスポンス処理、OpenAPIドキュメント生成 | `FastAPI`, `APIRouter`, `responses.py`, `openapi/` |
| Application | ルーティング、依存性解決、リクエストバリデーション | `routing.py`, `dependencies/`, `params.py`, `param_functions.py` |
| Domain | データ検証、モデル定義（Pydantic統合） | `_compat/`, Pydantic BaseModel |
| Infrastructure | ASGI基盤、ミドルウェア、セキュリティ | `middleware/`, `security/`, Starlette |

### レイヤー間の依存関係ルール

**依存の方向性：**
- Presentation → Application → Domain → Infrastructure

**許可される参照：**
- 上位レイヤーから下位レイヤーへの参照
- 同一レイヤー内のコンポーネント間参照

**禁止される参照：**
- 下位レイヤーから上位レイヤーへの直接参照
- ただし、コールバック/イベントによる間接的な通知は許可

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| アプリケーション基盤 | FastAPIアプリケーションのコア機能 | `applications.py`, `routing.py` |
| リクエスト処理 | パラメータ抽出・バリデーション | `params.py`, `param_functions.py`, `dependencies/` |
| セキュリティ | 認証・認可機能 | `security/` |
| OpenAPI | API仕様書自動生成 | `openapi/` |
| ミドルウェア | リクエスト前後処理 | `middleware/` |
| エンコーディング | データシリアライゼーション | `encoders.py`, `responses.py` |
| 例外処理 | エラーハンドリング | `exceptions.py`, `exception_handlers.py` |
| 互換性 | Pydantic v2互換レイヤー | `_compat/` |

### パッケージ構造

```
fastapi/
├── __init__.py              # パブリックAPI定義
├── __main__.py              # CLI エントリポイント
├── applications.py          # FastAPI クラス（メインアプリケーション）
├── routing.py               # APIRouter, ルーティング処理
├── params.py                # パラメータクラス定義（Query, Path, Body等）
├── param_functions.py       # パラメータヘルパー関数
├── encoders.py              # JSONエンコーダー
├── responses.py             # レスポンスクラス
├── requests.py              # リクエストクラス（Starlette再エクスポート）
├── exceptions.py            # 例外クラス定義
├── exception_handlers.py    # 例外ハンドラー
├── datastructures.py        # データ構造（UploadFile等）
├── concurrency.py           # 並行処理ユーティリティ
├── background.py            # バックグラウンドタスク
├── utils.py                 # ユーティリティ関数
├── types.py                 # 型定義
├── logger.py                # ロガー設定
├── websockets.py            # WebSocket（Starlette再エクスポート）
├── staticfiles.py           # 静的ファイル（Starlette再エクスポート）
├── templating.py            # テンプレート（Starlette再エクスポート）
├── testclient.py            # テストクライアント（Starlette再エクスポート）
├── cli.py                   # CLIコマンド
├── _compat/                 # Pydantic互換レイヤー
│   ├── __init__.py
│   ├── shared.py
│   └── v2.py
├── dependencies/            # 依存性注入システム
│   ├── __init__.py
│   ├── models.py            # Dependant データクラス
│   └── utils.py             # 依存性解決ユーティリティ
├── middleware/              # ミドルウェア
│   ├── __init__.py
│   ├── asyncexitstack.py    # AsyncExitStack ミドルウェア
│   ├── cors.py              # CORS（Starlette再エクスポート）
│   ├── gzip.py              # Gzip（Starlette再エクスポート）
│   ├── httpsredirect.py     # HTTPSリダイレクト
│   ├── trustedhost.py       # 信頼ホスト
│   └── wsgi.py              # WSGI互換
├── openapi/                 # OpenAPI仕様生成
│   ├── __init__.py
│   ├── constants.py         # 定数
│   ├── docs.py              # Swagger UI, ReDoc HTML生成
│   ├── models.py            # OpenAPIスキーマモデル
│   └── utils.py             # OpenAPI生成ユーティリティ
└── security/                # セキュリティ
    ├── __init__.py
    ├── base.py              # SecurityBase 基底クラス
    ├── api_key.py           # APIキー認証
    ├── http.py              # HTTP認証（Basic, Bearer, Digest）
    ├── oauth2.py            # OAuth2認証
    ├── open_id_connect_url.py # OpenID Connect
    └── utils.py             # セキュリティユーティリティ
```

### コンポーネント依存関係

**FastAPI（applications.py）の依存関係：**
- Starlette.Starlette を継承
- routing.APIRouter を内部で使用
- openapi/ モジュールでAPI仕様書を生成
- exception_handlers.py でデフォルトのエラーハンドラーを登録
- middleware/ でリクエスト処理パイプラインを構築

**APIRouter（routing.py）の依存関係：**
- Starlette.routing を継承
- dependencies/ で依存性を解決
- params.py でパラメータを定義
- encoders.py でレスポンスをシリアライズ

**依存性注入（dependencies/）の依存関係：**
- params.py のパラメータ定義を解析
- security/ のセキュリティスキームを統合
- Pydantic でモデルを検証

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| - | - | - | FastAPIコア機能にはDB接続機能は含まれない（アプリケーション側で選択） |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| - | - | - | FastAPIコア機能にはキャッシュ機能は含まれない |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| - | - | FastAPIコア機能にはMQ機能は含まれない |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| - | - | FastAPIコア機能には検索エンジン連携は含まれない |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| AsyncExitStackMiddleware | 内蔵 | 依存性のAsyncExitStackを管理 |
| CORSMiddleware | Starlette | クロスオリジンリソース共有 |
| GZipMiddleware | Starlette | レスポンス圧縮 |
| HTTPSRedirectMiddleware | Starlette | HTTPからHTTPSへのリダイレクト |
| TrustedHostMiddleware | Starlette | 信頼されたホストの検証 |
| ServerErrorMiddleware | Starlette | サーバーエラーのハンドリング |
| ExceptionMiddleware | Starlette | 例外処理 |

## データフロー

### リクエスト処理の流れ

FastAPIは、ASGIプロトコルに準拠したリクエスト処理パイプラインを実装している。

1. **ASGIサーバー受信**: Uvicorn等のASGIサーバーがHTTPリクエストを受信
2. **ミドルウェアスタック**: ServerErrorMiddleware → ExceptionMiddleware → AsyncExitStackMiddleware → アプリケーションミドルウェアの順で処理
3. **ルーティング**: APIRouterがパスマッチングを実行し、対応するエンドポイントを特定
4. **依存性解決**: Dependsで宣言された依存性をトポロジカル順序で解決（キャッシュ付き）
5. **パラメータ抽出**: Path, Query, Body, Header, Cookie等のパラメータをリクエストから抽出
6. **バリデーション**: Pydanticモデルによる入力データの検証
7. **エンドポイント実行**: async/sync対応でエンドポイント関数を実行
8. **レスポンスシリアライズ**: JSONResponseまたはカスタムレスポンスでデータをシリアライズ
9. **依存性クリーンアップ**: yield依存性のfinally部分を逆順で実行

### 非同期処理の流れ

FastAPIは、asyncio/anyioベースの非同期処理をネイティブサポートしている。

1. **async エンドポイント**: async defで定義されたエンドポイントは直接awaitで実行
2. **sync エンドポイント**: defで定義されたエンドポイントはrun_in_threadpoolでスレッドプールに委譲
3. **バックグラウンドタスク**: BackgroundTasksを使用してレスポンス送信後に非同期タスクを実行
4. **依存性のジェネレータ**: yield依存性はAsyncExitStackで管理され、リクエスト終了時に自動クリーンアップ

### データ永続化の流れ

FastAPIコア機能にはデータ永続化機能は含まれない。アプリケーション側でSQLAlchemy、SQLModel、Tortoise ORM等のORMを統合して使用する設計となっている。

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| APIKey（Query/Header/Cookie） | `security/api_key.py` | APIキーベースの認証 |
| HTTP Basic | `security/http.py` (HTTPBasic) | ユーザー名/パスワード認証 |
| HTTP Bearer | `security/http.py` (HTTPBearer) | JWTトークン等のBearer認証 |
| HTTP Digest | `security/http.py` (HTTPDigest) | Digest認証 |
| OAuth2 Password | `security/oauth2.py` (OAuth2PasswordBearer) | OAuth2パスワードフロー |
| OAuth2 Authorization Code | `security/oauth2.py` (OAuth2AuthorizationCodeBearer) | OAuth2認可コードフロー |
| OpenID Connect | `security/open_id_connect_url.py` | OpenID Connect認証 |
| Security Scopes | `security/oauth2.py` (SecurityScopes) | スコープベースの認可 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | Python logging（logger.py） | 標準出力/設定による |
| 監査ログ | アプリケーション側で実装 | アプリケーション側で設定 |
| アクセスログ | Uvicorn/ASGIサーバーが出力 | 標準出力/ファイル |

### エラーハンドリング

FastAPIは、構造化されたエラーハンドリングシステムを提供する。

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| HTTPException | http_exception_handler | JSON `{"detail": ...}` + HTTPステータス |
| RequestValidationError | request_validation_exception_handler | 422 Unprocessable Entity + バリデーションエラー詳細 |
| WebSocketRequestValidationError | websocket_request_validation_exception_handler | WebSocket Close (1008) |
| ResponseValidationError | アプリケーション側で設定 | 500 Internal Server Error |
| StarletteHTTPException | http_exception_handler（継承） | JSON `{"detail": ...}` |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| - | FastAPIコア機能には含まれない | アプリケーション側で設定（ORM依存） |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 依存性逆転の原則（DIP） | 依存性注入システム | `Depends`関数によるDI、セキュリティスキームの抽象化 |
| 単一責任の原則（SRP） | モジュール分割 | routing, params, security等の責務分離 |
| 開放閉鎖の原則（OCP） | ミドルウェア・例外ハンドラー | カスタムミドルウェア・ハンドラーの追加可能 |
| リスコフの置換原則（LSP） | Starlette継承 | FastAPIはStarletteと互換性を維持 |
| インターフェース分離の原則（ISP） | セキュリティモジュール | APIKey, HTTPBasic, OAuth2等の個別クラス |
| コンポジション over 継承 | APIRouter | include_routerによるルーター合成 |
| 型ヒント活用 | 全モジュール | Annotated, TypeVar, Generic等の活用 |

### コーディング規約

- **Python標準**: PEP 8準拠（Ruffによる自動チェック）
- **型ヒント**: 全公開APIに型アノテーション必須（mypy strictモード）
- **ドキュメント**: `annotated-doc`パッケージによる型ヒントとドキュメントの統合
- **テスト**: pytest使用、coverage測定、docs_srcのサンプルコードをテスト対象に含む
- **非推奨警告**: `typing_extensions.deprecated`デコレータと`FastAPIDeprecationWarning`の使用
- **インポート**: isortによる自動整列、`fastapi`, `pydantic`, `starlette`をサードパーティとして認識

## 備考

- FastAPIバージョン: 0.128.0
- Python対応バージョン: 3.9, 3.10, 3.11, 3.12, 3.13, 3.14
- 主要依存関係:
  - Starlette >= 0.40.0, < 0.51.0
  - Pydantic >= 2.7.0
  - typing-extensions >= 4.8.0
  - annotated-doc >= 0.0.2
- オプション依存関係（standard）:
  - fastapi-cli（CLI機能）
  - httpx（テストクライアント）
  - jinja2（テンプレート）
  - python-multipart（フォーム/ファイルアップロード）
  - uvicorn（ASGIサーバー）
  - pydantic-settings（設定管理）
