# FastAPI コードリーディングガイドライン

## はじめに

このガイドラインは、FastAPIフレームワークのコードベースを効率的に理解するための手引きです。
Pythonに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、Pythonの基本構文と概念を解説します。

### 1.1 プログラム構造

FastAPIはPython 3.9以上を対象としたWebフレームワークです。Pythonファイルは`.py`拡張子を持ち、モジュールとして扱われます。

**基本的なFastAPIアプリケーション構造:**
```python
# ファイル: docs_src/first_steps/tutorial001_py39.py:1-8
from fastapi import FastAPI

app = FastAPI()


@app.get("/")
async def root():
    return {"message": "Hello World"}
```

**解説:**
- `from fastapi import FastAPI`: FastAPIクラスをインポート
- `app = FastAPI()`: アプリケーションインスタンスを作成
- `@app.get("/")`: デコレータでHTTPメソッドとパスを定義
- `async def`: 非同期関数を定義（FastAPIは非同期をサポート）

### 1.2 データ型と変数

FastAPIはPythonの型ヒントを活用します。これによりエディタ補完とバリデーションが自動化されます。

**型ヒントの例:**
```python
# ファイル: fastapi/params.py:20-25
class ParamTypes(Enum):
    query = "query"
    header = "header"
    path = "path"
    cookie = "cookie"
```

**主要なデータ型:**
| 型 | 説明 | 例 |
|---|---|---|
| `str` | 文字列 | `"hello"` |
| `int` | 整数 | `42` |
| `float` | 浮動小数点 | `3.14` |
| `bool` | 真偽値 | `True`, `False` |
| `list[T]` | リスト | `list[str]` |
| `dict[K, V]` | 辞書 | `dict[str, Any]` |
| `Optional[T]` | Nullable型 | `Optional[str]` |
| `Union[A, B]` | 複合型 | `Union[str, int]` |

### 1.3 制御構造

**条件分岐:**
```python
# ファイル: fastapi/utils.py:39-53
def is_body_allowed_for_status_code(status_code: Union[int, str, None]) -> bool:
    if status_code is None:
        return True
    if status_code in {
        "default",
        "1XX",
        "2XX",
        "3XX",
        "4XX",
        "5XX",
    }:
        return True
    current_status_code = int(status_code)
    return not (current_status_code < 200 or current_status_code in {204, 205, 304})
```

**ループ:**
```python
# ファイル: fastapi/utils.py:131-146
def deep_dict_update(main_dict: dict[Any, Any], update_dict: dict[Any, Any]) -> None:
    for key, value in update_dict.items():
        if (
            key in main_dict
            and isinstance(main_dict[key], dict)
            and isinstance(value, dict)
        ):
            deep_dict_update(main_dict[key], value)
        # ...
```

### 1.4 関数/メソッド定義

FastAPIでは同期関数と非同期関数の両方がサポートされています。

**同期関数:**
```python
def get_path_param_names(path: str) -> set[str]:
    return set(re.findall("{(.*?)}", path))
```

**非同期関数:**
```python
# ファイル: docs_src/dependencies/tutorial001_py39.py:8-11
async def common_parameters(
    q: Union[str, None] = None, skip: int = 0, limit: int = 100
):
    return {"q": q, "skip": skip, "limit": limit}
```

**クラスメソッド:**
```python
# ファイル: fastapi/dependencies/models.py:53-60
@cached_property
def oauth_scopes(self) -> list[str]:
    scopes = self.parent_oauth_scopes.copy() if self.parent_oauth_scopes else []
    for scope in self.own_oauth_scopes or []:
        if scope not in scopes:
            scopes.append(scope)
    return scopes
```

### 1.5 モジュール/インポート

Pythonのモジュールシステムは`import`文を使用します。

**基本的なインポートパターン:**
```python
# ファイル: fastapi/__init__.py:1-25
"""FastAPI framework, high performance, easy to learn, fast to code, ready for production"""

__version__ = "0.128.0"

from starlette import status as status

from .applications import FastAPI as FastAPI
from .background import BackgroundTasks as BackgroundTasks
from .datastructures import UploadFile as UploadFile
from .exceptions import HTTPException as HTTPException
from .param_functions import Body as Body
from .param_functions import Cookie as Cookie
from .param_functions import Depends as Depends
# ...
```

**インポートの種類:**
| パターン | 説明 | 例 |
|---|---|---|
| `import module` | モジュール全体 | `import fastapi` |
| `from module import item` | 特定の項目 | `from fastapi import FastAPI` |
| `from . import module` | 相対インポート | `from .applications import FastAPI` |
| `from .. import module` | 親パッケージから | `from ..dependencies import get_token_header` |

---

## 2. プロジェクト固有の概念

> このセクションでは、FastAPI特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### デコレータベースのルーティング

FastAPIは関数デコレータでHTTPエンドポイントを定義します。

```python
# ファイル: docs_src/bigger_applications/app_py39/routers/items.py:16-25
@router.get("/")
async def read_items():
    return fake_items_db


@router.get("/{item_id}")
async def read_item(item_id: str):
    if item_id not in fake_items_db:
        raise HTTPException(status_code=404, detail="Item not found")
    return {"name": fake_items_db[item_id]["name"], "item_id": item_id}
```

#### 依存性注入（Dependency Injection）

`Depends`を使用して依存関係を宣言的に注入します。

```python
# ファイル: docs_src/dependencies/tutorial001_py39.py:14-21
@app.get("/items/")
async def read_items(commons: dict = Depends(common_parameters)):
    return commons


@app.get("/users/")
async def read_users(commons: dict = Depends(common_parameters)):
    return commons
```

#### Pydanticモデルによるバリデーション

FastAPIはPydanticを使用して自動的にリクエスト/レスポンスのバリデーションを行います。

```python
# ファイル: fastapi/exceptions.py:147-148
RequestErrorModel: type[BaseModel] = create_model("Request")
WebSocketErrorModel: type[BaseModel] = create_model("WebSocket")
```

### 2.2 プロジェクト独自のパターン

#### Annotatedによる型注釈

FastAPIでは`Annotated`を使用して追加のメタデータを型に付与します。

```python
# ファイル: fastapi/params.py:49-54
regex: Annotated[
    Optional[str],
    deprecated(
        "Deprecated in FastAPI 0.100.0 and Pydantic v2, use `pattern` instead."
    ),
] = None,
```

#### StarletteとPydanticの統合

FastAPIはStarlette（ASGIフレームワーク）とPydantic（データバリデーション）を統合しています。

```python
# ファイル: fastapi/applications.py:32-48
from starlette.applications import Starlette
from starlette.datastructures import State
from starlette.exceptions import HTTPException
# ...

class FastAPI(Starlette):
    """
    `FastAPI` app class, the main entrypoint to use FastAPI.
    """
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `snake_case.py` | Pythonモジュール | `applications.py`, `param_functions.py` |
| `__init__.py` | パッケージ初期化 | `fastapi/__init__.py` |
| `_compat/` | 互換性レイヤー | `fastapi/_compat/` |
| `test_*.py` | テストファイル | `test_datastructures.py` |
| `*_py39.py` | Python 3.9用サンプル | `tutorial001_py39.py` |
| `*_an_py39.py` | Annotated + Python 3.9 | `tutorial001_an_py39.py` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `get_*` | データ取得関数 | `get_openapi`, `get_dependant` |
| `create_*` | オブジェクト生成関数 | `create_model_field`, `create_cloned_field` |
| `is_*` | 真偽値を返す関数 | `is_body_allowed_for_status_code` |
| `_*` | プライベート/内部使用 | `_CLONED_TYPES_CACHE` |
| `__*__` | 特殊メソッド | `__init__`, `__repr__` |
| `*Error` | 例外クラス | `FastAPIError`, `ValidationException` |
| `*Model` | Pydanticモデル | `RequestErrorModel` |
| `*Base` | 基底クラス | `SecurityBase` |

### 3.3 プログラム分類一覧

| カテゴリ | ファイルパターン | 説明 |
|---------|----------------|------|
| コアアプリケーション | `applications.py` | FastAPIクラス定義 |
| ルーティング | `routing.py` | APIRouter、パス操作 |
| パラメータ処理 | `params.py`, `param_functions.py` | Query, Path, Body等 |
| 依存性注入 | `dependencies/` | Dependant、依存解決 |
| セキュリティ | `security/` | OAuth2, HTTP認証等 |
| OpenAPI | `openapi/` | スキーマ生成 |
| ミドルウェア | `middleware/` | CORS, GZIP等 |
| 例外処理 | `exceptions.py` | カスタム例外 |
| エンコーダ | `encoders.py` | JSON変換 |
| 互換性 | `_compat/` | Pydantic v2対応 |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
fastapi-master/
├── fastapi/                    # メインソースコード
│   ├── __init__.py            # パッケージ公開API
│   ├── applications.py        # FastAPIクラス
│   ├── routing.py             # ルーティングロジック
│   ├── params.py              # パラメータクラス
│   ├── param_functions.py     # パラメータ関数
│   ├── exceptions.py          # 例外定義
│   ├── encoders.py            # JSONエンコーダ
│   ├── utils.py               # ユーティリティ
│   ├── _compat/               # Pydantic互換性
│   │   ├── __init__.py
│   │   ├── shared.py
│   │   └── v2.py
│   ├── dependencies/          # 依存性注入
│   │   ├── __init__.py
│   │   ├── models.py          # Dependantクラス
│   │   └── utils.py           # 依存解決ロジック
│   ├── security/              # セキュリティ
│   │   ├── __init__.py
│   │   ├── oauth2.py          # OAuth2実装
│   │   ├── http.py            # HTTP認証
│   │   ├── api_key.py         # APIキー認証
│   │   └── base.py            # 基底クラス
│   ├── openapi/               # OpenAPI生成
│   │   ├── __init__.py
│   │   ├── models.py          # OpenAPIモデル
│   │   ├── utils.py           # スキーマ生成
│   │   ├── docs.py            # ドキュメントUI
│   │   └── constants.py       # 定数
│   └── middleware/            # ミドルウェア
│       ├── __init__.py
│       ├── cors.py
│       └── gzip.py
├── tests/                     # テストコード
├── docs_src/                  # ドキュメント用サンプル
├── docs/                      # ドキュメント
├── scripts/                   # ビルドスクリプト
├── pyproject.toml             # プロジェクト設定
└── README.md                  # プロジェクト説明
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `fastapi/` | フレームワーク本体 | `applications.py`, `routing.py` |
| `fastapi/dependencies/` | 依存性注入システム | `models.py`, `utils.py` |
| `fastapi/security/` | 認証・認可 | `oauth2.py`, `http.py` |
| `fastapi/openapi/` | OpenAPI生成 | `utils.py`, `docs.py` |
| `fastapi/middleware/` | HTTPミドルウェア | `cors.py`, `gzip.py` |
| `fastapi/_compat/` | Pydantic互換性 | `v2.py`, `shared.py` |
| `tests/` | 自動テスト | `test_*.py` |
| `docs_src/` | サンプルコード | `tutorial*.py` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

FastAPIはStarletteをベースとしたASGIフレームワークで、レイヤードアーキテクチャを採用しています。

```mermaid
graph TB
    subgraph "クライアント"
        Client[HTTPクライアント]
    end

    subgraph "FastAPI Application"
        Middleware[ミドルウェア層]
        Routing[ルーティング層]
        DI[依存性注入層]
        Validation[バリデーション層]
        Endpoint[エンドポイント層]
    end

    subgraph "基盤"
        Starlette[Starlette ASGI]
        Pydantic[Pydantic]
    end

    Client --> Middleware
    Middleware --> Routing
    Routing --> DI
    DI --> Validation
    Validation --> Endpoint

    Routing -.-> Starlette
    Validation -.-> Pydantic
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| アプリケーション層 | FastAPIインスタンス管理 | `applications.py` |
| ルーティング層 | パス解析、ルート登録 | `routing.py` |
| 依存性注入層 | 依存関係の解決 | `dependencies/utils.py` |
| バリデーション層 | 入出力の検証 | `params.py`, `_compat/v2.py` |
| セキュリティ層 | 認証・認可 | `security/*.py` |
| OpenAPI層 | ドキュメント生成 | `openapi/utils.py` |

### 5.3 データフロー

1. **リクエスト受信**: Starletteがリクエストを受け取る
2. **ミドルウェア処理**: CORS、GZIP等のミドルウェアを通過
3. **ルーティング**: パスとHTTPメソッドからエンドポイントを特定
4. **依存性解決**: `Depends`で宣言された依存関係を解決
5. **パラメータ検証**: Pydanticでリクエストデータを検証
6. **エンドポイント実行**: ユーザー定義の関数を実行
7. **レスポンス生成**: 戻り値をJSONにシリアライズ
8. **レスポンス送信**: クライアントに返却

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

FastAPIアプリケーションのエントリーポイントは`FastAPI`クラスです。

```python
# ファイル: fastapi/applications.py:48-62
class FastAPI(Starlette):
    """
    `FastAPI` app class, the main entrypoint to use FastAPI.
    """

    def __init__(
        self: AppType,
        *,
        debug: Annotated[
            bool,
            Doc(
                """
                Boolean indicating if debug tracebacks should be returned on server
                errors.
                """
            ),
        ] = False,
        # ... 多数のパラメータ
    ):
```

**使用例:**
```python
from fastapi import FastAPI

app = FastAPI(
    title="My API",
    description="API description",
    version="1.0.0"
)
```

### 6.2 ビジネスロジック

ルーティングとエンドポイント処理は`routing.py`で定義されています。

```python
# ファイル: fastapi/routing.py:80-117
def request_response(
    func: Callable[[Request], Union[Awaitable[Response], Response]],
) -> ASGIApp:
    """
    Takes a function or coroutine `func(request) -> response`,
    and returns an ASGI application.
    """
    f: Callable[[Request], Awaitable[Response]] = (
        func if is_async_callable(func) else functools.partial(run_in_threadpool, func)
    )

    async def app(scope: Scope, receive: Receive, send: Send) -> None:
        request = Request(scope, receive, send)

        async def app(scope: Scope, receive: Receive, send: Send) -> None:
            # Starts customization
            response_awaited = False
            async with AsyncExitStack() as request_stack:
                scope["fastapi_inner_astack"] = request_stack
                async with AsyncExitStack() as function_stack:
                    scope["fastapi_function_astack"] = function_stack
                    response = await f(request)
            # ...
```

### 6.3 データアクセス

依存性注入システムを通じてデータアクセスを抽象化します。

```python
# ファイル: fastapi/dependencies/models.py:31-51
@dataclass
class Dependant:
    path_params: list[ModelField] = field(default_factory=list)
    query_params: list[ModelField] = field(default_factory=list)
    header_params: list[ModelField] = field(default_factory=list)
    cookie_params: list[ModelField] = field(default_factory=list)
    body_params: list[ModelField] = field(default_factory=list)
    dependencies: list["Dependant"] = field(default_factory=list)
    name: Optional[str] = None
    call: Optional[Callable[..., Any]] = None
    request_param_name: Optional[str] = None
    websocket_param_name: Optional[str] = None
    # ...
```

### 6.4 ユーティリティ/共通機能

```python
# ファイル: fastapi/utils.py:56-57
def get_path_param_names(path: str) -> set[str]:
    return set(re.findall("{(.*?)}", path))
```

```python
# ファイル: fastapi/utils.py:123-128
def generate_unique_id(route: "APIRoute") -> str:
    operation_id = f"{route.name}{route.path_format}"
    operation_id = re.sub(r"\W", "_", operation_id)
    assert route.methods
    operation_id = f"{operation_id}_{list(route.methods)[0].lower()}"
    return operation_id
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| デコレータパターン | ルート定義 | 高 | `routing.py` |
| 依存性注入 | `Depends` | 高 | `dependencies/utils.py` |
| 型ヒント+Annotated | メタデータ付与 | 高 | `params.py` |
| ファクトリパターン | モデル生成 | 中 | `utils.py` |
| キャッシュパターン | 計算結果保存 | 中 | `dependencies/models.py` |

### 各パターンの詳細

#### パターン1: デコレータによるルート定義

**目的:** HTTPエンドポイントを宣言的に定義する

**実装例:**
```python
# ファイル: docs_src/bigger_applications/app_py39/routers/items.py:28-38
@router.put(
    "/{item_id}",
    tags=["custom"],
    responses={403: {"description": "Operation forbidden"}},
)
async def update_item(item_id: str):
    if item_id != "plumbus":
        raise HTTPException(
            status_code=403, detail="You can only update the item: plumbus"
        )
    return {"item_id": item_id, "name": "The great Plumbus"}
```

**解説:** `@router.put()`デコレータはHTTP PUTメソッドのエンドポイントを定義。パスパラメータ、タグ、レスポンス定義を含む。

#### パターン2: 依存性注入（Depends）

**目的:** 共通ロジックの再利用と疎結合化

**実装例:**
```python
# ファイル: docs_src/bigger_applications/app_py39/dependencies.py:1-11
from fastapi import Header, HTTPException


async def get_token_header(x_token: str = Header()):
    if x_token != "fake-super-secret-token":
        raise HTTPException(status_code=400, detail="X-Token header invalid")


async def get_query_token(token: str):
    if token != "jessica":
        raise HTTPException(status_code=400, detail="No Jessica token provided")
```

**解説:** 認証ロジックを独立した関数として定義し、`Depends`で注入することで再利用可能に。

#### パターン3: cached_propertyによるキャッシュ

**目的:** 計算コストの高い値を一度だけ計算

**実装例:**
```python
# ファイル: fastapi/dependencies/models.py:105-129
@cached_property
def is_gen_callable(self) -> bool:
    if self.call is None:
        return False
    if inspect.isgeneratorfunction(
        _impartial(self.call)
    ) or inspect.isgeneratorfunction(_unwrapped_call(self.call)):
        return True
    # ...
```

**解説:** `@cached_property`により、最初のアクセス時のみ計算され、以降はキャッシュ値が返される。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: HTTPリクエスト処理フロー

**概要:** GETリクエストがエンドポイント関数に到達するまでの流れ

**処理フロー:**
```
クライアント → Starlette → routing.py → dependencies/utils.py → エンドポイント関数
```

**詳細な追跡:**

1. **リクエスト受信** (`fastapi/routing.py:91-102`)
   ```python
   async def app(scope: Scope, receive: Receive, send: Send) -> None:
       request = Request(scope, receive, send)

       async def app(scope: Scope, receive: Receive, send: Send) -> None:
           response_awaited = False
           async with AsyncExitStack() as request_stack:
               scope["fastapi_inner_astack"] = request_stack
               async with AsyncExitStack() as function_stack:
                   scope["fastapi_function_astack"] = function_stack
                   response = await f(request)
   ```

2. **依存性解決** (`fastapi/routing.py:345-352`)
   ```python
   solved_result = await solve_dependencies(
       request=request,
       dependant=dependant,
       body=body,
       dependency_overrides_provider=dependency_overrides_provider,
       async_exit_stack=async_exit_stack,
       embed_body_fields=embed_body_fields,
   )
   ```

3. **エンドポイント実行** (`fastapi/routing.py:355-358`)
   ```python
   raw_response = await run_endpoint_function(
       dependant=dependant,
       values=solved_result.values,
       is_coroutine=is_coroutine,
   )
   ```

4. **レスポンス生成** (`fastapi/routing.py:377-389`)
   ```python
   content = await serialize_response(
       field=response_field,
       response_content=raw_response,
       include=response_model_include,
       exclude=response_model_exclude,
       by_alias=response_model_by_alias,
       # ...
   )
   response = actual_response_class(content, **response_args)
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `fastapi/__init__.py` - 公開API一覧
3. `fastapi/applications.py` - FastAPIクラス
4. `docs_src/first_steps/` - 基本的な使用例

#### 特定機能を理解したい場合
1. 該当する`docs_src/`ディレクトリのサンプル
2. `fastapi/`内の対応モジュール
3. `tests/`内の対応テスト

#### 改修作業を行う場合
1. 対象機能の`docs_src/`サンプル
2. `fastapi/`内の実装
3. `tests/`内の関連テスト
4. `pyproject.toml`の依存関係確認

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 最初に読む |
| `CONTRIBUTING.md` | 貢献ガイド | 開発開始時 |
| `SECURITY.md` | セキュリティポリシー | セキュリティ確認時 |
| `pyproject.toml` | プロジェクト設定 | 依存関係確認時 |
| `docs/` | 公式ドキュメント | 詳細仕様確認時 |
| `docs_src/` | サンプルコード | 実装参考時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `Annotated`とは何ですか？
A: Python 3.9+の型ヒント機能で、型に追加のメタデータを付与します。FastAPIでは`Query`、`Path`などのパラメータ情報を付与するために使用されます。

```python
from typing import Annotated
from fastapi import Query

def read_items(q: Annotated[str, Query(max_length=50)]):
    pass
```

#### Q: `Depends`の動作原理は？
A: `Depends`は依存性注入のマーカーです。FastAPIはエンドポイント呼び出し前に`Depends`で指定された関数を実行し、その戻り値を引数として渡します。

#### Q: 同期関数と非同期関数の使い分けは？
A: I/O待ちが発生する処理（DB、外部API）は`async def`、CPU bound処理は`def`を使用します。FastAPIは両方をサポートし、同期関数はスレッドプールで実行されます。

### デバッグのヒント

1. **依存関係の追跡**: `fastapi/dependencies/utils.py`の`solve_dependencies`関数にブレークポイントを設置
2. **リクエスト検証**: `fastapi/routing.py`の`get_request_handler`を確認
3. **OpenAPI生成**: `fastapi/openapi/utils.py`の`get_openapi`関数を追跡
4. **エラー詳細**: `RequestValidationError`の`errors()`メソッドでバリデーションエラー詳細を取得

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| ASGI | Asynchronous Server Gateway Interface、非同期Webサーバーインターフェース |
| Starlette | FastAPIの基盤となるASGIフレームワーク |
| Pydantic | データバリデーションライブラリ |
| OpenAPI | REST APIの仕様記述フォーマット（旧Swagger） |
| Dependency Injection | 依存性注入、依存関係を外部から注入するパターン |
| Path Parameter | URLパスに含まれるパラメータ（例: `/items/{item_id}`） |
| Query Parameter | URLクエリ文字列のパラメータ（例: `?skip=0&limit=10`） |
| Body | HTTPリクエストのボディ（JSON等） |
| Middleware | リクエスト/レスポンスを横断的に処理する層 |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `fastapi/__init__.py` | パッケージ公開API | FastAPI, APIRouter, Depends等のエクスポート |
| `fastapi/applications.py` | FastAPIクラス | アプリケーションメイン |
| `fastapi/routing.py` | ルーティング | APIRouter, パス操作処理 |
| `fastapi/params.py` | パラメータクラス | Path, Query, Body等 |
| `fastapi/dependencies/` | 依存性注入 | Dependant, solve_dependencies |
| `fastapi/security/` | セキュリティ | OAuth2, HTTPBasic等 |
| `fastapi/openapi/` | OpenAPI | スキーマ生成、ドキュメントUI |
| `tests/` | テスト | pytest形式のテスト |
| `docs_src/` | サンプル | ドキュメント用サンプルコード |

### C. 参考資料

- [FastAPI公式ドキュメント](https://fastapi.tiangolo.com/)
- [Starlette公式ドキュメント](https://www.starlette.dev/)
- [Pydantic公式ドキュメント](https://docs.pydantic.dev/)
- [OpenAPI仕様](https://spec.openapis.org/oas/latest.html)
- [Python型ヒント](https://docs.python.org/3/library/typing.html)
