# セキュリティ設計書

## 概要

本ドキュメントは、FastAPIフレームワークのセキュリティ機能に関する設計書です。FastAPIは、Pythonで構築された高性能なWeb APIフレームワークであり、OpenAPI準拠のセキュリティ機構を提供しています。本書では、認証・認可、通信セキュリティ、データセキュリティ、入出力対策等について記載します。

## 認証設計

### 認証方式

FastAPIは以下の認証方式をサポートしています。

| 認証方式 | クラス名 | 説明 |
| --- | --- | --- |
| OAuth2 Password Bearer | `OAuth2PasswordBearer` | パスワードフローでBearerトークンを取得する方式。`tokenUrl`を指定してトークン取得エンドポイントを定義 |
| OAuth2 Authorization Code | `OAuth2AuthorizationCodeBearer` | 認可コードフローでBearerトークンを取得する方式 |
| HTTP Basic認証 | `HTTPBasic` | RFC 7617準拠のBasic認証。Base64エンコードされた`username:password`形式 |
| HTTP Bearer認証 | `HTTPBearer` | Bearerトークンによる認証 |
| HTTP Digest認証 | `HTTPDigest` | RFC 7616準拠のDigest認証（スタブ実装） |
| API Key認証 | `APIKeyQuery`, `APIKeyHeader`, `APIKeyCookie` | クエリパラメータ、ヘッダー、またはCookieでAPIキーを送信 |
| OpenID Connect | `OpenIdConnect` | OpenID Connectによる認証（スタブ実装） |

### OAuth2フォーム処理

| クラス名 | 説明 |
| --- | --- |
| `OAuth2PasswordRequestForm` | OAuth2パスワードフローで`username`, `password`をフォームデータとして収集。`grant_type`はオプション |
| `OAuth2PasswordRequestFormStrict` | 上記の厳格版。`grant_type`が必須で、値は`"password"`に固定 |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | アプリケーション実装依存 | JWTトークンの`exp`クレームで制御（例：30分） |
| セッション固定化対策 | トークンベース認証 | セッションIDではなくJWTトークンを使用するため、セッション固定攻撃のリスクを軽減 |
| トークンリフレッシュ | `refreshUrl`で設定可能 | OAuth2フローでリフレッシュトークンURLを指定可能 |

### 認証エラー処理

認証失敗時は`HTTP 401 Unauthorized`を返却し、`WWW-Authenticate`ヘッダーを設定します。

| 認証方式 | WWW-Authenticateヘッダー |
| --- | --- |
| OAuth2/Bearer | `Bearer` または `Bearer scope="..."` |
| HTTP Basic | `Basic` または `Basic realm="..."` |
| HTTP Digest | `Digest` |
| API Key | `APIKey` |

## 認可設計

### 権限体系

FastAPIはOAuth2スコープベースの権限管理をサポートしています。

| ロール/スコープ | 権限 | 説明 |
| --- | --- | --- |
| アプリケーション定義 | アプリケーション依存 | `OAuth2PasswordBearer`の`scopes`パラメータで定義 |
| エンドポイント固有 | `Security(dependency, scopes=[...])` | 各エンドポイントで必要なスコープを指定 |

### スコープ管理

`SecurityScopes`クラスを使用して、依存関係チェーン全体のスコープを集約できます。

```python
# スコープ定義例
oauth2_scheme = OAuth2PasswordBearer(
    tokenUrl="token",
    scopes={"me": "Read current user info.", "items": "Read items."}
)

# エンドポイントでのスコープ要求例
@app.get("/users/me/items/")
async def read_own_items(
    current_user: Annotated[User, Security(get_current_active_user, scopes=["items"])]
):
    ...
```

### アクセス制御

FastAPIは依存性注入（Dependency Injection）システムを通じてアクセス制御を実装します。

| 方式 | 実装 |
| --- | --- |
| `Depends` | 通常の依存関係注入。認証チェック等に使用 |
| `Security` | `Depends`の拡張版。OAuth2スコープを指定可能 |
| `auto_error` | 認証失敗時の自動エラーレスポンス制御。`False`で任意認証に対応 |

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | `HTTPSRedirectMiddleware`でHTTPからHTTPSへ自動リダイレクト |
| HSTS | Starlette/ASGIレベルで設定可能（FastAPIはStarletteを内包） |
| Trusted Host | `TrustedHostMiddleware`で許可するホストを制限 |

### HTTPSリダイレクト

```python
from fastapi.middleware.httpsredirect import HTTPSRedirectMiddleware
app.add_middleware(HTTPSRedirectMiddleware)
```

### Trusted Hostミドルウェア

```python
from fastapi.middleware.trustedhost import TrustedHostMiddleware
app.add_middleware(TrustedHostMiddleware, allowed_hosts=["example.com"])
```

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | HTTPS/TLS（サーバー設定） |
| 保存データ | アプリケーション実装依存 |
| パスワードハッシュ | アプリケーション実装依存（例：Argon2id、bcrypt推奨） |
| JWTトークン | HS256等のアルゴリズムで署名 |

### 機密情報管理

FastAPIは以下の機構で機密情報を管理します。

| 項目 | 方式 |
| --- | --- |
| パスワード入力 | `Form(json_schema_extra={"format": "password"})`でOpenAPIで適切に表示 |
| 認証情報送信 | OAuth2仕様に従いHTTP Basic認証または本体に含める |
| トークン送信 | `Authorization: Bearer <token>`ヘッダーで送信 |

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Pydanticによる入力バリデーション、JSONレスポンスのデフォルト使用 |
| SQLインジェクション | Pydanticによる型検証・入力バリデーション（ORMレベルでの対策推奨） |
| CSRF | ステートレスなJWTベース認証により軽減。フォーム送信時は別途対策が必要 |
| 入力検証 | Pydantic v2による厳密な型検証とバリデーション |
| パターン検証 | `pattern`パラメータで正規表現によるバリデーション |

### バリデーション機能

`Param`クラス（`Query`, `Path`, `Header`, `Cookie`, `Body`, `Form`, `File`）で以下のバリデーションが可能：

| パラメータ | 説明 |
| --- | --- |
| `gt`, `ge`, `lt`, `le` | 数値の範囲制限 |
| `min_length`, `max_length` | 文字列長の制限 |
| `pattern` | 正規表現パターン |
| `strict` | 厳密な型チェック |
| `validation_alias` | バリデーション時のエイリアス |

### CORS設定

`CORSMiddleware`で Cross-Origin Resource Sharing を設定します。

```python
from fastapi.middleware.cors import CORSMiddleware

app.add_middleware(
    CORSMiddleware,
    allow_origins=["http://localhost:8080"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)
```

| 設定項目 | 説明 |
| --- | --- |
| `allow_origins` | 許可するオリジンのリスト |
| `allow_credentials` | 資格情報（Cookie等）の送信許可 |
| `allow_methods` | 許可するHTTPメソッド |
| `allow_headers` | 許可するヘッダー |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| アプリケーションログ | `fastapi`ロガーによるエラー・警告出力 | アプリケーション設定依存 |
| リクエストログ | ASGIサーバー（uvicorn等）で設定 | サーバー設定依存 |
| 認証ログ | アプリケーション実装で対応 | アプリケーション設定依存 |

### ロギング設定

```python
import logging
logger = logging.getLogger("fastapi")
```

## 例外処理

### セキュリティ関連例外

| 例外クラス | 説明 |
| --- | --- |
| `HTTPException` | HTTPエラーレスポンス用。認証失敗等で使用 |
| `WebSocketException` | WebSocketエラー用 |
| `RequestValidationError` | リクエストバリデーションエラー |
| `ResponseValidationError` | レスポンスバリデーションエラー |

### 認証失敗時のレスポンス

```python
HTTPException(
    status_code=401,
    detail="Not authenticated",
    headers={"WWW-Authenticate": "Bearer"}
)
```

## 備考

### フレームワーク特性

- FastAPIはStarletteをベースとしており、セキュリティミドルウェア（CORS、HTTPSリダイレクト、TrustedHost）はStarletteから継承
- Pydantic v2による強力な型検証・バリデーション機能を標準で提供
- OpenAPI 3.1準拠のセキュリティスキーマ定義をサポート
- 非同期処理（async/await）に完全対応

### 推奨事項

1. 本番環境では必ずHTTPSを使用すること
2. パスワードハッシュにはArgon2id、bcrypt等の強力なアルゴリズムを使用すること
3. JWTトークンの有効期限は短く設定し、リフレッシュトークンを併用すること
4. CORS設定では必要最小限のオリジンのみを許可すること
5. 機密情報は環境変数や外部シークレット管理サービスで管理すること
