# バッチ設計書 11-BackgroundTasks（サンプル）

## 概要

本ドキュメントは、FastAPI BackgroundTasksのチュートリアル用サンプルコードについての設計書です。HTTPレスポンス送信後にバックグラウンドでタスクを実行する機能のデモンストレーションコードの仕様と動作を記載します。

### 本バッチの処理概要

FastAPI BackgroundTasksは、HTTPレスポンスをクライアントに返却した後にバックグラウンドで処理を実行する機能を提供します。このサンプルコードは、その機能の使用方法を示すチュートリアル用の実装例です。

**業務上の目的・背景**：Webアプリケーションでは、ユーザーへの応答を高速化するために、時間のかかる処理をレスポンス送信後に遅延実行したいケースが多く存在します。例えば、メール通知送信、ファイル処理、ログ記録などの処理は、ユーザーがレスポンスを待つ必要がないため、バックグラウンドで実行することでユーザー体験を向上させることができます。本サンプルは、そのようなバックグラウンドタスクの実装パターンを開発者に教育する目的で作成されています。

**バッチの実行タイミング**：本サンプルは「バッチ処理」ではなく、HTTPリクエストを契機として実行されるバックグラウンドタスクのデモンストレーションです。エンドポイント `/send-notification/{email}` へのPOSTリクエスト受信時に、レスポンス返却後にバックグラウンドタスクが実行されます。

**主要な処理内容**：
1. `/send-notification/{email}` エンドポイントでPOSTリクエストを受信
2. BackgroundTasksオブジェクトに実行するタスク関数を登録（`add_task`メソッド）
3. クライアントに即座にJSONレスポンスを返却
4. レスポンス送信後にバックグラウンドで `write_notification` または `write_log` 関数を実行
5. `log.txt` ファイルにメッセージを書き込み

**前後の処理との関連**：本サンプルは独立したチュートリアルコードであり、他のバッチ処理との依存関係はありません。ただし、実際のプロダクション環境では、バックグラウンドタスクの完了を監視するジョブキューシステム（Celery、RQ等）との連携が検討されます。

**影響範囲**：本サンプルは `log.txt` ファイルへの書き込みを行います。tutorial001では上書きモード（mode="w"）、tutorial002では追記モード（mode="a"）で動作します。

## バッチ種別

デモンストレーション / チュートリアルサンプル（バックグラウンド通知処理）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（HTTPリクエスト受信時） |
| 実行時刻 | N/A |
| 実行曜日 | N/A |
| 実行日 | N/A |
| トリガー | HTTPリクエスト（POST /send-notification/{email}） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| FastAPIアプリケーションが起動していること | uvicorn等のASGIサーバーで起動 |
| 書き込み先ディレクトリへの書き込み権限 | `log.txt` ファイルを作成・書き込み可能であること |

### 実行可否判定

HTTPリクエストが `/send-notification/{email}` エンドポイントに到達した時点で自動的にバックグラウンドタスクが登録・実行されます。特別な実行可否判定ロジックはありません。

tutorial002系では、クエリパラメータ `q` が指定された場合のみ、追加のバックグラウンドタスク（クエリログ記録）が登録されます。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| email | str（パスパラメータ） | Yes | - | 通知先メールアドレス |
| q | str（クエリパラメータ） | No | None | 検索クエリ（tutorial002のみ） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| HTTPリクエスト | REST API | パスパラメータとクエリパラメータから入力を受け取る |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| log.txt | テキストファイル | 通知メッセージまたはログメッセージを出力 |
| HTTPレスポンス | JSON | `{"message": "Notification sent in the background"}` または `{"message": "Message sent"}` |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | log.txt |
| 出力先 | カレントディレクトリ |
| 文字コード | システムデフォルト（通常UTF-8） |
| 区切り文字 | N/A（プレーンテキスト） |

**tutorial001での出力フォーマット:**
```
notification for {email}: {message}
```

**tutorial002での出力フォーマット:**
```
found query: {q}
message to {email}
```

## 処理フロー

### 処理シーケンス

```
1. POSTリクエスト受信（/send-notification/{email}）
   └─ FastAPIがBackgroundTasksオブジェクトを自動生成
2. 依存関係の解決（tutorial002のみ）
   └─ get_query関数が呼び出され、qパラメータがあればログタスクを登録
3. バックグラウンドタスク登録
   └─ background_tasks.add_task()でwrite_notification/write_log関数を登録
4. HTTPレスポンス返却
   └─ クライアントに即座にJSONレスポンスを送信
5. バックグラウンドタスク実行
   └─ レスポンス送信後にlog.txtへの書き込みを実行
```

### フローチャート

```mermaid
flowchart TD
    A[POST /send-notification/email 受信] --> B{tutorial002?}
    B -->|Yes| C[get_query依存関係解決]
    C --> D{qパラメータあり?}
    D -->|Yes| E[クエリログタスク登録]
    D -->|No| F[スキップ]
    E --> G[通知タスク登録]
    F --> G
    B -->|No| G
    G --> H[JSONレスポンス返却]
    H --> I[バックグラウンドタスク実行]
    I --> J[log.txtに書き込み]
    J --> K[処理完了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本サンプルはデータベース操作を行いません。

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| N/A | N/A | N/A | データベース操作なし |

### テーブル別操作詳細

データベース操作は行われないため、該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| N/A | ファイル書き込みエラー | log.txtへの書き込み権限がない場合 | 書き込み権限を付与する |
| N/A | ディスク容量不足 | ディスク容量が不足している場合 | ディスク容量を確保する |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（サンプルコードのため未実装） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

### 障害時対応

本サンプルコードにはエラーハンドリングが実装されていません。バックグラウンドタスクが失敗した場合、例外がログに出力されますが、クライアントには影響しません（レスポンスは既に送信済みのため）。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | N/A（ファイル操作のみ） |
| コミットタイミング | N/A |
| ロールバック条件 | N/A |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | チュートリアル用のため制限なし |
| 目標処理時間 | 瞬時（ファイル書き込みのみ） |
| メモリ使用量上限 | 特に制限なし |

## 排他制御

本サンプルコードでは排他制御は実装されていません。同時に複数のリクエストが発生した場合、ファイル書き込みの競合が発生する可能性があります。

- tutorial001: 上書きモード（mode="w"）のため、後から書き込んだ内容で上書きされる
- tutorial002: 追記モード（mode="a"）のため、複数のメッセージが追記される（順序は保証されない）

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| アプリケーションログ | バックグラウンドタスク実行時 | log.txtファイルに通知/ログメッセージを出力 |

本サンプルは開始ログ、終了ログ、エラーログの出力機能を持っていません。

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| N/A | N/A | N/A |

本サンプルはチュートリアル用のため、監視・アラート機能は実装されていません。

## 備考

- 本サンプルはFastAPI公式ドキュメントのBackground Tasksチュートリアルに対応するコードです
- Python 3.9/3.10の両バージョン対応、およびAnnotated構文対応のバリエーションが用意されています
- 実際のプロダクション環境では、重い処理にはCeleryなどのタスクキューシステムの使用が推奨されます
- BackgroundTasksクラスはStarletteのBackgroundTasksを継承しており、FastAPIがラップして提供しています
- 同一リクエスト内で複数のBackgroundTasksオブジェクトが使用されても、FastAPIが自動的にマージして処理します

### サンプルファイル一覧

| ファイル名 | Python版 | 特徴 |
|-----------|---------|------|
| tutorial001_py39.py | 3.9+ | 基本的なBackgroundTasks使用例 |
| tutorial002_py39.py | 3.9+ | 依存関係でのBackgroundTasks使用例（Union構文） |
| tutorial002_py310.py | 3.10+ | 依存関係でのBackgroundTasks使用例（パイプ構文） |
| tutorial002_an_py39.py | 3.9+ | Annotated構文使用例（Union構文） |
| tutorial002_an_py310.py | 3.10+ | Annotated構文使用例（パイプ構文） |
