# バッチ設計書 2-translate

## 概要

本ドキュメントは、FastAPIプロジェクトにおけるドキュメント翻訳バッチ（translate.py）の設計仕様を記載する。

### 本バッチの処理概要

LLM（OpenAI GPT-5.2）を使用して英語ドキュメントを各言語に翻訳するCLIツールである。翻訳の追加、更新、削除機能を提供し、翻訳ドキュメントのライフサイクル全体を管理する。

**業務上の目的・背景**：FastAPIは国際的なプロジェクトであり、多言語ドキュメントの提供が重要である。従来の手動翻訳は時間がかかり、英語ドキュメントの更新に追従することが困難だった。LLMを活用することで、翻訳の初期作成や既存翻訳の更新を自動化し、翻訳者の負担を軽減しつつドキュメントの鮮度を保つことができる。

**バッチの実行タイミング**：GitHub Actions経由で手動実行または定期実行。翻訳PRの作成時やドキュメント更新イベント時に実行される。

**主要な処理内容**：
1. translate-page: 単一ページの翻訳を実行
2. translate-lang: 特定言語の全ページを翻訳
3. update-outdated: 古くなった翻訳を更新
4. add-missing: 未翻訳ページを追加
5. remove-removable: 英語版が削除されたページの翻訳を削除
6. make-pr: 翻訳変更をコミットしてPRを作成

**前後の処理との関連**：translation_fixer.pyと連携して翻訳品質を確保。翻訳完了後はnotify_translations.pyで通知される。

**影響範囲**：docs/{lang}/docs/配下の全翻訳ドキュメント、GitHubリポジトリ（ブランチ作成、PR作成）に影響する。

## バッチ種別

データ変換処理 / ドキュメント生成

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（イベントトリガー） |
| 実行時刻 | - |
| 実行曜日 | - |
| 実行日 | - |
| トリガー | GitHub Actions / 手動実行 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| LANGUAGE環境変数 | 対象言語コード（例：es, ja）が設定されていること |
| OpenAI APIキー | LLM（GPT-5.2）へのアクセスが可能であること |
| llm-prompt.md | 対象言語用の翻訳プロンプトファイルが存在すること |
| language_names.yml | 言語名定義ファイルが存在すること |

### 実行可否判定

- 対象言語のllm-prompt.mdが存在する場合のみ実行可能
- 英語（en）を対象言語とすることはできない

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| language | str | Yes | - | 翻訳対象の言語コード |
| en_path | Path | コマンドによる | - | 翻訳元の英語ドキュメントパス |
| command | str | No | - | 実行コマンド名 |
| github_token | str | PR作成時 | - | GitHub APIトークン |
| github_repository | str | PR作成時 | - | 対象リポジトリ名 |
| commit_in_place | bool | No | False | 現在のブランチに直接コミットするか |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| docs/en/docs/*.md | Markdown | 翻訳元の英語ドキュメント |
| docs/{lang}/llm-prompt.md | Markdown | 言語固有の翻訳プロンプト |
| scripts/general-llm-prompt.md | Markdown | 共通翻訳プロンプト |
| docs/language_names.yml | YAML | 言語コードと言語名のマッピング |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| docs/{lang}/docs/*.md | Markdown | 翻訳されたドキュメント |
| GitHub Pull Request | - | 翻訳更新時に自動作成されるPR |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | 英語版と同一のファイル名 |
| 出力先 | docs/{lang}/docs/ |
| 文字コード | UTF-8 |
| 改行コード | LF（newline="\n"） |

## 処理フロー

### 処理シーケンス

```
1. コマンド解析
   └─ typerによるCLI引数解析
2. 言語設定読み込み
   └─ language_names.ymlから言語名取得
   └─ llm-prompt.mdから言語固有プロンプト取得
3. 対象ファイル特定
   └─ 翻訳対象セクションの判定（reference等は除外）
   └─ 優先順位に基づくファイル列挙
4. Git履歴比較（update-outdated時）
   └─ 英語版と翻訳版の最終更新日時を比較
5. LLM翻訳実行
   └─ pydantic_aiのAgentを使用
   └─ 既存翻訳がある場合は差分更新
6. ファイル出力
   └─ 翻訳結果をMarkdownファイルとして保存
7. PR作成（make-pr時）
   └─ 新規ブランチ作成、コミット、プッシュ、PR作成
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[コマンド解析]
    B --> C{コマンド種別}
    C -->|translate-page| D[単一ページ翻訳]
    C -->|translate-lang| E[全ページ翻訳]
    C -->|update-outdated| F[古い翻訳更新]
    C -->|add-missing| G[未翻訳追加]
    C -->|remove-removable| H[不要翻訳削除]
    D --> I[LLM翻訳実行]
    E --> I
    F --> J[Git履歴比較]
    G --> I
    H --> K[ファイル削除]
    J --> I
    I --> L[ファイル出力]
    K --> M[バッチ終了]
    L --> M
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベースを使用しない。ファイルベースのデータ管理を行う。

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし（ファイルベース処理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | AssertionError | 英語を翻訳対象に指定 | エラーメッセージを表示して中断 |
| - | AssertionError | llm-prompt.mdが存在しない | エラーメッセージを表示して中断 |
| - | AssertionError | 英語ドキュメントパスが不正 | エラーメッセージを表示して中断 |
| - | typer.Exit(1) | 対象言語がLLM翻訳非対応 | 終了コード1で終了 |
| - | ValueError | ファイル処理エラー | エラーログ出力 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（明示的なリトライ処理は実装されていない） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

### 障害時対応

1. LLM APIエラーの場合、OpenAI側のステータスを確認
2. 翻訳品質に問題がある場合、llm-prompt.mdを調整して再実行
3. Git操作エラーの場合、リポジトリ状態を確認してクリーンアップ

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | ページ単位 |
| コミットタイミング | make-prコマンド実行時に一括コミット |
| ロールバック条件 | 各ページは独立して処理されるため、個別ロールバック不要 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1〜数百ページ |
| 目標処理時間 | ページあたり数十秒〜数分（LLM応答時間に依存） |
| メモリ使用量上限 | 特に制限なし |

## 排他制御

同一言語への同時実行は推奨しない。GitHub Actionsのconcurrency設定で制御することが望ましい。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | 翻訳開始時 | 翻訳対象ファイルパス、言語名 |
| 進捗ログ | 処理中 | 翻訳中のファイル、保存先パス |
| 終了ログ | 翻訳完了時 | 完了メッセージ |
| エラーログ | エラー発生時 | エラー内容 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 1時間（全ページ翻訳時） | GitHub Actions |
| エラー件数 | 1件 | GitHub Actionsログ |

## 備考

- 翻訳対象外セクション：reference、release-notes.md、fastapi-people.md、external-links.md、newsletter.md、management-tasks.md、management.md、contributing.md
- 既存翻訳がある場合は差分更新モードで動作し、変更箇所のみを更新する
- LLMモデルはOpenAI GPT-5.2（pydantic_ai経由）
- 翻訳優先順位：learn → tutorial → advanced → about → how-to → その他
